﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;

using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.DataModelMaker.Generators;

using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the workspace.
    /// </summary>
    public class WorkspaceViewModel : ViewModelBase
    {
        /// <summary>The root editor data model definition view model.</summary>
        private RootEditorDataModelDefinitionViewModel rootEditorDataModelVM = null;

        /// <summary>The root runtime data model definition view model.</summary>
        private RootRuntimeDataModelDefinitionViewModel rootRuntimeDataModelVM = null;

        /// <summary>The root binary data definition view model.</summary>
        private RootBinaryDataDefinitionViewModel rootBinaryDataModelVM = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public WorkspaceViewModel(ViewModelBase parent, WorkspaceDefinition dataModel) :
            base(parent, dataModel)
        {
        }

        /// <summary>
        /// Static event triggered when a data definition is modified.
        /// The sender is the view model of the modified definition.
        /// </summary>
        public static event EventHandler DataDefinitionModified = null;

        /// <summary>
        /// Get or set the root editor data model definition view model.
        /// </summary>
        public RootEditorDataModelDefinitionViewModel RootEditorDataModelDefinition
        {
            get
            {
                return this.rootEditorDataModelVM;
            }

            set
            {
                if (this.rootEditorDataModelVM != value)
                {
                    this.rootEditorDataModelVM = value;
                    this.OnPropertyChanged();
                }
            }
        }

        /// <summary>
        /// Get or set the root runtime data model definition view model.
        /// </summary>
        public RootRuntimeDataModelDefinitionViewModel RootRuntimeDataModelDefinition
        {
            get
            {
                return this.rootRuntimeDataModelVM;
            }

            set
            {
                if (this.rootRuntimeDataModelVM != value)
                {
                    this.rootRuntimeDataModelVM = value;
                    this.OnPropertyChanged();
                }
            }
        }

        /// <summary>
        /// Get or set the root binary data definition view model.
        /// </summary>
        public RootBinaryDataDefinitionViewModel RootBinaryDataDefinition
        {
            get
            {
                return this.rootBinaryDataModelVM;
            }

            set
            {
                if (this.rootBinaryDataModelVM != value)
                {
                    this.rootBinaryDataModelVM = value;
                    this.OnPropertyChanged();
                }
            }
        }

        /// <summary>
        /// Trigger DataDefinitionModified event.
        /// </summary>
        /// <param name="definitionOwnerViewModel">The view model of the modified definition.</param>
        public static void TriggerDataDefinitionModifiedEvent(
            ViewModelBase definitionOwnerViewModel)
        {
            // Trigger the event.
            if (WorkspaceViewModel.DataDefinitionModified != null)
            {
                WorkspaceViewModel.DataDefinitionModified(
                    definitionOwnerViewModel,
                    EventArgs.Empty);
            }
        }

        /// <summary>
        /// Load the workspace from file.
        /// </summary>
        /// <param name="basePath">The base folder path.</param>
        /// <returns>True on success.</returns>
        public bool LoadFromFile(string basePath)
        {
            var myDataModel = this.DataModel as WorkspaceDefinition;

            string editorDefFilePath = Path.Combine(basePath, myDataModel.EditorDataModelDefinitionFilePath);
            editorDefFilePath = Path.GetFullPath(editorDefFilePath);

            string runtimeDefFilePath = Path.Combine(basePath, myDataModel.RuntimeDataModelDefinitionFilePath);
            runtimeDefFilePath = Path.GetFullPath(runtimeDefFilePath);

            var editorDataModels = this.LoadEditorDataModelDefinitions(editorDefFilePath);
            var runtimeDataModels = this.LoadRuntimeDataModelDefinitions(runtimeDefFilePath);

            myDataModel.EditorDataModelDefinition = editorDataModels;
            myDataModel.RuntimeDataModelDefinition = runtimeDataModels;

            this.RootEditorDataModelDefinition =
                new RootEditorDataModelDefinitionViewModel(this, editorDataModels);
            this.Children.Add(this.rootEditorDataModelVM);

            this.RootRuntimeDataModelDefinition =
                new RootRuntimeDataModelDefinitionViewModel(this, runtimeDataModels);
            this.Children.Add(this.rootRuntimeDataModelVM);

            this.RootBinaryDataDefinition =
                new RootBinaryDataDefinitionViewModel(this, runtimeDataModels);
            this.Children.Add(this.rootBinaryDataModelVM);

            return true;
        }

        /// <summary>
        /// Load editor data model definitions.
        /// </summary>
        /// <param name="filePath">The file path.</param>
        /// <returns>The definitions.</returns>
        private EditorDataModelRootDefinition LoadEditorDataModelDefinitions(string filePath)
        {
            var generator = new DataModelGenerator();

            // Load the definition file.
            bool result = generator.LoadDefinitionFile(filePath);
            if (result == false)
            {
                return null;
            }

            return generator.GetDefinitionData() as EditorDataModelRootDefinition;
        }

        /// <summary>
        /// Load runtime data model definitions.
        /// </summary>
        /// <param name="filePath">The file path.</param>
        /// <returns>The definitions.</returns>
        private RuntimeDataModelRootDefinition LoadRuntimeDataModelDefinitions(string filePath)
        {
            var generator = new RuntimeDataModelGenerator();

            // Load the definition file.
            bool result = generator.LoadDefinitionFile(filePath);
            if (result == false)
            {
                return null;
            }

            return generator.GetDefinitionData() as RuntimeDataModelRootDefinition;
        }
    }
}
