﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the runtime data model include definition.
    /// </summary>
    public class RuntimeDataModelIncludeDefinitionViewModel : ViewModelBase
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public RuntimeDataModelIncludeDefinitionViewModel(
            ViewModelBase parent,
            RuntimeDataModelIncludeDefinition dataModel) :
            base(parent, dataModel)
        {
            this.OnCommitEditingExecutable = new AnonymousExecutable(this.OnCommitEditing);
            this.OnCancelEditingExecutable = new AnonymousExecutable(this.OnCancelEditing);
        }

        /// <summary>
        /// Get or set the executable for committing modification of the include definition.
        /// </summary>
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for canceling modification of the include definition.
        /// </summary>
        public IExecutable OnCancelEditingExecutable { get; set; }

        /// <summary>
        /// Handle execution when committing modification of the include definition.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCommitEditing(object parameter)
        {
            var values = parameter as KeyValuePair<string, object>[];
            if (values == null)
            {
                return;
            }

            try
            {
                foreach (var pair in values)
                {
                    var binder = new EffectMakerSetMemberBinder(pair.Key, true);
                    this.TrySetMember(binder, pair.Value);
                }

                var myParent = this.Parent as RuntimeDataModelDefinitionViewModel;
                if (myParent != null)
                {
                    myParent.NotifyIncludeDefinitionChanged(this);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "RuntimeDataModelIncludeDefinitionViewModel.OnCommitEditing => Failed setting values to the data model include definition.");
                Logger.Log(LogLevels.Error, "Exception : {0}, message : {1}", ex.GetType().ToString(), ex.Message);
            }
        }

        /// <summary>
        /// Handle execution when canceling modification of the include definition.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCancelEditing(object parameter)
        {
            var myParent = this.Parent as RuntimeDataModelDefinitionViewModel;
            if (myParent == null)
            {
                return;
            }

            myParent.DeleteIncludeDefinition(this);
        }
    }
}
