﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;

using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the runtime data model definition.
    /// </summary>
    public class RootRuntimeDataModelDefinitionViewModel : ViewModelBase
    {
        /// <summary>The runtime data model definition view models.</summary>
        private List<RuntimeDataModelDefinitionViewModel> dataModelViewModels =
            new List<RuntimeDataModelDefinitionViewModel>();

        /// <summary>The selected data model definition view model.</summary>
        private RuntimeDataModelDefinitionViewModel selectedDataModelVM = null;

        /// <summary>The data model definition view model to be displayed.</summary>
        private RuntimeDataModelDefinitionViewModel displayedDataModelVM = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public RootRuntimeDataModelDefinitionViewModel(
            ViewModelBase parent,
            RuntimeDataModelRootDefinition dataModel) :
            base(parent, dataModel)
        {
            this.OnCreateDataModelExecutable = new AnonymousExecutable(this.OnCreateDataModel);

            foreach (RuntimeDataModelDefinition def in dataModel.DataModels)
            {
                var child = new RuntimeDataModelDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.dataModelViewModels.Add(child);
            }
        }

        /// <summary>
        /// Static event triggered when the data model list is changed.
        /// </summary>
        public static event EventHandler DataModelListChanged = null;

        /// <summary>
        /// Enumerate runtime data model definition view models.
        /// </summary>
        public IEnumerable<RuntimeDataModelDefinitionViewModel> DataModelDefinitionViewModels
        {
            get
            {
                return from vm in this.dataModelViewModels
                       orderby vm.DataModelNamespace, vm.DataModelName
                       select vm;
            }
        }

        /// <summary>
        /// Get or set the data model definition view model to be displayed.
        /// </summary>
        public object DisplayedDataModel
        {
            get
            {
                if (this.displayedDataModelVM != null)
                {
                    return this.displayedDataModelVM;
                }
                else
                {
                    return this.selectedDataModelVM;
                }
            }

            set
            {
                if (value != null && (value is RuntimeDataModelDefinitionViewModel) == false)
                {
                    return;
                }

                this.SetValue(
                    ref this.displayedDataModelVM,
                    (RuntimeDataModelDefinitionViewModel)value);
            }
        }

        /// <summary>
        /// Get or set the selected data model definition view model.
        /// </summary>
        public object SelectedDataModel
        {
            get
            {
                return this.selectedDataModelVM;
            }

            set
            {
                if (value != null && (value is RuntimeDataModelDefinitionViewModel) == false)
                {
                    return;
                }

                this.SetValue(
                    ref this.selectedDataModelVM,
                    (RuntimeDataModelDefinitionViewModel)value);
            }
        }

        /// <summary>
        /// Get or set the executable for creating a new data model.
        /// </summary>
        public IExecutable OnCreateDataModelExecutable { get; set; }

        /// <summary>
        /// Delete the specified data model.
        /// </summary>
        /// <param name="viewModel">The view model of the data model to delete.</param>
        public void DeleteDataModel(RuntimeDataModelDefinitionViewModel viewModel)
        {
            var myDataModel = this.DataModel as RuntimeDataModelRootDefinition;
            if (myDataModel == null)
            {
                return;
            }

            int index = this.dataModelViewModels.IndexOf(viewModel);
            if (index < 0)
            {
                return;
            }

            if (this.SelectedDataModel == viewModel)
            {
                this.SelectedDataModel = null;
            }

            this.dataModelViewModels.RemoveAt(index);

            var def = viewModel.DataModel as RuntimeDataModelDefinition;
            myDataModel.DataModels.Remove(def);

            viewModel.Dispose();
            def.Dispose();

            this.OnPropertyChanged("DataModelDefinitionViewModels");

            // Trigger the event.
            if (DataModelListChanged != null)
            {
                DataModelListChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle execution for creating a new data model.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnCreateDataModel(object parameter)
        {
            var createParams = parameter as object[];
            if (createParams == null || createParams.Length < 2)
            {
                return;
            }

            if (createParams.Length >= 3 && createParams[2] is bool)
            {
                // First set the return value to false. (fail)
                createParams[2] = false;
            }

            var myDataModel = this.DataModel as RuntimeDataModelRootDefinition;
            if (myDataModel == null)
            {
                return;
            }

            string nameSpace = createParams[0] as string;
            string typeName = createParams[1] as string;
            if (string.IsNullOrEmpty(typeName) == true)
            {
                return;
            }

            // Check if the given namespace and name has already been registered.
            RuntimeTypeInfo info =
                TypeManager.FindRuntimeType(nameSpace, typeName);
            if (info != null)
            {
                return;
            }

            // Create the new data model and add it to the list.
            var dataModelDef = new RuntimeDataModelDefinition()
            {
                FileName = string.Empty,
                Namespace = nameSpace,
                Name = typeName
            };

            myDataModel.DataModels.Add(dataModelDef);

            // Create a view model for the new data model definition.
            var child = new RuntimeDataModelDefinitionViewModel(this, dataModelDef);
            this.Children.Add(child);
            this.dataModelViewModels.Add(child);

            // Issue property changed event.
            this.OnPropertyChanged("DataModelDefinitionViewModels");

            // Set the new view model selected.
            this.SelectedDataModel = child;

            if (createParams.Length >= 3 && createParams[2] is bool)
            {
                // Set the return value to true. (succeed)
                createParams[2] = true;
            }

            // Trigger the event.
            if (DataModelListChanged != null)
            {
                DataModelListChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle execution for deleting a data model.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnDeleteDataModel(object parameter)
        {
            var viewModel = parameter as RuntimeDataModelDefinitionViewModel;
            if (viewModel == null)
            {
                return;
            }

            this.DeleteDataModel(viewModel);
        }
    }
}
