﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the editor data model definition.
    /// </summary>
    public class EditorDataModelDefinitionViewModel : ViewModelBase
    {
        /// <summary>The editor data model property view models, including deleted properties.</summary>
        private List<EditorDataModelPropertyDefinitionViewModel> allPropertyViewModels =
            new List<EditorDataModelPropertyDefinitionViewModel>();

        /// <summary>The editor data model property view models.</summary>
        private List<EditorDataModelPropertyDefinitionViewModel> propertyViewModels =
            new List<EditorDataModelPropertyDefinitionViewModel>();

        /// <summary>The editor data model using namespace view models.</summary>
        private List<EditorDataModelNamespaceDefinitionViewModel> usingNamespaceViewModels =
            new List<EditorDataModelNamespaceDefinitionViewModel>();

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EditorDataModelDefinitionViewModel(
            ViewModelBase parent,
            EditorDataModelDefinition dataModel) :
            base(parent, dataModel)
        {
            this.OnCommitEditingExecutable = new AnonymousExecutable(this.OnCommitEditing);
            this.OnCancelEditingExecutable = new AnonymousExecutable(this.OnCancelEditing);
            this.OnCreatePropertyExecutable = new AnonymousExecutable(this.OnCreateProperty);
            this.OnDeletePropertyExecutable = new AnonymousExecutable(this.OnDeleteProperty);
            this.OnMovePropertyExecutable = new AnonymousExecutable(this.OnMoveProperty);
            this.OnCreateUsingNamespaceExecutable = new AnonymousExecutable(this.OnCreateUsingNamespace);
            this.OnDeleteUsingNamespaceExecutable = new AnonymousExecutable(this.OnDeleteUsingNamespace);

            foreach (EditorDataModelPropertyDefinition def in dataModel.AllProperties)
            {
                var child = new EditorDataModelPropertyDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.allPropertyViewModels.Add(child);

                if (def.EditingVersionDefinition != null &&
                    def.EditingVersionDefinition.Action != VersionActions.Delete)
                {
                    this.propertyViewModels.Add(child);
                }
            }

            foreach (EditorDataModelUsingNamespaceDefinition def in dataModel.UsingNamespaces)
            {
                var child = new EditorDataModelNamespaceDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.usingNamespaceViewModels.Add(child);
            }
        }

        /// <summary>
        /// Static event triggered when the data model is modified.
        /// </summary>
        public static event EventHandler DataModelModified = null;

        /// <summary>
        /// Static event triggered when the data model property list is changed.
        /// </summary>
        public static event EventHandler DataModelPropertyListChanged = null;

        /// <summary>
        /// Get the data model name.
        /// </summary>
        public string DataModelName
        {
            get { return ((EditorDataModelDefinition)this.DataModel).Name; }
        }

        /// <summary>
        /// Enumerate the property definition view models, including deleted properties.
        /// </summary>
        public IEnumerable<EditorDataModelPropertyDefinitionViewModel> AllPropertyDefinitionViewModels
        {
            get { return this.allPropertyViewModels; }
        }

        /// <summary>
        /// Enumerate the property definition view models.
        /// </summary>
        public IEnumerable<EditorDataModelPropertyDefinitionViewModel> PropertyDefinitionViewModels
        {
            get { return this.propertyViewModels; }
        }

        /// <summary>
        /// Enumerate the property definition view models.
        /// </summary>
        public IEnumerable<EditorDataModelNamespaceDefinitionViewModel> UsingNamespaceDefinitionViewModels
        {
            get { return this.usingNamespaceViewModels; }
        }

        /// <summary>
        /// Enumerate the the list of namespaces the data model uses.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> UsingNamespaceNameValuePairList
        {
            get
            {
                foreach (dynamic child in this.usingNamespaceViewModels)
                {
                    yield return new KeyValuePair<string, object>(child.Name, child);
                }
            }
        }

        /// <summary>
        /// Enumerate the the list of property the data model has.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> PropertyAndTypeNameValuePairList
        {
            get
            {
                foreach (dynamic child in this.propertyViewModels)
                {
                    yield return new KeyValuePair<string, object>(child.PropertyTypeWithoutNamespace + " " + child.Name, child);
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether the data model
        /// has been deleted in the current version.
        /// </summary>
        public bool IsDeletedInCurrentVersion
        {
            get
            {
                var myDataModel = this.DataModel as EditorDataModelDefinition;
                if (myDataModel.DeletedVersion == null)
                {
                    return false;
                }

                var rootDataModel = myDataModel.FindParent<EditorDataModelRootDefinition>();
                if (myDataModel.DeletedVersion.Version > rootDataModel.EditingVersion.Version)
                {
                    return false;
                }

                return true;
            }
        }

        /// <summary>
        /// Get the comma-separated string of the name (with namespace) of the inheriting data models.
        /// </summary>
        public string InheritingDataModels
        {
            get
            {
                // Get the data model.
                var myDataModel = this.DataModel as EditorDataModelDefinition;
                if (myDataModel == null)
                {
                    return string.Empty;
                }

                // Get the Guid of the inheriting data models.
                var guidList = myDataModel.SuperClasses;
                if (guidList == null)
                {
                    return string.Empty;
                }

                string str = string.Empty;
                foreach (Guid guid in guidList)
                {
                    // Find the inheriting data model definition.
                    var def =
                        WorkspaceManager.FindDefinition(guid) as EditorDataModelDefinition;
                    if (def != null)
                    {
                        if (str.Length > 0)
                        {
                            str += ", ";
                        }

                        // Return the inheriting data model name.
                        if (string.IsNullOrEmpty(def.Namespace) == true)
                        {
                            str += def.Name;
                        }
                        else
                        {
                            str += def.Namespace + "." + def.Name;
                        }
                    }
                }

                return str;
            }
        }

        /// <summary>
        /// Enumerate the name (without namespace) of the inheriting data models.
        /// </summary>
        public IEnumerable<string> InheritingDataModelsWithoutNamespace
        {
            get
            {
                // Get the data model.
                var myDataModel = this.DataModel as EditorDataModelDefinition;
                if (myDataModel == null)
                {
                    yield break;
                }

                // Get the Guid of the inheriting data models.
                var guidList = myDataModel.SuperClasses;
                if (guidList == null)
                {
                    yield break;
                }

                foreach (Guid guid in guidList)
                {
                    // Find the inheriting data model definition.
                    var def =
                        WorkspaceManager.FindDefinition(guid) as EditorDataModelDefinition;
                    if (def != null)
                    {
                        // Return the inheriting data model name.
                        yield return def.Name;
                    }
                }
            }
        }

        /// <summary>
        /// Get the last created editor data model property view model.
        /// </summary>
        public EditorDataModelPropertyDefinitionViewModel LastCreatedProperty { get; private set; }

        /// <summary>
        /// Get the last created editor data model namespace view model.
        /// </summary>
        public EditorDataModelNamespaceDefinitionViewModel LastCreatedNamespace { get; private set; }

        /// <summary>
        /// Get or set the executable for creating a new property.
        /// </summary>
        public IExecutable OnCreatePropertyExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for deleting a property.
        /// </summary>
        public IExecutable OnDeletePropertyExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for moving a property up or down in the list.
        /// </summary>
        public IExecutable OnMovePropertyExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for creating a new namespace.
        /// </summary>
        public IExecutable OnCreateUsingNamespaceExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for deleting a namespace.
        /// </summary>
        public IExecutable OnDeleteUsingNamespaceExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for committing modification of the data model.
        /// </summary>
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for canceling modification of the data model.
        /// </summary>
        public IExecutable OnCancelEditingExecutable { get; set; }

        /// <summary>
        /// Notify the specified data model property has been changed.
        /// </summary>
        /// <param name="viewModel">The view model that holds the data model property.</param>
        public void NotifyPropertyDefinitionChanged(
            EditorDataModelPropertyDefinitionViewModel viewModel)
        {
            this.UpdateViewModelsForProperties();
        }

        /// <summary>
        /// Delete the specified data model property.
        /// </summary>
        /// <param name="viewModel">The view model of the property to delete.</param>
        public void DeleteProperty(
            EditorDataModelPropertyDefinitionViewModel viewModel)
        {
            int index = this.propertyViewModels.IndexOf(viewModel);
            if (index < 0)
            {
                return;
            }

            this.propertyViewModels.RemoveAt(index);

            var myDataModel = this.DataModel as EditorDataModelDefinition;
            if (myDataModel == null)
            {
                return;
            }

            var def = viewModel.DataModel as EditorDataModelPropertyDefinition;
            myDataModel.RemovePropertyFromEditingVersion(def);

            viewModel.Dispose();

            // Issue property changed event.
            this.NotifyPropertyDefinitionChanged(null);

            if (DataModelPropertyListChanged != null)
            {
                DataModelPropertyListChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Notify the specified data model using namespace has been changed.
        /// </summary>
        /// <param name="viewModel">The view model that holds the used data model namespace.</param>
        public void NotifyNamespaceDefinitionChanged(
            EditorDataModelNamespaceDefinitionViewModel viewModel)
        {
            this.OnPropertyChanged("UsingNamespaceDefinitionViewModels");
            this.OnPropertyChanged("UsingNamespaceNameValuePairList");
        }

        /// <summary>
        /// Delete the specified data model property.
        /// </summary>
        /// <param name="viewModel">The view model of the property to delete.</param>
        public void DeleteUsingNamespace(
            EditorDataModelNamespaceDefinitionViewModel viewModel)
        {
            int index = this.usingNamespaceViewModels.IndexOf(viewModel);
            if (index < 0)
            {
                return;
            }

            this.usingNamespaceViewModels.RemoveAt(index);

            var myDataModel = this.DataModel as EditorDataModelDefinition;
            if (myDataModel == null)
            {
                return;
            }

            var def = viewModel.DataModel as EditorDataModelUsingNamespaceDefinition;
            myDataModel.UsingNamespaces.Remove(def);

            viewModel.Dispose();
            def.Dispose();

            this.OnPropertyChanged("UsingNamespaceNameValuePairList");
            this.OnPropertyChanged("UsingNamespaceDefinitionViewModels");
        }

        /// <summary>
        /// Handle execution when committing modification of the data model.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCommitEditing(object parameter)
        {
            var values = parameter as KeyValuePair<string, object>[];
            if (values == null)
            {
                return;
            }

            try
            {
                foreach (var pair in values)
                {
                    var binder = new EffectMakerSetMemberBinder(pair.Key, true);
                    this.TrySetMember(binder, pair.Value);
                }

                // Dispatch the data model modified event.
                if (DataModelModified != null)
                {
                    DataModelModified(this, EventArgs.Empty);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "EditorDataModelDefinitionViewModel.OnCommitEditing => Failed setting values to the editor data model.");
                Logger.Log(LogLevels.Error, "Exception : {0}, message : {1}", ex.GetType().ToString(), ex.Message);
            }
        }

        /// <summary>
        /// Handle execution when canceling modification of the data model.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCancelEditing(object parameter)
        {
            var myParent = this.Parent as RootEditorDataModelDefinitionViewModel;
            if (myParent == null)
            {
                return;
            }

            myParent.DeleteDataModel(this);
        }

        /// <summary>
        /// Handle execution for creating a new property.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnCreateProperty(object parameter)
        {
            var myDataModel = this.DataModel as EditorDataModelDefinition;
            if (myDataModel == null)
            {
                return;
            }

            // Create a unique name for the property.
            string defaultName = "Property_";
            var regex = new Regex("^" + defaultName + "([0-9]+)$");

            List<int> existingSerialList = new List<int>();

            // Collect all the numbers that is used.
            foreach (EditorDataModelPropertyDefinition property in myDataModel.Properties)
            {
                Match match = regex.Match(property.Name);
                if (match.Groups.Count < 2)
                {
                    continue;
                }

                int currSerial;
                if (int.TryParse(match.Groups[1].Value, out currSerial) == true)
                {
                    existingSerialList.Add(currSerial);
                }
            }

            // Find an unused number for the default name.
            defaultName += this.FindUnusedSerialNumber(existingSerialList).ToString();

            // Find the root definition to get the latest version.
            var rootDef = myDataModel.FindParent<EditorDataModelRootDefinition>();

            // Create the new property and add it to the list.
            var newProperty = myDataModel.CreatePropertyForEditingVersion();
            newProperty.Description = string.Empty;
            newProperty.Name = defaultName;

            // Create a view model for the new property definition.
            var child = new EditorDataModelPropertyDefinitionViewModel(this, newProperty);
            this.Children.Add(child);
            this.propertyViewModels.Add(child);

            // Issue property changed event.
            this.NotifyPropertyDefinitionChanged(null);

            // Remember the created property.
            var binder = new EffectMakerSetMemberBinder("LastCreatedProperty", true);
            this.TrySetMember(binder, child);

            if (DataModelPropertyListChanged != null)
            {
                DataModelPropertyListChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle execution for deleting a property.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnDeleteProperty(object parameter)
        {
            var viewModel = parameter as EditorDataModelPropertyDefinitionViewModel;
            if (viewModel == null)
            {
                return;
            }

            this.DeleteProperty(viewModel);
        }

        /// <summary>
        /// Handle execution for moving a property up or down in the list.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnMoveProperty(object parameter)
        {
            var dataModelDef = this.DataModel as EditorDataModelDefinition;
            if (dataModelDef == null)
            {
                return;
            }

            var myParam = parameter as Tuple<object, int>;
            if (myParam == null || myParam.Item2 == 0)
            {
                return;
            }

            var propertyVM = myParam.Item1 as EditorDataModelPropertyDefinitionViewModel;
            if (propertyVM == null)
            {
                return;
            }

            var propertyDef = propertyVM.DataModel as EditorDataModelPropertyDefinition;
            if (propertyDef == null)
            {
                return;
            }

            int currIndex = dataModelDef.AllProperties.IndexOf(propertyDef);
            int newIndex = currIndex + myParam.Item2;

            dataModelDef.AllProperties.RemoveAt(currIndex);

            newIndex = Math.Min(dataModelDef.AllProperties.Count, Math.Max(0, newIndex));

            dataModelDef.AllProperties.Insert(newIndex, propertyDef);

            // First remove all the property view models.
            for (int i = this.Children.Count - 1; i >= 0; --i)
            {
                if (this.Children[i] is EditorDataModelPropertyDefinitionViewModel)
                {
                    this.Children[i].Dispose();
                    this.Children.RemoveAt(i);
                }
            }

            this.propertyViewModels.Clear();

            // Update the view models for the properties.
            EditorDataModelPropertyDefinitionViewModel viewModelToSelect = null;
            foreach (EditorDataModelPropertyDefinition def in dataModelDef.Properties)
            {
                var child = new EditorDataModelPropertyDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.propertyViewModels.Add(child);

                if (propertyDef == def)
                {
                    viewModelToSelect = child;
                }
            }

            // Issue property changed event.
            this.NotifyPropertyDefinitionChanged(null);

            // Remember the created property.
            var binder = new EffectMakerSetMemberBinder("LastCreatedProperty", true);
            this.TrySetMember(binder, viewModelToSelect);

            if (DataModelPropertyListChanged != null)
            {
                DataModelPropertyListChanged(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle execution for creating a new namespace.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnCreateUsingNamespace(object parameter)
        {
            var myDataModel = this.DataModel as EditorDataModelDefinition;
            if (myDataModel == null)
            {
                return;
            }

            // Create the new property and add it to the list.
            var newUsedNamespace = new EditorDataModelUsingNamespaceDefinition()
            {
                Name = "System"
            };

            myDataModel.UsingNamespaces.Add(newUsedNamespace);

            // Create a view model for the new property definition.
            var child = new EditorDataModelNamespaceDefinitionViewModel(this, newUsedNamespace);
            this.Children.Add(child);
            this.usingNamespaceViewModels.Add(child);

            // Issue property changed event.
            this.OnPropertyChanged("UsingNamespaceNameValuePairList");
            this.OnPropertyChanged("UsingNamespaceDefinitionViewModels");

            // Remember the created property.
            var binder = new EffectMakerSetMemberBinder("LastCreatedNamespace", true);
            this.TrySetMember(binder, child);
        }

        /// <summary>
        /// Handle execution for deleting a namespace.
        /// </summary>
        /// <param name="parameter">The execution parameter.</param>
        private void OnDeleteUsingNamespace(object parameter)
        {
            var viewModel = parameter as EditorDataModelNamespaceDefinitionViewModel;
            if (viewModel == null)
            {
                return;
            }

            this.DeleteUsingNamespace(viewModel);
        }

        /// <summary>
        /// Find an unused serial number from the given used serial number list.
        /// </summary>
        /// <param name="usedSerialNumList">The serial numbers that has already being used.</param>
        /// <returns>An unused serial number.</returns>
        private int FindUnusedSerialNumber(List<int> usedSerialNumList)
        {
            int serialNumber = 1;
            bool found = true;
            while (found == true)
            {
                found = false;
                foreach (int i in usedSerialNumList)
                {
                    if (i == serialNumber)
                    {
                        ++serialNumber;
                        found = true;
                        break;
                    }
                }
            }

            return serialNumber;
        }

        /// <summary>
        /// Update view models for the editor data model property definitions.
        /// </summary>
        private void UpdateViewModelsForProperties()
        {
            var myDataModel = this.DataModel as EditorDataModelDefinition;

            // First clear all property view models.
            var viewModels = this.allPropertyViewModels.ToArray();
            foreach (EditorDataModelPropertyDefinitionViewModel vm in viewModels)
            {
                this.Children.Remove(vm);
            }

            this.allPropertyViewModels.Clear();
            this.propertyViewModels.Clear();

            // Create view models again for the properties.
            foreach (EditorDataModelPropertyDefinition def in myDataModel.AllProperties)
            {
                var child = new EditorDataModelPropertyDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.allPropertyViewModels.Add(child);

                if (def.EditingVersionDefinition != null &&
                    def.EditingVersionDefinition.Action != VersionActions.Delete)
                {
                    this.propertyViewModels.Add(child);
                }
            }

            // Issue property changed event.
            this.OnPropertyChanged("PropertyAndTypeNameValuePairList");
            this.OnPropertyChanged("PropertyDefinitionViewModels");
            this.OnPropertyChanged("AllPropertyDefinitionViewModels");
        }
    }
}
