﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.UIControls.BasicControls;

using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.DataModelMaker.UIControls.SourceCodeEditor
{
    /// <summary>
    /// Dialog for source code editor.
    /// </summary>
    public partial class SourceCodeEditorDialog : BindableDialog
    {
        /// <summary>The tag for source code template.</summary>
        private const string SourceCodeTemplateTag = "{SourceCodeTemplate:Editable}";

        /// <summary>The original source code, for testing if the contents are modified.</summary>
        private string originalSourceCode = string.Empty;

        /// <summary>The backing field for the source code template.</summary>
        private string sourceCodeTemplate = string.Empty;

        /// <summary>The backing field for the source code indent.</summary>
        private int sourceCodeIndent = 0;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="sourceCodeName">The name of the source code to display on the dialog title.</param>
        /// <param name="dataContext">The data context to bind to.</param>
        /// <param name="dataSourcePropertyName">
        /// The property name of the data context that contains the source code string.
        /// </param>
        public SourceCodeEditorDialog(
            string sourceCodeName,
            object dataContext,
            string dataSourcePropertyName) : base()
        {
            this.InitializeComponent();

            this.Text = sourceCodeName;

            this.saveMenuItem.Click += (s, e) => this.CommitEditingSourceCode();
            this.discardChangeMenuItem.Click += (s, e) => this.UpdateFromDataSource();
            this.closeMenuItem.Click += (s, e) => this.Close();

            this.sourceCodeEditor.KeyDown += this.OnSourceCodeTextBoxKeyDown;
            this.sourceCodeEditor.KeyUp += this.OnSourceCodeTextBoxKeyUp;
            this.sourceCodeEditor.TextChanged += (s, e) => this.UpdateFullSourceCodeWithTemplate();

            this.AddBinding("SourceCode", dataSourcePropertyName);
            this.DataContext = dataContext;
        }

        /// <summary>
        /// Get or set the template for the source code.
        /// Use {SourceCodeTemplate:Editable} to mark the editable part.
        /// The rest of the text is fixed and cannot be modified.
        /// </summary>
        public string SourceCodeTemplate
        {
            get
            {
                return this.sourceCodeTemplate;
            }

            set
            {
                this.sourceCodeTemplate = value;
                this.UpdateFullSourceCodeWithTemplate();
            }
        }

        /// <summary>
        /// Get or set the indent count for the source code.
        /// </summary>
        public int SourceCodeIndent
        {
            get
            {
                return this.sourceCodeIndent;
            }

            set
            {
                this.sourceCodeIndent = value;
                this.UpdateFullSourceCodeWithTemplate();
            }
        }

        /// <summary>
        /// Get or set the source code to edit.
        /// </summary>
        public string SourceCode
        {
            get
            {
                return this.sourceCodeEditor.Text;
            }

            set
            {
                this.sourceCodeEditor.Text = value;
                this.originalSourceCode = value;
                this.UpdateFullSourceCodeWithTemplate();
            }
        }

        /// <summary>
        /// Handle Closing event for the dialog.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnClosing(CancelEventArgs e)
        {
            base.OnClosing(e);

            if (this.SourceCode != this.originalSourceCode)
            {
                DialogResult result = MessageBox.Show(
                    this,
                    Properties.Resources.SourceCodeEditorConfirmSaveChanges,
                    Properties.Resources.SourceCodeEditorConfirmSaveChangesCaption,
                    MessageBoxButtons.YesNoCancel,
                    MessageBoxIcon.Question,
                    MessageBoxDefaultButton.Button3);

                if (result == DialogResult.Yes)
                {
                    this.CommitEditingSourceCode();
                }
                else if (result == DialogResult.No)
                {
                    // Do nothing, just close the dialog without saving.
                }
                else
                {
                    e.Cancel = true;
                }
            }
        }

        /// <summary>
        /// Handle KeyDown event for the source code text box.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSourceCodeTextBoxKeyDown(object sender, KeyEventArgs e)
        {
            var tb = sender as TextBox;
            if (tb == null)
            {
                return;
            }

            if (e.KeyCode == Keys.Enter)
            {
                // Insert the same indent as the previous line.
                var currLine = tb.Lines[tb.GetLineFromCharIndex(tb.SelectionStart)];

                var str1 = tb.Text.Substring(0, tb.SelectionStart) +
                           Environment.NewLine +
                           this.GetLeadingSpaces(currLine);
                var str2 = tb.Text.Substring(tb.SelectionStart + tb.SelectionLength);

                tb.Text = str1 + str2;
                tb.Select(str1.Length, 0);

                e.SuppressKeyPress = true;
                e.Handled = true;
            }
            else if (e.KeyCode == Keys.Tab)
            {
                // Replace tabs with spaces.
                string[] tabSpaces = { "    ", "   ", "  ", " " };

                // Compute the current character index in the line.
                int lineIndex = tb.GetLineFromCharIndex(tb.SelectionStart);
                int charIndexInLine =
                    tb.SelectionStart - tb.GetFirstCharIndexFromLine(lineIndex);

                var str1 = tb.Text.Substring(0, tb.SelectionStart);
                var str2 = tabSpaces[charIndexInLine % 4];
                var str3 = tb.Text.Substring(tb.SelectionStart + tb.SelectionLength);

                tb.Text = str1 + str2 + str3;
                tb.Select(str1.Length + str2.Length, 0);

                e.SuppressKeyPress = true;
                e.Handled = true;
            }
        }

        /// <summary>
        /// Handle KeyDown event for the source code text box.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSourceCodeTextBoxKeyUp(object sender, KeyEventArgs e)
        {
            var tb = sender as TextBox;
            if (tb == null)
            {
                return;
            }

            switch (e.KeyCode)
            {
                case Keys.Enter:
                case Keys.Tab:
                    {
                        e.SuppressKeyPress = true;
                        e.Handled = true;
                        break;
                    }
            }
        }

        /// <summary>
        /// Get a string that contains the leading space characters of the given string.
        /// </summary>
        /// <param name="str">The string.</param>
        /// <returns>The string with all the leading space characters.</returns>
        private string GetLeadingSpaces(string str)
        {
            char ch;
            for (int i = 0; i < str.Length; ++i)
            {
                ch = str[i];
                if (ch != ' ' && ch != '\t')
                {
                    return str.Substring(0, i);
                }
            }

            return string.Empty;
        }

        /// <summary>
        /// Commits the editing contents. (the source code)
        /// </summary>
        private void CommitEditingSourceCode()
        {
            this.LogicalTreeElementExtender.NotifyPropertyChanged(
                BindingUpdateType.PropertyChanged,
                "SourceCode");

            // Update the bound data again from the data source.
            this.UpdateFromDataSource();
        }

        /// <summary>
        /// Update the bound data from data source and discard any
        /// changes that had been made to the data.
        /// </summary>
        private void UpdateFromDataSource()
        {
            foreach (Binder binder in this.Bindings)
            {
                binder.UpdateElement();
            }
        }

        /// <summary>
        /// Update full source code text box with the editing source code and the template.
        /// </summary>
        private void UpdateFullSourceCodeWithTemplate()
        {
            if (string.IsNullOrEmpty(this.SourceCodeTemplate) == true)
            {
                this.fullSourceCodeTextBox.Text = this.SourceCode;
            }
            else
            {
                string source = this.SourceCode;
                if (this.SourceCodeIndent > 0)
                {
                    source = RegexUtility.IndentString(
                        this.SourceCode,
                        Environment.NewLine,
                        this.SourceCodeIndent);
                }

                this.fullSourceCodeTextBox.Text =
                    this.SourceCodeTemplate.Replace(SourceCodeTemplateTag, source);
            }
        }
    }
}
