﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.Core.Converters;

using EffectMaker.DataModelMaker.UIControls.Extenders;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.SelectConverter
{
    /// <summary>
    /// The viewport class for rendering the converter list.
    /// </summary>
    internal class ConverterListViewport : Viewport
    {
        /// <summary>Constant for the margin around the view port.</summary>
        private const float ViewportMargin = 0.0f;

        /// <summary>The converter list item that the cursor is currently on.</summary>
        private ConverterListItemRenderable mouseOverItem = null;

        /// <summary>The converter list item that is selected.</summary>
        private ConverterListItemRenderable selectedItem = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="renderingControl">The control to be rendered to.</param>
        public ConverterListViewport(Control renderingControl) :
            base(renderingControl)
        {
            this.FillColor = Color.Black;

            foreach (ConverterInfo info in ConverterManager.Converters)
            {
                this.AddRenderable(new ConverterListItemRenderable(this)
                    {
                        ConverterInfo = info
                    });
            }
        }

        /// <summary>
        /// Enumerate all the converter renderables.
        /// </summary>
        public IEnumerable<ConverterListItemRenderable> ConverterRenderables
        {
            get
            {
                return from rd in this.Renderables
                       where rd is ConverterListItemRenderable
                       select rd as ConverterListItemRenderable;
            }
        }

        /// <summary>
        /// Get or set the converter list item that the cursor is currently hovering over.
        /// </summary>
        public ConverterListItemRenderable MouseOverItem
        {
            get
            {
                return this.mouseOverItem;
            }

            set
            {
                if (this.mouseOverItem != value)
                {
                    if (this.mouseOverItem != null)
                    {
                        this.mouseOverItem.IsMouseOver = false;
                    }

                    this.mouseOverItem = value;

                    if (this.mouseOverItem != null)
                    {
                        this.mouseOverItem.IsMouseOver = true;
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the converter list item that is selected.
        /// </summary>
        public ConverterListItemRenderable SelectedItem
        {
            get
            {
                return this.selectedItem;
            }

            set
            {
                if (this.selectedItem != value)
                {
                    if (this.selectedItem != null)
                    {
                        this.selectedItem.IsSelected = false;
                    }

                    this.selectedItem = value;

                    if (this.selectedItem != null)
                    {
                        this.selectedItem.IsSelected = true;
                    }
                }
            }
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            float childWidth = this.Width - (ViewportMargin * 2.0f) - 1.0f;

            // Loop through data model list items to update the content size.
            foreach (RenderableBase child in this.Renderables)
            {
                child.Size = new SizeF(childWidth, 10.0f);
            }

            base.UpdateChildren(context);

            float height = ViewportMargin;

            // Loop through converter list items to update the content size.
            foreach (RenderableBase child in this.Renderables)
            {
                if ((child is ConverterListItemRenderable) == false)
                {
                    continue;
                }

                // Setup location and size.
                child.Location = new PointF(ViewportMargin, height);

                height += child.Height;
            }

            height += ViewportMargin;

            this.Size = new SizeF(this.Width, height + 1.0f);
        }
    }
}
