﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.UIControls.DataModelEditor;
using EffectMaker.DataModelMaker.UIControls.DataModelPropertyEditor;

using EffectMaker.DataModelMaker.UILogic.ViewModels;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.ObjectPicking;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Render.ScrollBar;

using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.DataModelMaker.UIControls.DataModelView
{
    /// <summary>
    /// Panel to show the information of a data model.
    /// </summary>
    internal class DataModelInfoPanel : Panel
    {
        /// <summary>Constant for the factor applying to the vertical mouse scroll speed.</summary>
        private const int VerticalScrollSpeedFactor = 50;

        /// <summary>The viewport for rendering the data model information.</summary>
        private DataModelInfoViewport dataModelInfoViewport = null;

        /// <summary>The scroll bars.</summary>
        private AutoHiddenScrollBar scrollBars = null;

        /// <summary>The data model property item that the cursor is hovering over.</summary>
        private DataModelInfoPropertyRenderable mouseOverItem = null;

        /// <summary>The data model property item that is currently selected.</summary>
        private DataModelInfoPropertyRenderable selectedItem = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public DataModelInfoPanel()
        {
            // Enable double buffer.
            this.DoubleBuffered = true;

            // Make the panel selectable.
            this.SetStyle(ControlStyles.Selectable, true);
            this.TabStop = true;

            // Create the viewport for rendering the data model information.
            this.dataModelInfoViewport = new DataModelInfoViewport(this)
            {
                Bounds = new Rectangle(
                    0,
                    0,
                    this.ClientRectangle.Width,
                    this.ClientRectangle.Height)
            };

            // Create and set up the scroll bars.
            this.scrollBars = new AutoHiddenScrollBar(this)
            {
                ForeColor = Color.Black,
                BackColor = Color.Black,
                IncrementV = VerticalScrollSpeedFactor
            };

            this.scrollBars.RenderRequired += (s, e) => { this.Invalidate(); };
            this.scrollBars.Scroll += (s, e) =>
            {
                this.dataModelInfoViewport.Translation =
                    new PointF(-this.ScrollPosition.X, -this.ScrollPosition.Y);
                this.Invalidate();
            };

            this.scrollBars.SetContentSize(new Size(
                (int)this.dataModelInfoViewport.ContentSize.Width,
                (int)this.dataModelInfoViewport.ContentSize.Height));
        }

        /// <summary>
        /// Get or set the data context.
        /// </summary>
        public object DataContext
        {
            get { return this.dataModelInfoViewport.DataContext; }
            set { this.dataModelInfoViewport.DataContext = value; }
        }

        /// <summary>
        /// Get or set the scroll position.
        /// </summary>
        public Point ScrollPosition
        {
            get { return this.scrollBars.ScrollPosition; }
            set { this.SetScrollPosition(value.X, value.Y); }
        }

        /// <summary>
        /// Get or set the data model property item that the cursor is currently hovering over.
        /// </summary>
        public DataModelInfoPropertyRenderable MouseOverItem
        {
            get
            {
                return this.mouseOverItem;
            }

            set
            {
                if (this.mouseOverItem != value)
                {
                    if (this.mouseOverItem != null)
                    {
                        this.mouseOverItem.IsMouseOver = false;
                    }

                    this.mouseOverItem = value;

                    if (this.mouseOverItem != null)
                    {
                        this.mouseOverItem.IsMouseOver = true;
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the selected data model property item.
        /// </summary>
        public DataModelInfoPropertyRenderable SelectedItem
        {
            get
            {
                return this.selectedItem;
            }

            set
            {
                if (this.selectedItem != value)
                {
                    if (this.selectedItem != null)
                    {
                        this.selectedItem.IsSelected = false;
                    }

                    this.selectedItem = value;

                    if (this.selectedItem != null)
                    {
                        this.selectedItem.IsSelected = true;
                    }
                }
            }
        }

        /// <summary>
        /// Set scroll position.
        /// </summary>
        /// <param name="x">The horizontal position.</param>
        /// <param name="y">The vertical position.</param>
        public void SetScrollPosition(int x, int y)
        {
            var contentSize = new Size(
                (int)this.dataModelInfoViewport.ContentSize.Width,
                (int)this.dataModelInfoViewport.ContentSize.Height);

            int maxX = contentSize.Width - this.Width;
            int maxY = contentSize.Height - this.Height;

            int posX = Math.Max(Math.Min(x, maxX), 0);
            int posY = Math.Max(Math.Min(y, maxY), 0);

            this.scrollBars.ScrollPosition = new Point(posX, posY);

            this.dataModelInfoViewport.Translation = new PointF(-posX, -posY);
        }

        /// <summary>
        /// Clean up any resources being used.
        /// </summary>
        /// <param name="disposing">true if managed resources should be disposed; otherwise, false.</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing == true)
            {
                if (this.dataModelInfoViewport != null)
                {
                    this.dataModelInfoViewport.Dispose();
                }
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// Handle MouseWheel event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            int increment = -(e.Delta / 120) * VerticalScrollSpeedFactor;

            this.SetScrollPosition(this.ScrollPosition.X, this.ScrollPosition.Y + increment);

            this.Invalidate();
        }

        /// <summary>
        /// Handle MouseDoubleClick event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDoubleClick(MouseEventArgs e)
        {
            base.OnMouseDoubleClick(e);

            // Do mouse picking.
            ObjectPickContext context = this.dataModelInfoViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0 &&
                context.PickedObjects[0] is DataModelInfoPropertyRenderable)
            {
                this.MouseOverItem =
                    context.PickedObjects[0] as DataModelInfoPropertyRenderable;
                if (this.MouseOverItem != null)
                {
                    var dialog = new DataModelPropertyEditorDialog()
                    {
                        DataSource = this.MouseOverItem.DataContext,
                        StartPosition = FormStartPosition.CenterParent
                    };

                    dialog.ShowDialog(this);
                }
            }
        }

        /// <summary>
        /// Handle MouseDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            // Do mouse picking.
            ObjectPickContext context = this.dataModelInfoViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0 &&
                context.PickedObjects[0] is DataModelInfoPropertyRenderable)
            {
                this.SelectedItem =
                    context.PickedObjects[0] as DataModelInfoPropertyRenderable;
            }
            else if (this.SelectedItem != null)
            {
                this.SelectedItem = null;
            }
        }

        /// <summary>
        /// Handle MouseUp event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);
        }

        /// <summary>
        /// Handle MouseMove event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (this.Focused == false)
            {
                this.Focus();
            }

            base.OnMouseMove(e);

            // Do mouse picking.
            ObjectPickContext context = this.dataModelInfoViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0 &&
                context.PickedObjects[0] is DataModelInfoPropertyRenderable)
            {
                this.MouseOverItem =
                    context.PickedObjects[0] as DataModelInfoPropertyRenderable;
            }
            else if (this.MouseOverItem != null)
            {
                this.MouseOverItem = null;
            }
        }

        /// <summary>
        /// Handle MouseLeave event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);

            if (this.MouseOverItem != null)
            {
                this.MouseOverItem = null;
            }
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            if (this.dataModelInfoViewport != null)
            {
                this.dataModelInfoViewport.Size = this.ClientSize;
            }

            Point scrollPos = this.scrollBars.ScrollPosition;

            this.scrollBars.SetViewSize(this.Size);
            this.scrollBars.SetDisplayRectangle(this.DisplayRectangle);

            // Reset to the same scroll position again.
            this.scrollBars.ScrollPosition = scrollPos;
        }

        /// <summary>
        /// Handle Paint event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            e.Graphics.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.HighQuality;
            e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;

            // Render the viewport.
            if (this.dataModelInfoViewport != null && this.scrollBars != null)
            {
                this.dataModelInfoViewport.Update(e.Graphics);
                this.dataModelInfoViewport.Draw(e.Graphics);

                // The data model list is updated, now set the updated content size to the scroll bars.
                this.scrollBars.SetContentSize(new Size(
                    (int)this.dataModelInfoViewport.ContentSize.Width,
                    (int)this.dataModelInfoViewport.ContentSize.Height));

                // Render the scroll bars.
                this.scrollBars.Render(e.Graphics);
            }
        }
    }
}
