﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.UIControls.Interfaces;

using EffectMaker.DataModelMaker.UILogic.ViewModels;

using EffectMaker.UIControls.Layout;

namespace EffectMaker.DataModelMaker.UIControls.DataModelPropertyEditor
{
    /// <summary>
    /// Dialog class for editing editor data model property.
    /// </summary>
    public partial class DataModelPropertyEditorDialog : Form
    {
        /// <summary>The data model property editor panel.</summary>
        private IDataModelEditor propertyEditorPanel = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public DataModelPropertyEditorDialog()
        {
            this.InitializeComponent();
        }

        /// <summary>
        /// Get or set the flag indicating whether should delete the data source if the editing
        /// is canceled.
        /// </summary>
        public bool ShouldDeleteDataSourceOnCancel { get; set; }

        /// <summary>
        /// Get or set the data source for the editor.
        /// </summary>
        public object DataSource
        {
            get
            {
                if (this.propertyEditorPanel == null)
                {
                    return null;
                }

                return this.propertyEditorPanel.DataContext;
            }

            set
            {
                if (value == null)
                {
                    return;
                }

                if (value is EditorDataModelPropertyDefinitionViewModel)
                {
                    var editor = new EditorDataModelPropertyEditor();

                    this.mainTableLayout.Controls.Add(editor, 0, 0);
                    this.mainTableLayout.SetColumnSpan(editor, 3);

                    editor.Dock = DockStyle.Fill;
                    editor.Visibility = Visibility.Visible;

                    editor.DataContext = value;

                    this.Size = new System.Drawing.Size(820, 500);

                    this.propertyEditorPanel = editor;
                }
                else if (value is RuntimeDataModelPropertyDefinitionViewModel)
                {
                    var editor = new RuntimeDataModelPropertyEditor();

                    this.mainTableLayout.Controls.Add(editor, 0, 0);
                    this.mainTableLayout.SetColumnSpan(editor, 3);

                    editor.Dock = DockStyle.Fill;
                    editor.Visibility = Visibility.Visible;

                    editor.DataContext = value;

                    this.Size = new System.Drawing.Size(571, 386);

                    this.propertyEditorPanel = editor;
                }
                else
                {
                    return;
                }

                this.UpdateDialogSize();
            }
        }

        /// <summary>
        /// Update dialog size according to the editor panel size.
        /// </summary>
        public void UpdateDialogSize()
        {
            var panel = this.propertyEditorPanel as Control;
            if (panel == null)
            {
                return;
            }

            int size = 0;
            foreach (Control child in panel.Controls)
            {
                size = Math.Max(size, child.Bottom);
            }

            this.Size = new System.Drawing.Size(this.Width, size + 64);
        }

        /// <summary>
        /// Handle FormClosing event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            if (this.DialogResult == DialogResult.OK)
            {
                if (this.propertyEditorPanel.CommitEditing() == false)
                {
                    e.Cancel = true;
                }
            }
            else
            {
                this.propertyEditorPanel.CancelEditing(this.ShouldDeleteDataSourceOnCancel);
            }

            base.OnFormClosing(e);
        }
    }
}
