﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.BinaryDataView
{
    /// <summary>
    /// Class for rendering data model list item.
    /// </summary>
    internal class BinaryDataListItemRenderable : RoundedRectangleShape, ILogicalTreeElement
    {
        /// <summary>The large font for rendering.</summary>
        private static Font largeFont = null;

        /// <summary>The small font for rendering.</summary>
        private static Font smallFont = null;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The flag indicating if mouse is over this item.</summary>
        private bool isMouseOver = false;

        /// <summary>The flag indicating if the item is selected.</summary>
        private bool isSelected = false;

        /// <summary>The name of the binary data.</summary>
        private string binaryDataName = string.Empty;

        /// <summary>The name of the file the binary data definition outputs to.</summary>
        private string fileName = string.Empty;

        /// <summary>
        /// Static constructor.
        /// </summary>
        static BinaryDataListItemRenderable()
        {
            // Setup the fonts for rendering.
            largeFont = new Font(FontFamily.GenericSansSerif, 14.0f, FontStyle.Bold);
            smallFont = new Font(FontFamily.GenericSansSerif, 8.0f);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public BinaryDataListItemRenderable(BinaryDataListViewport owner) :
            base()
        {
            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            this.Owner = owner;

            this.SetCornerRadius(5.0f);
            this.Size = new SizeF(0.0f, 45.0f);

            // Set up binding.
            this.Bindings.Add(new Binder(this, "BinaryDataName", "Name"));
            this.Bindings.Add(new Binder(this, "FileName", "FileName"));

            this.UpdateAppearance();
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the owner binary data list renderable.
        /// </summary>
        public BinaryDataListViewport Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner; }
            protected set { this.Owner = value as BinaryDataListViewport; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get or set the flag indicating if mouse is over this item.
        /// </summary>
        public bool IsMouseOver
        {
            get
            {
                return this.isMouseOver;
            }

            set
            {
                if (this.isMouseOver != value)
                {
                    this.isMouseOver = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating if this item is selected.
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (this.isSelected != value)
                {
                    this.isSelected = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Get or set the name of the binary data.
        /// </summary>
        public string BinaryDataName
        {
            get
            {
                return this.binaryDataName;
            }

            set
            {
                this.binaryDataName = value;
                this.RequestRedraw();
            }
        }

        /// <summary>
        /// Get or set the name of the file the binary data definition outputs to.
        /// </summary>
        public string FileName
        {
            get
            {
                return this.fileName;
            }

            set
            {
                this.fileName = value;
                this.RequestRedraw();
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            var rect = this.TransformedBounds;

            // Compute the text size of the file name.
            SizeF fileNameTextSize = g.MeasureString(
                this.FileName,
                BinaryDataListItemRenderable.smallFont);

            // Compute the text size of the binary data name.
            SizeF binaryDataNameTextSize = g.MeasureString(
                this.BinaryDataName,
                BinaryDataListItemRenderable.largeFont);

            // Compute the rectangle for rendering.
            RectangleF fileNameTextRect = rect;
            fileNameTextRect.X += 3.0f;
            fileNameTextRect.Y += 6.0f;
            fileNameTextRect.Width = rect.Width - 3.0f;
            fileNameTextRect.Height = fileNameTextSize.Height;

            // Compute the rectangle for rendering.
            RectangleF binaryDataNameTextRect = rect;
            binaryDataNameTextRect.X += 2.0f;
            binaryDataNameTextRect.Y = rect.Bottom - binaryDataNameTextSize.Height - 5.0f;
            binaryDataNameTextRect.Width = rect.Width - 2.0f;
            binaryDataNameTextRect.Height = binaryDataNameTextSize.Height;

            using (var strFormat = new StringFormat())
            {
                // Top-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Far;

                strFormat.Trimming = StringTrimming.EllipsisPath;

                g.DrawString(
                    this.FileName,
                    BinaryDataListItemRenderable.smallFont,
                    Brushes.DarkCyan,
                    fileNameTextRect,
                    strFormat);

                strFormat.LineAlignment = StringAlignment.Near;
                strFormat.Trimming = StringTrimming.EllipsisCharacter;

                g.DrawString(
                    this.BinaryDataName,
                    BinaryDataListItemRenderable.largeFont,
                    Brushes.DarkBlue,
                    binaryDataNameTextRect,
                    strFormat);
            }
        }

        /// <summary>
        /// Update the appearance of the item.
        /// </summary>
        protected virtual void UpdateAppearance()
        {
            if (this.IsSelected == true)
            {
                this.BorderThickness = 2.0f;
                this.BorderColor = Color.Red;
                this.FillColor = Color.Khaki;
            }
            else if (this.IsMouseOver == true)
            {
                this.BorderThickness = 0.0f;
                this.BorderColor = Color.Transparent;
                this.FillColor = Color.Khaki;
            }
            else
            {
                this.BorderThickness = 0.0f;
                this.BorderColor = Color.Transparent;
                this.FillColor = Color.Beige;
            }
        }
    }
}
