﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;

using EffectMaker.DataModelLogic;

using EffectMaker.DataModelMaker.Core.Converters;
using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.DataModelMaker.UIControls.ConversionView;
using EffectMaker.DataModelMaker.UIControls.Selection;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.ObjectPicking;
using EffectMaker.Foundation.Render.Renderable;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.BinaryDataView
{
    /// <summary>
    /// Class for rendering binary data property information.
    /// </summary>
    internal class BinaryDataInfoFieldRenderable :
        RectangleShape,
        ILogicalTreeElement,
        IDestinationConnectible
    {
        /// <summary>Constant for the margin of the child renderables.</summary>
        public const float ChildMargin = 5.0f;

        /// <summary>Constant for the height of the fields when it contains no field group.</summary>
        private const float EmptyFieldHeight = 20.0f;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The flag indicating whether should show the connector or not.</summary>
        private bool shouldShowConnector = false;

        /// <summary>The renderable for rendering the binary field group.</summary>
        private BinaryDataInfoFieldGroupRenderable fieldGroupRenderable = null;

        /// <summary>The connection renderables.</summary>
        private List<ConnectionRenderable> connectionRenderables =
            new List<ConnectionRenderable>();

        /// <summary>The Guid of the connected sources.</summary>
        private IEnumerable<Guid> connectedSources = null;

        /// <summary>The target Guid.</summary>
        private Guid targetGuid = Guid.Empty;

        /// <summary>The flag indicating whether the target is a binary data.</summary>
        private bool isTargetBinaryData = false;

        /// <summary>The converter used for converting data from the connected data model properties.</summary>
        private ConverterInfo converter = null;

        /// <summary>The flag indicating if mouse is over this item.</summary>
        private bool isMouseOver = false;

        /// <summary>The flag indicating if the item is selected.</summary>
        private bool isSelected = false;

        /// <summary>
        /// Static constructor.
        /// </summary>
        static BinaryDataInfoFieldRenderable()
        {
            // Set up the fonts for rendering.
            PropertyTypeFont = new Font(FontFamily.GenericSansSerif, 8.0f);
            PropertyNameFont = new Font(FontFamily.GenericSansSerif, 8.0f);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public BinaryDataInfoFieldRenderable(RenderableBase owner) :
            base()
        {
            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            this.Size = new SizeF(0.0f, EmptyFieldHeight);

            this.BorderColor = Color.Transparent;
            this.BorderThickness = 0.0f;
            this.FillColor = Color.Transparent;

            this.Owner = owner;

            this.Connector = new ConnectorRenderable(this)
            {
                ConnectorLength = 20.0f,
                ConnectorDirection = ConnectorRenderable.ConnectorDirections.Left,
                Offset = new SizeF(0.0f, EmptyFieldHeight * 0.5f)
            };

            this.AddRenderable(this.Connector);

            // Set up binding.
            this.Bindings.Add(new Binder(this, "IsBinaryData", "IsBinaryData"));
            this.Bindings.Add(new Binder(this, "TargetName", "TargetName"));
            this.Bindings.Add(new Binder(this, "TargetType", "TargetType"));
            this.Bindings.Add(new Binder(this, "Description", "TargetDescription"));
            this.Bindings.Add(new Binder(this, "TargetGuid", "OutputFieldGuid"));
            this.Bindings.Add(new Binder(this, "Converter", "ConverterInfo"));
            this.Bindings.Add(new Binder(this, "FieldType", "FieldType"));
            this.Bindings.Add(new Binder(this, "FieldSize", "FieldSize"));
            this.Bindings.Add(new Binder(this, "BinaryFieldGroupDataSource", "BinaryFieldGroupViewModel"));

            this.Bindings.Add(new Binder(this, "ConnectedSources", "ConnectedSources"));
            this.Bindings.Add(new Binder(this, "OnAddConnectionSource", "OnAddConnectionSource"));
            this.Bindings.Add(new Binder(this, "OnRemoveConnectionSource", "OnRemoveConnectionSource"));
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the font for rendering the property type.
        /// </summary>
        public static Font PropertyTypeFont { get; private set; }

        /// <summary>
        /// Get the font for rendering the property name.
        /// </summary>
        public static Font PropertyNameFont { get; private set; }

        /// <summary>
        /// Get the owner field group.
        /// </summary>
        public RenderableBase Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner as ILogicalTreeElement; }
            protected set { this.Owner = value as RenderableBase; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get the connector.
        /// </summary>
        public ConnectorRenderable Connector { get; private set; }

        /// <summary>
        /// Get or set the binary field type.
        /// </summary>
        public BinaryFieldTypes FieldType { get; set; }

        /// <summary>
        /// Get or set the binary field size.
        /// </summary>
        public int FieldSize { get; set; }

        /// <summary>
        /// Get the flag indicating whether the connection of the connectible is set.
        /// </summary>
        public bool IsConnectionSet
        {
            get
            {
                if (this.FieldType == BinaryFieldTypes.Normal)
                {
                    if (this.connectedSources == null ||
                        this.connectedSources.Count() <= 0)
                    {
                        return false;
                    }
                }

                return true;
            }
        }

        /// <summary>
        /// Get the location of the connector.
        /// </summary>
        public PointF ConnectorLocation
        {
            get
            {
                if (this.Connector.Visible == false &&
                    this.fieldGroupRenderable != null)
                {
                    return this.fieldGroupRenderable.ConnectorLocation;
                }
                else
                {
                    return this.Connector.ConnectorLocation;
                }
            }
        }

        /// <summary>
        /// Get the transformed location of the connector.
        /// </summary>
        public PointF TransformedConnectorLocation
        {
            get
            {
                if (this.Connector.Visible == false &&
                    this.fieldGroupRenderable != null)
                {
                    return this.fieldGroupRenderable.TransformedConnectorLocation;
                }
                else
                {
                    return this.Connector.TransformedConnectorLocation;
                }
            }
        }

        /// <summary>
        /// Enumerate the Guid of the connected sources.
        /// </summary>
        public IEnumerable<Guid> ConnectedSources
        {
            get
            {
                return this.connectedSources;
            }

            set
            {
                this.connectedSources = value;
                this.UpdateConnections();
            }
        }

        /// <summary>
        /// Enumerate connections that connect to the connectible.
        /// </summary>
        public IEnumerable<ConnectionRenderable> Connections
        {
            get { return this.connectionRenderables; }
        }

        /// <summary>
        /// Get or set the converter for converting the data from the source connectible.
        /// </summary>
        public ConverterInfo Converter
        {
            get
            {
                return this.converter;
            }

            set
            {
                if (this.converter != value)
                {
                    this.converter = value;
                    this.LogicalTreeElementExtender.NotifyPropertyChanged();
                }
            }
        }

        /// <summary>
        /// Get or set the executable for adding a connection source.
        /// </summary>
        public IExecutable OnAddConnectionSource { get; set; }

        /// <summary>
        /// Get or set the executable for removing a connection source.
        /// </summary>
        public IExecutable OnRemoveConnectionSource { get; set; }

        /// <summary>
        /// Get or set description of the target.
        /// </summary>
        public string Description { get; set; }

        /// <summary>
        /// Get the string as the type of the connectible value.
        /// </summary>
        public string ValueType
        {
            get { return this.TargetType; }
        }

        /// <summary>
        /// Get the name of the connectible.
        /// </summary>
        public string DisplayName
        {
            get { return this.TargetName; }
        }

        /// <summary>
        /// Get or set the flag indicating whether this field is a binary data.
        /// </summary>
        public bool IsBinaryData { get; set; }

        /// <summary>
        /// Get or set name of the target name.
        /// </summary>
        public string TargetName { get; set; }

        /// <summary>
        /// Get or set type of the target.
        /// </summary>
        public string TargetType { get; set; }

        /// <summary>
        /// Get or set the Guid of the target.
        /// </summary>
        public Guid TargetGuid
        {
            get
            {
                return this.targetGuid;
            }

            set
            {
                this.targetGuid = value;

                var def = WorkspaceManager.FindDefinition(value) as BinaryDataDefinition;
                if (def != null)
                {
                    this.isTargetBinaryData = true;

                    if (this.fieldGroupRenderable == null)
                    {
                        this.fieldGroupRenderable = new BinaryDataInfoFieldGroupRenderable(this)
                        {
                            Bounds = new RectangleF(
                                0.0f,
                                ChildMargin,
                                this.Width,
                                0.0f)
                        };

                        this.AddRenderable(this.fieldGroupRenderable);
                    }

                    this.fieldGroupRenderable.IsTargetBinaryData = true;
                    this.fieldGroupRenderable.TargetGuid = value;
                    this.fieldGroupRenderable.TargetName = def.Name;
                    this.fieldGroupRenderable.Visible = true;
                    this.shouldShowConnector = false;
                }
                else
                {
                    this.isTargetBinaryData = false;
                    this.shouldShowConnector = true;
                }
            }
        }

        /// <summary>
        /// Get the binary field group renderable.
        /// </summary>
        public BinaryDataInfoFieldGroupRenderable BinaryFieldGroupRenderable
        {
            get { return this.fieldGroupRenderable; }
        }

        /// <summary>
        /// Get or set the data source of the binary field group.
        /// </summary>
        public object BinaryFieldGroupDataSource
        {
            get
            {
                if (this.fieldGroupRenderable == null)
                {
                    return null;
                }

                return this.fieldGroupRenderable.DataContext;
            }

            set
            {
                if (this.fieldGroupRenderable != null)
                {
                    this.fieldGroupRenderable.DataContext = value;
                }
                else if (value != null)
                {
                    this.fieldGroupRenderable = new BinaryDataInfoFieldGroupRenderable(this)
                    {
                        DataContext = value,
                        Bounds = new RectangleF(
                            0.0f,
                            ChildMargin,
                            this.Width,
                            0.0f)
                    };

                    this.AddRenderable(this.fieldGroupRenderable);
                }

                if (this.fieldGroupRenderable != null)
                {
                    if (value != null || this.isTargetBinaryData == true)
                    {
                        this.fieldGroupRenderable.Visible = true;
                        this.shouldShowConnector = false;
                    }
                    else
                    {
                        this.fieldGroupRenderable.Visible = false;
                        this.shouldShowConnector = true;
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating if mouse is over this item.
        /// </summary>
        public bool IsMouseOver
        {
            get
            {
                return this.isMouseOver;
            }

            set
            {
                if (this.isMouseOver != value)
                {
                    this.isMouseOver = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating if this item is selected.
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (this.isSelected != value)
                {
                    this.isSelected = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Determine if the specified target can be connected to this connectible item.
        /// </summary>
        /// <param name="target">The connection target.</param>
        /// <returns>True if can connect.</returns>
        public bool CanConnect(IConnectible target)
        {
            if (target == null ||
                target is IDestinationConnectible)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Update all the connections.
        /// </summary>
        public void UpdateConnections()
        {
            var selectedObj =
                SelectionManager.GetSelectedObject(SelectionGroups.BinaryConvertConnection);
            if (selectedObj is ConnectionRenderable &&
                this.connectionRenderables.Contains(selectedObj) == true)
            {
                // If any of the selections are selected, deselect them first.
                SelectionManager.ClearSelection(SelectionGroups.BinaryConvertConnection);
                SelectionManager.ClearMouseOver(SelectionGroups.BinaryConvertConnection);
            }

            // First remove all the connection renderables.
            this.RemoveRenderableRange(this.connectionRenderables);

            foreach (ConnectionRenderable connection in this.connectionRenderables)
            {
                connection.Dispose();
            }

            this.connectionRenderables.Clear();

            foreach (Guid guid in this.ConnectedSources)
            {
                ISourceConnectible source =
                    ConnectionSourceManager.FindSourceConnectible(guid);
                DefinitionBase def = WorkspaceManager.FindDefinition(guid);
                if (source == null)
                {
                    continue;
                }

                var connection = new ConnectionRenderable()
                {
                    Source = source
                };

                if (this.shouldShowConnector == true)
                {
                    connection.Destination = this;
                }
                else
                {
                    connection.Destination = this.fieldGroupRenderable;
                }

                this.connectionRenderables.Add(connection);
            }

            this.AddRenderableRange(this.connectionRenderables);

            // Update connections of the child field group.
            if (this.BinaryFieldGroupRenderable != null)
            {
                this.BinaryFieldGroupRenderable.UpdateConnections();
            }
        }

        /// <summary>
        /// Add a new connection source.
        /// </summary>
        /// <param name="sourceGuid">The Guid of the connection source.</param>
        public void AddConnectionSource(Guid sourceGuid)
        {
            if (this.OnAddConnectionSource != null)
            {
                this.OnAddConnectionSource.Execute(sourceGuid);
            }
        }

        /// <summary>
        /// Remove existing connection sources.
        /// </summary>
        /// <param name="sourceGuidList">The list of Guid of the connection sources.</param>
        public void RemoveConnectionSources(IEnumerable<Guid> sourceGuidList)
        {
            if (this.OnRemoveConnectionSource != null)
            {
                this.OnRemoveConnectionSource.Execute(sourceGuidList);
            }
        }

        /// <summary>
        /// Release the resources the renderable uses.
        /// </summary>
        /// <param name="isDisposing">True if the renderable is being disposed.</param>
        protected override void Dispose(bool isDisposing)
        {
            base.Dispose(isDisposing);

            foreach (ConnectionRenderable connection in this.connectionRenderables)
            {
                connection.Dispose();
            }

            this.connectionRenderables.Clear();
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            // Adjust the field group's width according to mine.
            if (this.fieldGroupRenderable != null && this.fieldGroupRenderable.Visible == true)
            {
                this.fieldGroupRenderable.Size =
                    new SizeF(this.Width, this.fieldGroupRenderable.Height);
            }

            var myRC = context as ConnectorRenderContext;
            if (myRC != null &&
                myRC.ShowConnector == true &&
                this.shouldShowConnector == true)
            {
                // Update connector length
                float length = 15.0f;
                ILogicalTreeElement element = this.Parent;
                while (element != null)
                {
                    if (element is BinaryDataInfoFieldRenderable)
                    {
                        length += BinaryDataInfoFieldRenderable.ChildMargin;
                    }

                    element = element.Parent;
                }

                this.Connector.Visible = true;
                this.Connector.ConnectorLength = length;
            }
            else
            {
                this.Connector.Visible = false;
            }

            base.UpdateChildren(context);

            // Adjust my own height according to the field group's height.
            if (this.fieldGroupRenderable != null && this.fieldGroupRenderable.Visible == true)
            {
                this.Size =
                    new SizeF(this.Width, this.fieldGroupRenderable.Height + (ChildMargin * 2.0f));
            }
            else
            {
                this.Size = new SizeF(this.Width, EmptyFieldHeight);
            }
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            // Do not render if contains a field group.
            if (this.fieldGroupRenderable != null &&
                this.fieldGroupRenderable.Visible == true)
            {
                return;
            }

            const float SpaceWidth = 5.0f;

            string typeString;
            if (this.IsBinaryData == true)
            {
                typeString = Properties.Resources.BinaryFieldGroupTargetTypeBinaryData;
            }
            else
            {
                typeString = this.TargetType;
            }

            SizeF typeTextSize = g.MeasureString(
                typeString,
                BinaryDataInfoFieldRenderable.PropertyTypeFont);

            SizeF nameTextSize = g.MeasureString(
                this.TargetName,
                BinaryDataInfoFieldRenderable.PropertyNameFont);

            // The value type text rectangle.
            RectangleF valueTypeTextRect = this.TransformedBounds;
            valueTypeTextRect.Width = typeTextSize.Width;

            // The property name occupies the right side of the item.
            RectangleF propertyNameRect = this.TransformedBounds;
            propertyNameRect.X += typeTextSize.Width + SpaceWidth;
            propertyNameRect.Width = nameTextSize.Width;

            using (var strFormat = new StringFormat())
            {
                // Bottom-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Center;

                strFormat.Trimming = StringTrimming.EllipsisPath;

                g.DrawString(
                    typeString,
                    BinaryDataInfoFieldRenderable.PropertyTypeFont,
                    Brushes.DimGray,
                    valueTypeTextRect,
                    strFormat);

                strFormat.Trimming = StringTrimming.EllipsisCharacter;

                g.DrawString(
                    this.TargetName,
                    BinaryDataInfoFieldRenderable.PropertyNameFont,
                    Brushes.Blue,
                    propertyNameRect,
                    strFormat);
            }
        }

        /// <summary>
        /// Draw children items.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        /// <returns>True if any children is rendered.</returns>
        protected override bool DrawChildren(Graphics g)
        {
            if (this.Visible == false)
            {
                return false;
            }

            bool isAnyChildRendered = false;

            // Always render the connections first, so they will be rendered under the connectors.
            foreach (ConnectionRenderable connection in this.connectionRenderables)
            {
                isAnyChildRendered |= connection.Draw(g);
            }

            // Render the rest of the children.
            foreach (RenderableBase child in this.Renderables)
            {
                if (child is ConnectionRenderable)
                {
                    continue;
                }

                isAnyChildRendered |= child.Draw(g);
            }

            return isAnyChildRendered;
        }

        /// <summary>
        /// Update the appearance of the item.
        /// </summary>
        private void UpdateAppearance()
        {
            if (this.fieldGroupRenderable != null)
            {
                if (this.IsSelected == true)
                {
                    this.fieldGroupRenderable.SeparatorLineColor = Color.White;
                    this.fieldGroupRenderable.BorderColor = Color.Black;
                    this.fieldGroupRenderable.FillColor = Color.DarkSeaGreen;
                }
                else if (this.IsMouseOver == true)
                {
                    this.fieldGroupRenderable.SeparatorLineColor = Color.Silver;
                    this.fieldGroupRenderable.BorderColor = Color.Black;
                    this.fieldGroupRenderable.FillColor = Color.Beige;
                }
                else
                {
                    this.fieldGroupRenderable.SeparatorLineColor = Color.Silver;
                    this.fieldGroupRenderable.BorderColor = Color.CadetBlue;
                    this.fieldGroupRenderable.FillColor = Color.Beige;
                }
            }
        }
    }
}
