﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;

using EffectMaker.DataModelMaker.Core.Collections;
using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;
using EffectMaker.DataModelMaker.Core.Template;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelMaker.Core.Writers
{
    /// <summary>
    /// エディタデータモデルのライターです.
    /// </summary>
    public class EditorDataModelWriter : SourceCodeWriterBase
    {
        /// <summary>The data context that holds all the data needed for the writer.</summary>
        private WriterContext context = null;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public EditorDataModelWriter()
        {
            /* DO_NOTHING */
        }

        /// <summary>
        /// 定義からソースコードとなる文字列を生成します.
        /// </summary>
        /// <param name="def">エディタデータモデルの定義です.</param>
        /// <returns>生成されたソースコードを返却します.</returns>
        public string Write(EditorDataModelDefinition def)
        {
            this.context = new WriterContext(def);

            // 改行コード.
            string endline = Environment.NewLine;

            // Check if the data model enables versioning.
            string mainTemplate;
            if (def.IsVersionEnabled == false ||
                def.CreatedVersionIndex == def.EditingVersionIndex)
            {
                mainTemplate = Properties.Resources.EDM_Main_NotVersioned;
            }
            else
            {
                mainTemplate = Properties.Resources.EDM_Main;
            }

            // Use the selected template for the data model source file.
            var classWriter = new SourceCodeWriter<EditorDataModelDefinition>(mainTemplate);
            int length = classWriter.EvaluateTemplateTokens(def);

            // Process using name spaces.
            {
                var namespaceList = new List<EditorDataModelUsingNamespaceDefinition>();

                // Add distinct name spaces.
                namespaceList.AddRange(
                    def.UsingNamespaces.GroupBy(ns => ns.Name).Select(gp => gp.First()));

                // 必須定義がなければ追加する.
                this.CheckAndAddNamespaceDefinition(namespaceList, "System");
                this.CheckAndAddNamespaceDefinition(namespaceList, "System.Linq");
                this.CheckAndAddNamespaceDefinition(namespaceList, "System.Xml.Serialization");
                this.CheckAndAddNamespaceDefinition(namespaceList, "EffectMaker.DataModel.Attributes");
                this.CheckAndAddNamespaceDefinition(namespaceList, "EffectMaker.DataModel.DataModels");
                this.CheckAndAddNamespaceDefinition(namespaceList, "EffectMaker.DataModel.Extensions");
                this.CheckAndAddNamespaceDefinition(namespaceList, "EffectMaker.DataModel.Specific.DataModels");
                this.CheckAndAddNamespaceDefinition(namespaceList, "EffectMaker.Foundation.Serialization");

                // 特殊型の判別.
                bool comboBoxItemPropertyFound = def.Properties.Any(
                    p => p.Type == "ComboBoxItemType" || p.ElementType == "ComboBoxItemType");

                if (comboBoxItemPropertyFound == true)
                {
                    this.CheckAndAddNamespaceDefinition(namespaceList, "ComboBoxItemType = System.Collections.Generic.KeyValuePair<string, object>");
                    this.CheckAndAddNamespaceDefinition(namespaceList, "System.Collections.Generic");
                }

                // 名前空間の定義を評価します.
                foreach (var namespaceDef in namespaceList.OrderBy(ns => ns.Name))
                {
                    var writer = new SourceCodeWriter<EditorDataModelUsingNamespaceDefinition>(Properties.Resources.EDM_UsingNamespace);
                    length += writer.EvaluateTemplateTokens(namespaceDef);
                    this.context.UsingNamespaceWriters.Add(writer);
                }
            }

            // 継承クラスの定義を評価します.
            if (def.SuperClasses.Count > 0)
            {
                foreach (Guid baseClassGuid in def.SuperClasses)
                {
                    var baseDef = WorkspaceManager.FindDefinition(baseClassGuid) as EditorDataModelDefinition;
                    if (baseDef != null)
                    {
                        var parentDef = new EditorDataModelSuperClassDefinition()
                        {
                            Name = (this.context.SuperClassWriters.Count > 0 ? ", " : string.Empty) + baseDef.Name,
                        };

                        var writer = new SourceCodeWriter<EditorDataModelSuperClassDefinition>(Properties.Resources.EDM_Inherit);
                        length += writer.EvaluateTemplateTokens(parentDef);
                        this.context.SuperClassWriters.Add(writer);
                    }
                }
            }
            else
            {
                EditorDataModelSuperClassDefinition parentDef = new EditorDataModelSuperClassDefinition() { Name = "DataModelBase" };
                var writer = new SourceCodeWriter<EditorDataModelSuperClassDefinition>(Properties.Resources.EDM_Inherit);
                length += writer.EvaluateTemplateTokens(parentDef);
                this.context.SuperClassWriters.Add(writer);
            }

            // 各プロパティごとに処理.
            foreach (EditorDataModelPropertyDefinition property in def.Properties)
            {
                length += this.WriteProperty(property);
            }

            // ストリングビルダーを生成します.
            StringBuilder builder = new StringBuilder(length);

            // ビルダーに文字列を追加していく.
            for (int i = 0; i < classWriter.EvaluatedTokens.Length; ++i)
            {
                string token = classWriter.EvaluatedTokens[i];
                if (token != null)
                {
                    builder.Append(token);
                }
                else
                {
                    TemplateToken templateToken = classWriter.GetToken(i);
                    if ((templateToken != null) &&
                        (string.IsNullOrEmpty(templateToken.Name) == false) &&
                        (this.context.WriterListMap.ContainsKey(templateToken.Name) == true))
                    {
                        this.WriteEntryList(builder, this.context.WriterListMap[templateToken.Name]);
                    }
                }
            }

            // 生成した文字列を返却します.
            return builder.ToString();
        }

        /// <summary>
        /// Write data model property source code.
        /// </summary>
        /// <param name="property">The property definition.</param>
        /// <returns>The length of the evaluated source code.</returns>
        private int WriteProperty(EditorDataModelPropertyDefinition property)
        {
            string endline = Environment.NewLine;
            int length = 0;

            bool isVersionEnabled =
                this.context.DataModelDefinition.IsVersionEnabled == true &&
                this.context.DataModelDefinition.CreatedVersionIndex < this.context.DataModelDefinition.EditingVersionIndex;

            EditorTypeInfo primaryType =
                TypeManager.FindEditorType(property.Namespace, property.Type);

            EditorTypeInfo elementType =
                TypeManager.FindEditorType(property.ElementNamespace, property.ElementType);

            if (primaryType == null)
            {
                return length;
            }

            // メンバの宣言定義を評価します.
            if (property.HasCustomProperty == false ||
                property.EnableFieldDeclaration == true)
            {
                // 元データを退避.
                string origDesc = property.Description;

                string[] delimiter = { endline, "\n" };

                string[] commentLines = property.Description.Split(delimiter, System.StringSplitOptions.None);

                // インデントのため空白を入れる(コメント).
                for (int idx = 0; idx < commentLines.Length; ++idx)
                {
                    string end = (idx < commentLines.Length - 1 ? endline : string.Empty);
                    commentLines[idx] = "        /// " + commentLines[idx] + end;
                }

                // 差し替え.
                property.Description = string.Concat(commentLines);

                SourceCodeWriter<EditorDataModelPropertyDefinition> writer =
                    this.CreateDeclMemberWriter(primaryType);
                if (writer != null)
                {
                    length += writer.EvaluateTemplateTokens(property);
                    this.context.InitMemberWriters.Add(writer);
                }

                // 元に戻す.
                property.Description = origDesc;
            }

            // メンバーの生成定義を評価します.
            if (property.HasCustomProperty == false)
            {
                // 元データを退避.
                string origDesc = property.Description;

                string[] delimiter = { endline, "\n" };

                string[] commentLines = property.Description.Split(delimiter, System.StringSplitOptions.None);

                // インデントのため空白を入れる(コメント).
                for (int idx = 0; idx < commentLines.Length; ++idx)
                {
                    string end = (idx < commentLines.Length - 1 ? endline : string.Empty);
                    commentLines[idx] = "        /// " + commentLines[idx] + end;
                }

                // 差し替え.
                property.Description = string.Concat(commentLines);

                var writer = this.CreateInitMemberWriter(primaryType, property.DefaultValue);
                if (writer != null)
                {
                    length += writer.EvaluateTemplateTokens(property);
                    this.context.ConstructMemberWriters.Add(writer);
                }

                // 元に戻す.
                property.Description = origDesc;
            }

            // プロパティの定義を評価します.
            if (property.HasCustomProperty == true)
            {
                // 元データを退避.
                string origType = property.Type;
                string origSetter = property.CustomSetter;
                string origGetter = property.CustomGetter;
                string origDesc = property.Description;

                string[] delimiter = { endline, "\n" };

                if (property.CustomGetter == null)
                {
                    property.CustomGetter = string.Empty;
                }

                if (property.CustomSetter == null)
                {
                    property.CustomSetter = string.Empty;
                }

                if (property.Description == null)
                {
                    property.Description = string.Empty;
                }

                // 改行コードで分割.
                string[] setterLines = property.CustomSetter.Split(delimiter, System.StringSplitOptions.None);
                string[] getterLines = property.CustomGetter.Split(delimiter, System.StringSplitOptions.None);
                string[] commentLines = property.Description.Split(delimiter, System.StringSplitOptions.None);

                if (primaryType.IsGeneric == true)
                {
                    if (elementType != null)
                    {
                        property.Type += "<" + elementType.TypeName + ">";
                    }
                }

                // インデントのため空白を入れる(Setter).
                for (int idx = 0; idx < setterLines.Length; ++idx)
                {
                    // カスタムコードで間に空行がある場合はインデントを入れずに改行のみにする
                    if (string.IsNullOrWhiteSpace(setterLines[idx]))
                    {
                        setterLines[idx] = endline;
                    }
                    else
                    {
                        setterLines[idx] = "                " + setterLines[idx] + endline;
                    }
                }

                // インデントのため空白を入れる(Getter).
                for (int idx = 0; idx < getterLines.Length; ++idx)
                {
                    // カスタムコードで間に空行がある場合はインデントを入れずに改行のみにする
                    if (string.IsNullOrWhiteSpace(getterLines[idx]))
                    {
                        getterLines[idx] = endline;
                    }
                    else
                    {
                        getterLines[idx] = "                " + getterLines[idx] + endline;
                    }
                }

                // インデントのため空白を入れる(コメント).
                for (int idx = 0; idx < commentLines.Length; ++idx)
                {
                    commentLines[idx] = "        /// " + commentLines[idx] + endline;
                }

                // 差し替え.
                property.CustomGetter = string.Concat(getterLines);
                property.CustomSetter = string.Concat(setterLines);
                property.Description = string.Concat(commentLines);

                if (property.CustomGetterEnabled && property.CustomSetterEnabled)
                {
                    //// Getter/Settter両方設定.

                    var writer = new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_Custom_GetterSetter);
                    if (writer != null)
                    {
                        length += writer.EvaluateTemplateTokens(property);
                        this.context.PropertyWriters.Add(writer);
                    }
                }
                else if (property.CustomSetterEnabled)
                {
                    //// Settterのみ設定.
                    var writer = new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_Custom_Setter);
                    if (writer != null)
                    {
                        string origCustomSetter = property.CustomSetter;

                        length += writer.EvaluateTemplateTokens(property);
                        this.context.PropertyWriters.Add(writer);
                    }
                }
                else
                {
                    //// Getterのみ設定.
                    var writer = new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_Custom_Getter);
                    if (writer != null)
                    {
                        string origCustomGetter = property.CustomGetter;

                        length += writer.EvaluateTemplateTokens(property);
                        this.context.PropertyWriters.Add(writer);
                    }
                }

                // 元に戻す.
                property.Description = origDesc;
                property.CustomSetter = origSetter;
                property.CustomGetter = origGetter;
                property.Type = origType;
            }
            else
            {
                // 元データを退避しておく.
                string origType = property.Type;
                string origDecription = property.Description;

                string[] delimiter = { endline, "\n" };

                // タブ文字置き換え.
                property.Description = property.Description.Replace("\t", "    ");

                // 改行コードで分割.
                string[] commentLines = property.Description.Split(delimiter, System.StringSplitOptions.None);

                // インデントのために空白を入れる.
                for (int idx = 0; idx < commentLines.Length; ++idx)
                {
                    commentLines[idx] = "        /// " + commentLines[idx] + endline;
                }

                // 連結.
                property.Description = string.Concat(commentLines);

                // ジェネリック用のタイプを設定.
                if (primaryType.IsGeneric == true &&
                    primaryType.TypeName == "IEnumerable")
                {
                    if (elementType != null)
                    {
                        property.Type = primaryType.TypeName + "<" + elementType.TypeName + ">";
                    }
                }

                bool isNull = false;
                if (property.DefaultValue == "null")
                {
                    isNull = true;
                }

                var writer = this.CreatePropertyWriter(primaryType, elementType, isNull);
                length += writer.EvaluateTemplateTokens(property);
                this.context.PropertyWriters.Add(writer);

                property.Type = origType;
                property.Description = origDecription;
            }

            // メンバーコピーの定義を評価します.
            bool hasPropertySetter =
                (property.HasCustomProperty == false) ||
                ((property.HasCustomProperty == true) && (property.CustomSetterEnabled == true));

            bool hasPropertyGetter =
                (property.HasCustomProperty == false) ||
                ((property.HasCustomProperty == true) && (property.CustomGetterEnabled == true));

            bool hasVersionConverter =
                (property.HasEditingVersion == true ||
                 property.AlwaysApplyVersionConverter) &&
                string.IsNullOrEmpty(property.VersionConverter) == false;

            // If there is no version converter, default property converter will be disabled
            // when the editing version is the created version.
            if (hasVersionConverter == false &&
                property.CreatedVersion.VersionIndex >= this.context.DataModelDefinition.EditingVersionIndex)
            {
                isVersionEnabled = false;
            }

            if (hasPropertySetter == true)
            {
                var writer = this.CreateCopyMemberWriter(primaryType, elementType);
                length += writer.EvaluateTemplateTokens(property);
                this.context.CopyMemberWriters.Add(writer);

                if (hasPropertyGetter == true && property.XmlIgnore == false)
                {
                    writer = this.CreateWriteMemberXmlWriter(primaryType, elementType);
                    length += writer.EvaluateTemplateTokens(property);
                    this.context.WriteMemberXmlWriters.Add(writer);

                    writer = this.CreateReadMemberXmlWriter(primaryType, elementType, property.EnableNullOriginal);
                    length += writer.EvaluateTemplateTokens(property);
                    this.context.ReadMemberXmlWriters.Add(writer);
                }

                if (isVersionEnabled == true)
                {
                    string versionConverter = property.VersionConverter;

                    // Indent the source code.
                    property.VersionConverter =
                        RegexUtility.IndentString(versionConverter, endline, 12);

                    writer = this.CreateConvertPrevVersionMemberWriter(property);
                    length += writer.EvaluateTemplateTokens(property);
                    this.context.ConvertPrevVersionMemberWriters.Add(writer);

                    if (hasVersionConverter == true)
                    {
                        writer = this.CreateVersionConverterWriter(property);
                        length += writer.EvaluateTemplateTokens(property);
                        this.context.VersionConverterWriters.Add(writer);
                    }

                    // Revert the indented source code.
                    property.VersionConverter = versionConverter;
                }
            }

            return length;
        }

        #region Private Methods

        /// <summary>
        /// メンバ宣言定義のライターを生成します.
        /// </summary>
        /// <param name="typeInfo">型情報.</param>
        /// <returns>ソースコードライターを返却します.</returns>
        private SourceCodeWriter<EditorDataModelPropertyDefinition> CreateDeclMemberWriter(EditorTypeInfo typeInfo)
        {
            // プリミティブは自動プロパティを利用するので，プリミティブでないことをチェック.
            if (typeInfo.IsICloneable || typeInfo.IsISettable || typeInfo.IsGeneric)
            {
                if (typeInfo.IsCollection)
                {
                    return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_DeclMember_IList);
                }
                else
                {
                    // メンバ宣言のテンプレートを設定.
                    return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_DeclMember_NP);
                }
            }

            return null;
        }

        /// <summary>
        /// メンバ初期化定義のライターを生成します.
        /// </summary>
        /// <param name="typeInfo">型情報.</param>
        /// <param name="defaultValue">デフォルト値</param>
        /// <returns>ソースコードライターを返却します.</returns>
        private SourceCodeWriter<EditorDataModelPropertyDefinition> CreateInitMemberWriter(EditorTypeInfo typeInfo, string defaultValue)
        {
            // プリミティブ
            if (typeInfo.IsPrimitive)
            {
                // 引数有り.
                if (!string.IsNullOrEmpty(defaultValue))
                {
                    return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_InitMember_P);
                }
            }
            else
            {
                // GUID
                if (typeInfo.TypeName == "Guid")
                {
                    // 引数無し.
                    if (string.IsNullOrEmpty(defaultValue))
                    {
                        return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_InitMember_Guid);
                    }

                    // 引数有り.
                    return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_InitMember_P);
                }
                else if (typeInfo.IsGeneric && typeInfo.IsCollection)
                {
                    // IList, List<T>の場合.
                    if (typeInfo.TypeName == "IList" || typeInfo.TypeName == "List")
                    {
                        // 引数有り.
                        if (!string.IsNullOrEmpty(defaultValue))
                        {
                            return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_InitMember_IList);
                        }
                    }
                }
                else if (!typeInfo.IsDataModel)
                {
                    // 引数有り.
                    if (!string.IsNullOrEmpty(defaultValue))
                    {
                        return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_InitMember_NP);
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// メンバコピー定義のライターを生成します.
        /// </summary>
        /// <param name="typeInfo">型情報.</param>
        /// <param name="elementTypeInfo">要素の型情報.</param>
        /// <returns>ソースコードライターを返却します.</returns>
        private SourceCodeWriter<EditorDataModelPropertyDefinition> CreateCopyMemberWriter(EditorTypeInfo typeInfo, EditorTypeInfo elementTypeInfo)
        {
            return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_CopyMember_P);
        }

        /// <summary>
        /// メンバコピー定義のライターを生成します.
        /// </summary>
        /// <param name="typeInfo">型情報.</param>
        /// <param name="elementTypeInfo">要素の型情報.</param>
        /// <returns>ソースコードライターを返却します.</returns>
        private SourceCodeWriter<EditorDataModelPropertyDefinition> CreateWriteMemberXmlWriter(EditorTypeInfo typeInfo, EditorTypeInfo elementTypeInfo)
        {
            if (typeInfo.TypeName != "ArrayCollection" &&
                typeInfo.IsCollection == true)
            {
                return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_WriteMemberXml_Collection);
            }
            else
            {
                return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_WriteMemberXml_Regular);
            }
        }

        /// <summary>
        /// メンバコピー定義のライターを生成します.
        /// </summary>
        /// <param name="typeInfo">型情報.</param>
        /// <param name="elementTypeInfo">要素の型情報.</param>
        /// <param name="isEnableNullOriginal">読み込む際に規定値をnullとして扱うか否か.</param>
        /// <returns>ソースコードライターを返却します.</returns>
        private SourceCodeWriter<EditorDataModelPropertyDefinition> CreateReadMemberXmlWriter(
            EditorTypeInfo typeInfo, EditorTypeInfo elementTypeInfo, bool isEnableNullOriginal)
        {
            if (typeInfo.TypeName == "List" && typeInfo.Namespace == "System.Collections.Generic")
            {
                return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_ReadMemberXml_List);
            }
            else if (isEnableNullOriginal)
            {
                return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_ReadMemberXml_NullOriginal);
            }
            else
            {
                return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_ReadMemberXml_Regular);
            }
        }

        /// <summary>
        /// プロパティバージョンコピーのライターを生成します.
        /// </summary>
        /// <param name="property">プロパティ定義.</param>
        /// <returns>ソースコードライターを返却します.</returns>
        private SourceCodeWriter<EditorDataModelPropertyDefinition> CreateConvertPrevVersionMemberWriter(EditorDataModelPropertyDefinition property)
        {
            if ((property.HasEditingVersion == true ||
                 property.AlwaysApplyVersionConverter == true) &&
                string.IsNullOrEmpty(property.VersionConverter) == false)
            {
                return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_ConvertPrevVersionMember);
            }
            else
            {
                if (property.IsDataModelCollection == true)
                {
                    // TODO:UserPageDataだけをハードコードで回避、本来はバージョニングされない全データで回避すべき
                    if (property.ElementType.Contains("UserPageData"))
                    {
                        return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_ConvertNonVersionList);
                    }
                    else
                    {
                        return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_ConvertPrevVersionList);
                    }
                }
                else
                {
                    return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_CopyMember_P);
                }
            }
        }

        /// <summary>
        /// プロパティバージョンコンバータのライターを生成します.
        /// </summary>
        /// <param name="property">プロパティ定義.</param>
        /// <returns>ソースコードライターを返却します.</returns>
        private SourceCodeWriter<EditorDataModelPropertyDefinition> CreateVersionConverterWriter(EditorDataModelPropertyDefinition property)
        {
            return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_VersionConverter);
        }

        /// <summary>
        /// プロパティ定義のライターを生成します.
        /// </summary>
        /// <param name="typeInfo">型情報.</param>
        /// <param name="elementTypeInfo">要素の型情報.</param>
        /// <param name="isNull">デフォルト値がnullかどうかを示すフラグ.</param>
        /// <returns>ソースコードライターを返却します.</returns>
        private SourceCodeWriter<EditorDataModelPropertyDefinition> CreatePropertyWriter(EditorTypeInfo typeInfo, EditorTypeInfo elementTypeInfo, bool isNull)
        {
            // ジェネリック系.
            if (typeInfo.IsGeneric)
            {
                if (typeInfo.TypeName == "List" || typeInfo.TypeName == "IList")
                {
                    if (elementTypeInfo.IsDataModel)
                    {
                        // 要素TがList<U>系.
                        return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_IList_NP_DM);
                    }
                    else if (elementTypeInfo.IsICloneable)
                    {
                        // 要素TがICloneable系.
                        return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_IList_NP_C);
                    }

                    // 要素Tがプリミティブ.
                    return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_IList_P);
                }
                else if (typeInfo.TypeName == "IEnumerable" || typeInfo.TypeName == "Enumerable")
                {
                    return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_IEnumerable);
                }
                else if (typeInfo.TypeName == "ArrayCollection")
                {
                    return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_ArrayCollection);
                }

                // プリミティブ.
                return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_P);
            }
            else
            {
                if (typeInfo.IsDataModel)
                {
                    // nullの場合は引数をtrueとする.
                    if (isNull)
                    {
                        return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_NP_DM_T);
                    }

                    // 非nullの場合は引数をfalseとする
                    return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_NP_DM_F);
                }
                else if (typeInfo.IsISettable)
                {
                    // nullの場合は引数をtrueとする.
                    if (isNull)
                    {
                        return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_NP_S_T);
                    }

                    // 非nullの場合は引数をfalseとする
                    return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_NP_S_F);
                }

                // プリミティブの場合.
                return new SourceCodeWriter<EditorDataModelPropertyDefinition>(Properties.Resources.EDM_Property_P);
            }
        }

        /// <summary>
        /// 名前空間の定義をチェックし，無ければ定義を追加します.
        /// </summary>
        /// <param name="list">出力リスト</param>
        /// <param name="name">検索名.</param>
        private void CheckAndAddNamespaceDefinition(
            List<EditorDataModelUsingNamespaceDefinition> list,
            string name)
        {
            if (list.Any(ns => ns.Name == name) == false)
            {
                // 検索にヒットしなかったらリストに追加する.
                var itemDef = new EditorDataModelUsingNamespaceDefinition() { Name = name };
                list.Add(itemDef);
            }
        }

        #endregion

        /// <summary>
        /// Data context holds data needed for this writer.
        /// </summary>
        private class WriterContext
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="def">The data model definition to write.</param>
            public WriterContext(EditorDataModelDefinition def)
            {
                this.DataModelDefinition = def;

                this.WriterListMap = new Dictionary<string, List<SourceCodeWriterBase>>();

                this.SuperClassWriters = new List<SourceCodeWriterBase>();
                this.UsingNamespaceWriters = new List<SourceCodeWriterBase>();
                this.InitMemberWriters = new List<SourceCodeWriterBase>();
                this.ConstructMemberWriters = new List<SourceCodeWriterBase>();
                this.PropertyWriters = new List<SourceCodeWriterBase>();
                this.CopyMemberWriters = new List<SourceCodeWriterBase>();
                this.WriteMemberXmlWriters = new List<SourceCodeWriterBase>();
                this.ReadMemberXmlWriters = new List<SourceCodeWriterBase>();
                this.ConvertPrevVersionMemberWriters = new List<SourceCodeWriterBase>();
                this.VersionConverterWriters = new List<SourceCodeWriterBase>();

                this.WriterListMap.Add("SuperClass", this.SuperClassWriters);
                this.WriterListMap.Add("UsingNamespace", this.UsingNamespaceWriters);
                this.WriterListMap.Add("InitMember", this.InitMemberWriters);
                this.WriterListMap.Add("ConstructMember", this.ConstructMemberWriters);
                this.WriterListMap.Add("Property", this.PropertyWriters);
                this.WriterListMap.Add("CopyMember", this.CopyMemberWriters);
                this.WriterListMap.Add("WriteMemberXml", this.WriteMemberXmlWriters);
                this.WriterListMap.Add("ReadMemberXml", this.ReadMemberXmlWriters);
                this.WriterListMap.Add("ConvertPrevVersionMember", this.ConvertPrevVersionMemberWriters);
                this.WriterListMap.Add("VersionConverters", this.VersionConverterWriters);
            }

            /// <summary>Get the data model definition.</summary>
            public EditorDataModelDefinition DataModelDefinition { get; private set; }

            /// <summary>Get the dictionary of the writer lists.</summary>
            public Dictionary<string, List<SourceCodeWriterBase>> WriterListMap { get; private set; }

            /// <summary>Get the writers for the super classes.</summary>
            public List<SourceCodeWriterBase> SuperClassWriters { get; private set; }

            /// <summary>Get the writers for the using name spaces.</summary>
            public List<SourceCodeWriterBase> UsingNamespaceWriters { get; private set; }

            /// <summary>Get the writers for member field initialization.</summary>
            public List<SourceCodeWriterBase> InitMemberWriters { get; private set; }

            /// <summary>Get the writers for member field initialization in the constructor.</summary>
            public List<SourceCodeWriterBase> ConstructMemberWriters { get; private set; }

            /// <summary>Get the writers for the properties.</summary>
            public List<SourceCodeWriterBase> PropertyWriters { get; private set; }

            /// <summary>Get the writers for copying member values.</summary>
            public List<SourceCodeWriterBase> CopyMemberWriters { get; private set; }

            /// <summary>Get the writers for serialize member values to XML document.</summary>
            public List<SourceCodeWriterBase> WriteMemberXmlWriters { get; private set; }

            /// <summary>Get the writers for deserialize member values from XML document.</summary>
            public List<SourceCodeWriterBase> ReadMemberXmlWriters { get; private set; }

            /// <summary>Get the writers for converting each member from previous version.</summary>
            public List<SourceCodeWriterBase> ConvertPrevVersionMemberWriters { get; private set; }

            /// <summary>Get the writers for version converter methods.</summary>
            public List<SourceCodeWriterBase> VersionConverterWriters { get; private set; }
        }
    }
}
