﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Security;
using System.Text;
using EffectMaker.DataModelMaker.Core.Definitions;

namespace EffectMaker.DataModelMaker.Generators
{
    /// <summary>
    /// XPath テーブルを生成します。
    /// </summary>
    public class XPathTableGenerator
    {
        /// <summary>
        /// エディタのデータモデル定義
        /// </summary>
        private EditorDataModelRootDefinition root;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="definition">エディタのデータモデル定義</param>
        public XPathTableGenerator(EditorDataModelRootDefinition definition)
        {
            this.root = definition;
        }

        /// <summary>
        /// XPath テーブルを出力します。
        /// </summary>
        /// <param name="path">出力ファイルパス</param>
        /// <returns>出力に成功したらtrueを返します。</returns>
        public bool Generate(string path)
        {
            // データモデル毎のXPath
            const string EmitterDataPath = "//EmitterSetData/EmitterList/EmitterData";

            const string EmitterBasicSettingPath = EmitterDataPath + "/EmitterBasicSettingData";
            const string EmitterEmissionPath = EmitterDataPath + "/EmitterEmissionData";
            const string EmitterEmitterPath = EmitterDataPath + "/EmitterEmitterData";
            const string EmitterParticlePath = EmitterDataPath + "/EmitterParticleData";
            const string EmitterCombinerPath = EmitterDataPath + "/EmitterCombinerData";
            const string EmitterTextureFilePath = EmitterDataPath + "/Textures/Texture[012]";
            const string EmitterColorPath = EmitterDataPath + "/EmitterColorData";
            const string EmitterScalePath = EmitterDataPath + "/EmitterScaleData";
            const string EmitterRotationPath = EmitterDataPath + "/EmitterRotationData";

            // 出力するデータモデルと、そのデータモデルの親のXPath
            string[][] outputDefs = new string[][] {
                new[] { EmitterBasicSettingPath, "EmitterBasicBasicData"  },  // エミッタノード/基本設定タブ/基本設定グループデータです。
                new[] { EmitterBasicSettingPath, "EmitterBasicRenderData" },  // エミッタノード/基本設定タブ/描画設定グループデータです。
                new[] { EmitterBasicSettingPath, "EmitterBasicRandomData" },  // エミッタノード/乱数設定グループデータです。
                new[] { EmitterBasicSettingPath, "EmitterBasicLodData"    },  // エミッタノード/基本設定タブ/LODグループデータです。
                // new[]  "EmitterBasicInheritanceData"
                new[] { EmitterEmissionPath, "EmitterEmissionTimingData"   },  // エミッタノード/放出タブ/放出タイミンググループデータです。
                new[] { EmitterEmissionPath, "EmitterEmissionVelocityData" },  // エミッタノード/放出タブ/放出時の位置と初速グループデータです。
                new[] { EmitterEmissionPath, "EmitterEmissionGravityData"  },  // エミッタノード/放出タブ/その他運動設定グループデータです。
                //
                new[] { EmitterEmitterPath, "EmitterEmitterShapeData"     },  // エミッタノード/エミッタタブ/エミッタ設定グループです。
                new[] { EmitterEmitterPath, "EmitterEmitterTransformData" },  // エミッタノード/エミッタタブ/エミッタトランスフォームグループデータです。
                new[] { EmitterEmitterPath, "EmitterEmitterColorData"     },  // エミッタノード/エミッタタブ/グローバル乗算カラー・アルファグループデータです。
                //
                new[] { EmitterParticlePath, "EmitterParticleLifeData"        },  // エミッタノード/パーティクルタブ/パーティクル寿命グループデータです。
                // new[] { EmitterParticlePath, "EmitterParticleRandomData"      },  // ★未使用：エミッタノード/パーティクルタブ/運動量ランダムグループです。
                new[] { EmitterParticlePath, "EmitterParticleShapeData"       },  // エミッタノード/パーティクルタブ/パーティクル設定グループです。
                new[] { EmitterParticlePath, "EmitterParticleFluctuationData" },  // エミッタノード/パーティクルタブ/揺らぎ設定グループデータです。
                //
                new[] { EmitterCombinerPath, "EmitterCombinerEquationData" },  // エミッタノード/コンバイナタブ/カラー・アルファコンバイナグループです。
                new[] { EmitterCombinerPath, "EmitterCombinerShaderData"   },  // エミッタノート/コンバイナタブ/パーティクルシェーダグループです。
                //
                new[] { EmitterTextureFilePath, "EmitterTextureFileData"             },  // エミッタノード/テクスチャタブ/テクスチャファイルグループデータです。
                new[] { EmitterTextureFilePath, "EmitterTextureUvAnimationData"      },  // エミッタノード/テクスチャタブ/テクスチャ座標アニメーションデータです。
                new[] { EmitterTextureFilePath, "EmitterTexturePatternAnimationData" },  // エミッタノード/テクスチャタブ/テクスチャパターンアニメーショングループデータです。
                //
                new[] { EmitterColorPath, "EmitterColorAlphaProcessingData" },  // エミッタノード/カラータブ/アルファ処理グループデータです。
                new[] { EmitterColorPath, "EmitterColorCalculationData"     },  // エミッタノード/カラータブ/カラー計算グループデータです。
                new[] { EmitterColorPath, "EmitterColorSettingsData"        },  // エミッタノード/カラータブ/カラー・アルファ設定グループデータです。
                new[] { EmitterScalePath, "EmitterScaleBasicData"               },  // エミッタノード/スケールタブ/基本スケールグループデータです。
                new[] { EmitterScalePath, "EmitterScaleAdditionalAnimationData" },  // エミッタノード/スケールタブ/スケールアニメーショングループデータです。
                new[] { EmitterRotationPath, "EmitterRotationBasicData" },  // エミッタノード/回転タブ/回転設定グループデータです。
                new[] { EmitterRotationPath, "EmitterRotationXData"     },  // エミッタノード/回転タブ/回転軸Xグループです。
                new[] { EmitterRotationPath, "EmitterRotationYData"     },  // エミッタノード/回転タブ/回転軸Yグループデータです。
                new[] { EmitterRotationPath, "EmitterRotationZData"     },  // エミッタノード/回転タブ/回転軸Zグループデータです。
            };

            StringBuilder xpathTableHtml = new StringBuilder();
            StringBuilder xpathTableXml = new StringBuilder();

            xpathTableHtml.Append(
                "<head>\r\n" +
                "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\">" +
                "</head>\r\n" +
                "<body>\r\n" +
                "<table border=1>\r\n"
            );

            xpathTableXml.Append(
                "<?xml version=\"1.0\"?>\r\n" +
                "<SearchTable xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\">\r\n" +
                "  <SearchItems>\r\n"
            );

            // データモデル毎の XPath を出力
            foreach (string[] outputDef in outputDefs)
            {
                EditorDataModelDefinition definition = this.root.DataModels.FirstOrDefault(x => x.Name == outputDef[1]);
                if (definition == null) continue;

                string description = System.Text.RegularExpressions.Regex.Replace(definition.Description, @"(データです。)|(です。)$", "");

                xpathTableHtml.Append("<tr bgcolor=\"#000000\"><td></td><td></td><td></td></tr>\r\n");

                xpathTableHtml.Append("<tr bgcolor=\"#ccffcc\">");
                xpathTableHtml.Append(string.Format("<td>{0}</td>", description));
                xpathTableHtml.Append("<td></td><td></td>");
                xpathTableHtml.Append("</tr>\r\n");

                foreach (EditorDataModelPropertyDefinition prop in definition.Properties)
                {
                    string propDesc = System.Text.RegularExpressions.Regex.Replace(prop.Description, @"(です。)|(します。)$", "");

                    string safeDesc   = SecurityElement.Escape(propDesc);
                    string safeUIPath = SecurityElement.Escape(description);
                    string safeXPath  = SecurityElement.Escape(outputDef[0] + "/" + outputDef[1] + "/" + prop.Name);
                    string safeType   = SecurityElement.Escape(prop.FullTypeWithoutNamespace);

                    // HTML
                    xpathTableHtml.Append("<tr>");
                    xpathTableHtml.Append(string.Format("<td>{0}</td>", safeDesc));
                    xpathTableHtml.Append(string.Format("<td>{0}</td>", safeXPath));
                    xpathTableHtml.Append(string.Format("<td>{0}</td>", safeType));
                    xpathTableHtml.Append("</tr>\r\n");

                    // XML
                    xpathTableXml.Append(string.Format("    <SearchItem>\r\n"));
                    xpathTableXml.Append(string.Format("      <UIPath>{0}</UIPath>\r\n", safeUIPath));
                    xpathTableXml.Append(string.Format("      <Title>{0}</Title>\r\n", safeDesc));
                    xpathTableXml.Append(string.Format("      <XPath>{0}</XPath>\r\n", safeXPath));
                    xpathTableXml.Append(string.Format("      <Type>{0}</Type>\r\n", safeType));
                    xpathTableXml.Append(string.Format("    </SearchItem>\r\n"));
                }
            }

            xpathTableHtml.Append(
                "</table>\r\n" +
                "</body>\r\n"
            );

            xpathTableXml.Append(
                "  </SearchItems>\r\n" +
                "</SearchTable>\r\n"
            );

            // HTML 出力
            File.WriteAllText(path + "\\XPathTable.html", xpathTableHtml.ToString());

            // XML 出力
            File.WriteAllText(path + "\\XPathTable.xml", xpathTableXml.ToString());

            return true;
        }
    }
}
