﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Reflection;
using System.Text.RegularExpressions;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.Core.DataTypes
{
    /// <summary>
    /// Class for managing types.
    /// </summary>
    public static class TypeManager
    {
        /// <summary>The map of the editor type names and the registered types.</summary>
        private static Dictionary<string, List<EditorTypeInfo>> editorTypeMap =
            new Dictionary<string, List<EditorTypeInfo>>();

        /// <summary>The list of all the registered editor namespace.</summary>
        private static Dictionary<string, List<EditorTypeInfo>> editorNamespaceMap =
            new Dictionary<string, List<EditorTypeInfo>>();

        /// <summary>The map of the runtime type names and the registered types.</summary>
        private static Dictionary<string, List<RuntimeTypeInfo>> runtimeTypeMap =
            new Dictionary<string, List<RuntimeTypeInfo>>();

        /// <summary>The list of all the registered runtime namespace.</summary>
        private static Dictionary<string, List<RuntimeTypeInfo>> runtimeNamespaceMap =
            new Dictionary<string, List<RuntimeTypeInfo>>();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static TypeManager()
        {
            AddDotNetSystemNamespaces();

            string emptyNS = string.Empty;

            // Editor-side types.
            AddEditorPrimitiveType(emptyNS, "bool");
            AddEditorPrimitiveType(emptyNS, "int");
            AddEditorPrimitiveType(emptyNS, "uint");
            AddEditorPrimitiveType(emptyNS, "sbyte");
            AddEditorPrimitiveType(emptyNS, "byte");
            AddEditorPrimitiveType(emptyNS, "short");
            AddEditorPrimitiveType(emptyNS, "ushort");
            AddEditorPrimitiveType(emptyNS, "long");
            AddEditorPrimitiveType(emptyNS, "ulong");
            AddEditorPrimitiveType(emptyNS, "float");
            AddEditorPrimitiveType(emptyNS, "double");
            AddEditorPrimitiveType(emptyNS, "char");
            AddEditorPrimitiveType(emptyNS, "string");

            // using ComboBoxItemType = KeyValuePair<string, object>
            // will be outputted to all the data model source files.
            AddEditorType(emptyNS, "ComboBoxItemType", false, false, false, false);

            AddEditorType("System", "Guid", false, false, false, false);
            AddEditorType("System.Collections.Generic", "List", true, true, false, false);
            AddEditorType("System.Collections.Generic", "IEnumerable", true, true, false, false);

            AddEditorType("EffectMaker.DataModel.AnimationTable", "AnimationTableData", false, false, true, true);
            AddEditorType("EffectMaker.DataModel.RandomColor", "RandomColorTable", false, false, true, true);

            AddEditorType("EffectMaker.Foundation.Collections.Generic", "ArrayCollection", true, true, true, true);
            AddEditorType("EffectMaker.Foundation.Collections", "StringPair", false, false, true, true);

            string primitiveNS = "EffectMaker.Foundation.Primitives";

            AddEffectMakerPrimitiveType(primitiveNS, "Vector1i", emptyNS, "int", 1);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector1u", emptyNS, "uint", 1);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector1f", emptyNS, "float", 1);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector1d", emptyNS, "double", 1);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector2i", emptyNS, "int", 2);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector2u", emptyNS, "uint", 2);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector2f", emptyNS, "float", 2);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector2d", emptyNS, "double", 2);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector3i", emptyNS, "int", 3);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector3u", emptyNS, "uint", 3);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector3f", emptyNS, "float", 3);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector3d", emptyNS, "double", 3);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector4i", emptyNS, "int", 4);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector4u", emptyNS, "uint", 4);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector4f", emptyNS, "float", 4);
            AddEffectMakerPrimitiveType(primitiveNS, "Vector4d", emptyNS, "double", 4);
            AddEffectMakerPrimitiveType(primitiveNS, "ColorRgba", emptyNS, "float", 4);
            AddEffectMakerPrimitiveType(primitiveNS, "ColorHsva", emptyNS, "float", 4);

            // Runtime-side types.
            AddRuntimeType(emptyNS, "short", 2);
            AddRuntimeType(emptyNS, "signed short", 2);
            AddRuntimeType(emptyNS, "unsigned short", 2);
            AddRuntimeType(emptyNS, "int", 4);
            AddRuntimeType(emptyNS, "signed int", 4);
            AddRuntimeType(emptyNS, "unsigned int", 4);
            AddRuntimeType(emptyNS, "long", 8);
            AddRuntimeType(emptyNS, "signed long", 8);
            AddRuntimeType(emptyNS, "unsigned long", 8);
            AddRuntimeType(emptyNS, "float", 4);
            AddRuntimeType(emptyNS, "double", 8);
            AddRuntimeType(emptyNS, "char", 1);
            AddRuntimeType(emptyNS, "unsigned char", 1);
            AddRuntimeType(emptyNS, "bool", 1);
            AddRuntimeType(emptyNS, "s8", 1);
            AddRuntimeType(emptyNS, "s16", 2);
            AddRuntimeType(emptyNS, "s32", 4);
            AddRuntimeType(emptyNS, "s64", 8);
            AddRuntimeType(emptyNS, "u8", 1);
            AddRuntimeType(emptyNS, "u16", 2);
            AddRuntimeType(emptyNS, "u32", 4);
            AddRuntimeType(emptyNS, "u64", 8);
            AddRuntimeType(emptyNS, "f32", 4);
            AddRuntimeType(emptyNS, "f64", 8);
            AddRuntimeType(emptyNS, "int8_t", 1);
            AddRuntimeType(emptyNS, "int16_t", 2);
            AddRuntimeType(emptyNS, "int32_t", 4);
            AddRuntimeType(emptyNS, "int64_t", 8);
            AddRuntimeType(emptyNS, "uint8_t", 1);
            AddRuntimeType(emptyNS, "uint16_t", 2);
            AddRuntimeType(emptyNS, "uint32_t", 4);
            AddRuntimeType(emptyNS, "uint64_t", 8);
            AddRuntimeType("nw::math", "MTX44", 64);
            AddRuntimeType("nw::math", "VEC2", 8);
            AddRuntimeType("nw::math", "VEC3", 12);
            AddRuntimeType("nw::math", "VEC4", 16);
            AddRuntimeType("nw::math", "VEC4*", 4);
            AddRuntimeType("nw::eftcom", "Guid", 16);
            AddRuntimeType("nn::util", "Float2", 8);
            AddRuntimeType("nn::util", "Float3", 12);
            AddRuntimeType("nn::util", "Float4", 16);
            AddRuntimeType("nn::util", "Float4*", 4);
        }

        /// <summary>
        /// Enumerate all the editor types.
        /// </summary>
        public static IEnumerable<EditorTypeInfo> EditorTypes
        {
            get { return editorTypeMap.SelectMany(item => item.Value); }
        }

        /// <summary>
        /// Enumerate all the editor namespaces.
        /// </summary>
        public static IEnumerable<string> EditorNamespaces
        {
            get { return editorNamespaceMap.Select(item => item.Key); }
        }

        /// <summary>
        /// Enumerate all the runtime types.
        /// </summary>
        public static IEnumerable<RuntimeTypeInfo> RuntimeTypes
        {
            get { return runtimeTypeMap.SelectMany(item => item.Value); }
        }

        /// <summary>
        /// Enumerate all the runtime namespaces.
        /// </summary>
        public static IEnumerable<string> RuntimeNamespaces
        {
            get { return runtimeNamespaceMap.Select(item => item.Key); }
        }

        /// <summary>
        /// Debug method for outputting registered types.
        /// </summary>
        public static void OutputRegisteredTypes()
        {
            // First get all the registered data models.
            IEnumerable<EditorTypeInfo> editorDataModels =
                editorTypeMap.SelectMany(pair => pair.Value).Where(item => item.IsDataModel == true);

            IEnumerable<RuntimeTypeInfo> runtimeDataModels =
                runtimeTypeMap.SelectMany(pair => pair.Value).Where(item => item.IsStructure == true);

            // Count the registered data models.
            int editorDataModelCount = editorDataModels.Count();
            int runtimeDataModelCount = runtimeDataModels.Count();
            if (editorDataModelCount <= 0 &&
                runtimeDataModelCount <= 0)
            {
                // No data model is still registered, bail out.
                return;
            }

            Logger.Log(LogLevels.Warning, "================================================================================");
            Logger.Log(LogLevels.Warning, "{0} data models are still registered.", editorDataModelCount + runtimeDataModelCount);
            Logger.Log(LogLevels.Warning, "================================================================================");

            foreach (EditorTypeInfo info in editorDataModels)
            {
                Logger.Log(LogLevels.Warning, "{0} {1}", info.FullName);
            }

            foreach (RuntimeTypeInfo info in runtimeDataModels)
            {
                Logger.Log(LogLevels.Warning, "{0} {1}", info.FullName);
            }

            Logger.Log(LogLevels.Warning, "================================================================================");
        }

        /// <summary>
        /// Add a new non-data model type.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type.</param>
        /// <param name="name">The name of the type.</param>
        /// <param name="isCollection">Flag indicating whether the type is a collection.</param>
        /// <param name="isGeneric">Flag indicating whether the type is generic.</param>
        /// <param name="isICloneable">Flag indicating whether the type is ICloneable.</param>
        /// <param name="isISettable">Flag indicating whether the type is ISettable.</param>
        public static void AddEditorType(
            string nameSpace,
            string name,
            bool isCollection,
            bool isGeneric,
            bool isICloneable,
            bool isISettable)
        {
            if (string.IsNullOrEmpty(name) == true)
            {
                Logger.Log(LogLevels.Warning, "RuntimeTypeManager.AddEditorType : Unable to add a new type without a name.");
                return;
            }

            if (nameSpace == null)
            {
                nameSpace = string.Empty;
            }

            // First get the list of types those have the specified type name.
            List<EditorTypeInfo> list;
            if (editorTypeMap.TryGetValue(name, out list) == false)
            {
                // The type name has not been registered yet.
                list = new List<EditorTypeInfo>();
                editorTypeMap.Add(name, list);
            }

            // Check if the type has already been added.
            if (list.Any(item => item.Namespace == nameSpace) == true)
            {
                ////Logger.Log(LogLevels.Warning, "EditorTypeManager.AddEditorType : The type {0}.{1} has already been added.", nameSpace, name);
                return;
            }

            // Add the type.
            var info = new EditorTypeInfo(nameSpace, name)
            {
                IsDataModel = false,
                IsEffectMakerPrimitive = false,
                IsPrimitive = false,
                IsCollection = isCollection,
                IsGeneric = isGeneric,
                IsICloneable = isICloneable,
                IsISettable = isISettable,
            };

            list.Add(info);

            // Add the type info to the namespace map.
            if (editorNamespaceMap.TryGetValue(nameSpace, out list) == false)
            {
                // The namespace has not been registered yet.
                list = new List<EditorTypeInfo>();
                editorNamespaceMap.Add(nameSpace, list);
            }

            list.Add(info);
        }

        /// <summary>
        /// Add a new primitive type.
        /// (E.g. int, float, bool, ...etc.)
        /// </summary>
        /// <param name="nameSpace">The namespace of the type.</param>
        /// <param name="name">The name of the type.</param>
        public static void AddEditorPrimitiveType(
            string nameSpace,
            string name)
        {
            if (string.IsNullOrEmpty(name) == true)
            {
                Logger.Log(LogLevels.Warning, "RuntimeTypeManager.AddEditorPrimitiveType : Unable to add a new type without a name.");
                return;
            }

            if (nameSpace == null)
            {
                nameSpace = string.Empty;
            }

            // First get the list of types those have the specified type name.
            List<EditorTypeInfo> list;
            if (editorTypeMap.TryGetValue(name, out list) == false)
            {
                // The type name has not been registered yet.
                list = new List<EditorTypeInfo>();
                editorTypeMap.Add(name, list);
            }

            // Check if the type has already been added.
            if (list.Any(item => item.Namespace == nameSpace) == true)
            {
                ////Logger.Log(LogLevels.Warning, "EditorTypeManager.AddEditorPrimitiveType : The type {0}.{1} has already been added.", nameSpace, name);
                return;
            }

            // Add the type.
            var info = new EditorTypeInfo(nameSpace, name)
            {
                IsDataModel = false,
                IsEffectMakerPrimitive = false,
                IsPrimitive = true,
                IsCollection = false,
                IsGeneric = false,
            };

            list.Add(info);

            // Add the type info to the namespace map.
            if (editorNamespaceMap.TryGetValue(nameSpace, out list) == false)
            {
                // The namespace has not been registered yet.
                list = new List<EditorTypeInfo>();
                editorNamespaceMap.Add(nameSpace, list);
            }

            list.Add(info);
        }

        /// <summary>
        /// Add a new primitive type.
        /// (E.g. Vector3f, Vector2i, ColorRgba, ...etc.)
        /// </summary>
        /// <param name="nameSpace">The namespace of the type.</param>
        /// <param name="name">The name of the type.</param>
        /// <param name="elementNamespace">The namespace of the underlying element type.</param>
        /// <param name="elementTypeName">The name of the underlying element type.</param>
        /// <param name="elementCount">The number of the underlying elements.</param>
        public static void AddEffectMakerPrimitiveType(
            string nameSpace,
            string name,
            string elementNamespace,
            string elementTypeName,
            int elementCount)
        {
            if (string.IsNullOrEmpty(name) == true)
            {
                Logger.Log(LogLevels.Warning, "RuntimeTypeManager.AddEffectMakerPrimitiveType : Unable to add a new type without a name.");
                return;
            }

            if (nameSpace == null)
            {
                nameSpace = string.Empty;
            }

            // First get the list of types those have the specified type name.
            List<EditorTypeInfo> list;
            if (editorTypeMap.TryGetValue(name, out list) == false)
            {
                // The type name has not been registered yet.
                list = new List<EditorTypeInfo>();
                editorTypeMap.Add(name, list);
            }

            // Check if the type has already been added.
            if (list.Any(item => item.Namespace == nameSpace) == true)
            {
                ////Logger.Log(LogLevels.Warning, "EditorTypeManager.AddEffectMakerPrimitiveType : The type {0}.{1} has already been added.", nameSpace, name);
                return;
            }

            EditorTypeInfo elementTypeInfo = FindEditorType(elementNamespace, elementTypeName);

            // Add the type.
            var info = new EditorTypeInfo(nameSpace, name)
            {
                IsDataModel = false,
                IsEffectMakerPrimitive = true,
                IsPrimitive = false,
                IsCollection = false,
                IsGeneric = false,
                IsICloneable = true,
                IsISettable = true,
                ElementCount = elementCount,
                ElementType = elementTypeInfo
            };

            list.Add(info);

            // Add the type info to the namespace map.
            if (editorNamespaceMap.TryGetValue(nameSpace, out list) == false)
            {
                // The namespace has not been registered yet.
                list = new List<EditorTypeInfo>();
                editorNamespaceMap.Add(nameSpace, list);
            }

            list.Add(info);
        }

        /// <summary>
        /// Add a new data model.
        /// </summary>
        /// <param name="def">The data model definition.</param>
        public static void AddEditorDataModel(EditorDataModelDefinition def)
        {
            if (def == null)
            {
                return;
            }

            string name = def.Name;
            string nameSpace = def.Namespace;

            if (string.IsNullOrEmpty(name) == true)
            {
                Logger.Log(LogLevels.Warning, "RuntimeTypeManager.AddEditorDataModel : Unable to add a new type without a name.");
                return;
            }

            if (nameSpace == null)
            {
                nameSpace = string.Empty;
            }

            // First get the list of types those have the specified type name.
            List<EditorTypeInfo> list;
            if (editorTypeMap.TryGetValue(name, out list) == false)
            {
                // The type name has not been registered yet.
                list = new List<EditorTypeInfo>();
                editorTypeMap.Add(name, list);
            }

            // Check if the type has already been added.
            if (list.Any(item => item.Namespace == nameSpace) == true)
            {
                ////Logger.Log(LogLevels.Warning, "EditorTypeManager.AddEditorDataModel : The type {0}.{1} has already been added.", nameSpace, name);
                return;
            }

            // Add the type.
            var info = new EditorTypeInfo(nameSpace, name)
            {
                IsDataModel = true,
                IsEffectMakerPrimitive = false,
                IsPrimitive = false,
                IsCollection = false,
                IsGeneric = false,
                IsICloneable = true,
                IsISettable = true,
                DataModelDefinition = def
            };

            list.Add(info);

            // Add the type info to the namespace map.
            if (editorNamespaceMap.TryGetValue(nameSpace, out list) == false)
            {
                // The namespace has not been registered yet.
                list = new List<EditorTypeInfo>();
                editorNamespaceMap.Add(nameSpace, list);
            }

            list.Add(info);
        }

        /// <summary>
        /// Remove a editor type.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type.</param>
        /// <param name="name">The name of the type.</param>
        public static void RemoveEditorType(string nameSpace, string name)
        {
            if (string.IsNullOrEmpty(name) == true)
            {
                return;
            }

            if (nameSpace == null)
            {
                nameSpace = string.Empty;
            }

            // First get the list of types those have the specified type name.
            List<EditorTypeInfo> list;
            if (editorTypeMap.TryGetValue(name, out list) == true)
            {
                // Remove all the matching type info.
                list.RemoveAll(item => item.Namespace == nameSpace);

                // If the list has nothing left, remove it from the map.
                if (list.Count <= 0)
                {
                    editorTypeMap.Remove(name);
                }
            }

            // Remove the type info from the namespace map.
            if (editorNamespaceMap.TryGetValue(nameSpace, out list) == true)
            {
                // Remove all the matching type info.
                list.RemoveAll(item => item.TypeName == name);
            }
        }

        /// <summary>
        /// Find the specified editor type info.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type.</param>
        /// <param name="name">The name of the type.</param>
        /// <returns>The type info or null if the type is not found.</returns>
        public static EditorTypeInfo FindEditorType(string nameSpace, string name)
        {
            if (string.IsNullOrEmpty(name) == true)
            {
                return null;
            }

            if (nameSpace == null)
            {
                nameSpace = string.Empty;
            }

            // First get the list of types those have the specified type name.
            List<EditorTypeInfo> list;
            if (editorTypeMap.TryGetValue(name, out list) == false)
            {
                return null;
            }

            return list.Find(item => item.Namespace == nameSpace);
        }

        /// <summary>
        /// Find editor types that their name starts with the given string.
        /// </summary>
        /// <param name="str">The string to find.</param>
        /// <returns>A enumerable with matching types.</returns>
        public static IEnumerable<EditorTypeInfo> FindEditorTypesNameStartWith(string str)
        {
            if (string.IsNullOrEmpty(str) == true)
            {
                return Enumerable.Empty<EditorTypeInfo>();
            }

            var cmp = StringComparison.InvariantCultureIgnoreCase;
            return editorTypeMap.Where(it => it.Key.StartsWith(str, cmp)).SelectMany(it => it.Value);
        }

        /// <summary>
        /// Parse the types from the full type name the user inputs.
        /// This method does check the registered editor type info,
        /// if the type is not found, false is returned.
        /// </summary>
        /// <param name="fullName">The full type name.</param>
        /// <param name="primaryTypeInfo">The primary type info.</param>
        /// <param name="elementTypeInfo">The element type info if the primary type is generic.</param>
        /// <returns>True on success.</returns>
        public static bool ParseEditorType(
            string fullName,
            out EditorTypeInfo primaryTypeInfo,
            out EditorTypeInfo elementTypeInfo)
        {
            primaryTypeInfo = null;
            elementTypeInfo = null;

            // Trim the space characters.
            fullName = fullName.Trim();
            if (string.IsNullOrEmpty(fullName) == true)
            {
                return false;
            }

            // Parse the primary type first.
            var expParsePrimaryType = new Regex("^([^<]+)");
            Match matchPrimaryType = expParsePrimaryType.Match(fullName);
            if (matchPrimaryType.Success == false)
            {
                return false;
            }

            string primaryTypeString = matchPrimaryType.Value;

            // Split the namespace and the type name, then search for the type info.
            int lastDotIndex = primaryTypeString.LastIndexOf('.');
            if (lastDotIndex < 0)
            {
                // The type name does not include namespace.
                primaryTypeInfo = FindEditorType(string.Empty, primaryTypeString);
            }
            else
            {
                primaryTypeInfo = FindEditorType(
                    primaryTypeString.Substring(0, lastDotIndex),
                    primaryTypeString.Substring(lastDotIndex + 1));
            }

            if (primaryTypeInfo == null)
            {
                return false;
            }

            // Parse the element type if the type is generic.
            if (primaryTypeInfo.IsGeneric == true)
            {
                var expParseElementType = new Regex("<([^>]+)>$");
                Match matchElementType = expParseElementType.Match(fullName);
                if (matchElementType.Success == true)
                {
                    string elementTypeString = matchElementType.Groups[1].Value;

                    lastDotIndex = elementTypeString.LastIndexOf('.');
                    if (lastDotIndex < 0)
                    {
                        // The type name does not include namespace.
                        elementTypeInfo = FindEditorType(string.Empty, elementTypeString);
                    }
                    else
                    {
                        elementTypeInfo = FindEditorType(
                            elementTypeString.Substring(0, lastDotIndex),
                            elementTypeString.Substring(lastDotIndex + 1));
                    }

                    if (elementTypeInfo == null)
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// Parse the types from the full type name the user inputs.
        /// This method does not check the registered editor type info,
        /// it merely process the string into namespace and type names.
        /// </summary>
        /// <param name="fullName">The full type name.</param>
        /// <param name="primaryNamespace">The primary type namespace.</param>
        /// <param name="primaryTypeName">The primary type name.</param>
        /// <param name="elementNamespace">The element namespace if the primary type is generic.</param>
        /// <param name="elementTypeName">The element type name if the primary type is generic.</param>
        /// <returns>True on success.</returns>
        public static bool ParseEditorType(
            string fullName,
            out string primaryNamespace,
            out string primaryTypeName,
            out string elementNamespace,
            out string elementTypeName)
        {
            primaryNamespace = string.Empty;
            primaryTypeName = string.Empty;
            elementNamespace = string.Empty;
            elementTypeName = string.Empty;

            // Trim the space characters.
            fullName = fullName.Trim();
            if (string.IsNullOrEmpty(fullName) == true)
            {
                return false;
            }

            // Parse the primary type first.
            var expParsePrimaryType = new Regex("^([^<]+)");
            Match matchPrimaryType = expParsePrimaryType.Match(fullName);
            if (matchPrimaryType.Success == false)
            {
                return false;
            }

            string primaryTypeString = matchPrimaryType.Value;

            // Split the namespace and the type name, then search for the type info.
            int lastDotIndex = primaryTypeString.LastIndexOf('.');
            if (lastDotIndex < 0)
            {
                // The type name does not include namespace.
                primaryNamespace = string.Empty;
                primaryTypeName = primaryTypeString;
            }
            else
            {
                primaryNamespace = primaryTypeString.Substring(0, lastDotIndex);
                primaryTypeName = primaryTypeString.Substring(lastDotIndex + 1);
            }

            // Parse the element type if the type is generic.
            var expParseElementType = new Regex("<([^>]+)>$");
            Match matchElementType = expParseElementType.Match(fullName);
            if (matchElementType.Success == true)
            {
                string elementTypeString = matchElementType.Groups[1].Value;

                lastDotIndex = elementTypeString.LastIndexOf('.');
                if (lastDotIndex < 0)
                {
                    // The type name does not include namespace.
                    elementNamespace = string.Empty;
                    elementTypeName = elementTypeString;
                }
                else
                {
                    elementNamespace = elementTypeString.Substring(0, lastDotIndex);
                    elementTypeName = elementTypeString.Substring(lastDotIndex + 1);
                }
            }

            return true;
        }

        /// <summary>
        /// Compose the full name of the editor type.
        /// The method would first look for registered type with the given namespace
        /// and type name, if found, use the full name of the type info, otherwise
        /// compose the full name with the given namespace and name.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type to search for.</param>
        /// <param name="typeName">The type name to search for.</param>
        /// <param name="composedName">The composed type name.</param>
        /// <returns>True if the editor type info is found.</returns>
        public static bool ComposeEditorTypeFullName(
            string nameSpace,
            string typeName,
            out string composedName)
        {
            EditorTypeInfo info = FindEditorType(nameSpace, typeName);
            if (info != null)
            {
                composedName = info.FullName;
                return true;
            }
            else
            {
                if (string.IsNullOrEmpty(nameSpace) == true)
                {
                    composedName = typeName;
                }
                else
                {
                    composedName = nameSpace + "." + typeName;
                }

                return false;
            }
        }

        /// <summary>
        /// Compose the name of the editor type.
        /// The method would first look for registered type with the given namespace
        /// and type name, if found, use the full name of the type info, otherwise
        /// compose the full name with the given type name.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type to search for.</param>
        /// <param name="typeName">The type name to search for.</param>
        /// <param name="composedName">The composed type name.</param>
        /// <returns>True if the editor type info is found.</returns>
        public static bool ComposeEditorTypeName(
            string nameSpace,
            string typeName,
            out string composedName)
        {
            EditorTypeInfo info = FindEditorType(nameSpace, typeName);
            if (info != null)
            {
                composedName = info.FullName;
                return true;
            }
            else
            {
                composedName = typeName;
                return false;
            }
        }

        /// <summary>
        /// Convert the type name to its short form that the namespace is not included.
        /// </summary>
        /// <param name="fullName">The full type name to convert.</param>
        /// <returns>The converted type name.</returns>
        public static string ToShortTypeName(string fullName)
        {
            string primaryNamespace, primaryType, elementNamespace, elementType;
            if (ParseEditorType(
                fullName,
                out primaryNamespace,
                out primaryType,
                out elementNamespace,
                out elementType) == false)
            {
                return fullName;
            }

            if (string.IsNullOrEmpty(elementType) == true)
            {
                return primaryType;
            }
            else
            {
                return string.Format("{0}<{1}>", primaryType, elementType);
            }
        }

        /// <summary>
        /// Add a new runtime type.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type.</param>
        /// <param name="name">The name of the type.</param>
        /// <param name="size">The size of the type.</param>
        public static void AddRuntimeType(
            string nameSpace,
            string name,
            int size)
        {
            if (string.IsNullOrEmpty(name) == true)
            {
                Logger.Log(LogLevels.Warning, "RuntimeTypeManager.AddRuntimeType : Unable to add a new type without a name.");
                return;
            }

            if (nameSpace == null)
            {
                nameSpace = string.Empty;
            }

            // First get the list of types those have the specified type name.
            List<RuntimeTypeInfo> list;
            if (runtimeTypeMap.TryGetValue(name, out list) == false)
            {
                // The type name has not been registered yet.
                list = new List<RuntimeTypeInfo>();
                runtimeTypeMap.Add(name, list);
            }

            // Check if the type has already been added.
            if (list.Any(item => item.Namespace == nameSpace) == true)
            {
                ////Logger.Log(LogLevels.Warning, "TypeManager.AddRuntimeType : The type {0}.{1} has already been added.", nameSpace, name);
                return;
            }

            // Add the type.
            var info = new RuntimeTypeInfo(nameSpace, name)
            {
                IsStructure = false,
                DataSize = size
            };

            list.Add(info);

            // Add the type info to the namespace map.
            if (runtimeNamespaceMap.TryGetValue(nameSpace, out list) == false)
            {
                // The namespace has not been registered yet.
                list = new List<RuntimeTypeInfo>();
                runtimeNamespaceMap.Add(nameSpace, list);
            }

            list.Add(info);
        }

        /// <summary>
        /// Add a new runtime data structure/class.
        /// </summary>
        /// <param name="def">The data structure definition.</param>
        public static void AddRuntimeDataModel(RuntimeDataModelDefinition def)
        {
            if (def == null)
            {
                return;
            }

            string name = def.Name;
            string nameSpace = def.Namespace;

            if (string.IsNullOrEmpty(name) == true)
            {
                Logger.Log(LogLevels.Warning, "RuntimeTypeManager.AddDataStructure : Unable to add a new type without a name.");
                return;
            }

            if (nameSpace == null)
            {
                nameSpace = string.Empty;
            }

            // First get the list of types those have the specified type name.
            List<RuntimeTypeInfo> list;
            if (runtimeTypeMap.TryGetValue(name, out list) == false)
            {
                // The type name has not been registered yet.
                list = new List<RuntimeTypeInfo>();
                runtimeTypeMap.Add(name, list);
            }

            // Check if the type has already been added.
            if (list.Any(item => item.Namespace == nameSpace) == true)
            {
                ////Logger.Log(LogLevels.Warning, "RuntimeTypeManager.AddDataStructure : The type {0}.{1} has already been added.", nameSpace, name);
                return;
            }

            // Add the type.
            var info = new RuntimeTypeInfo(nameSpace, name)
            {
                IsStructure = true,
                StructureDefinition = def
            };

            list.Add(info);

            // Add the type info to the namespace map.
            if (runtimeNamespaceMap.TryGetValue(nameSpace, out list) == false)
            {
                // The namespace has not been registered yet.
                list = new List<RuntimeTypeInfo>();
                runtimeNamespaceMap.Add(nameSpace, list);
            }

            list.Add(info);
        }

        /// <summary>
        /// Remove a runtime type.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type.</param>
        /// <param name="name">The name of the type.</param>
        public static void RemoveRuntimeType(string nameSpace, string name)
        {
            if (string.IsNullOrEmpty(name) == true)
            {
                return;
            }

            if (nameSpace == null)
            {
                nameSpace = string.Empty;
            }

            // First get the list of types those have the specified type name.
            List<RuntimeTypeInfo> list;
            if (runtimeTypeMap.TryGetValue(name, out list) == true)
            {
                // Remove all the matching type info.
                list.RemoveAll(item => item.Namespace == nameSpace);

                // If the list has nothing left, remove it from the map.
                if (list.Count <= 0)
                {
                    runtimeTypeMap.Remove(name);
                }
            }

            // Remove the type info from the namespace map.
            if (runtimeNamespaceMap.TryGetValue(nameSpace, out list) == true)
            {
                // Remove all the matching type info.
                list.RemoveAll(item => item.TypeName == name);
            }
        }

        /// <summary>
        /// Find the specified runtime type info.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type.</param>
        /// <param name="name">The name of the type.</param>
        /// <returns>The type info or null if the type is not found.</returns>
        public static RuntimeTypeInfo FindRuntimeType(string nameSpace, string name)
        {
            if (string.IsNullOrEmpty(name) == true)
            {
                return null;
            }

            if (nameSpace == null)
            {
                nameSpace = string.Empty;
            }

            // First get the list of types those have the specified type name.
            List<RuntimeTypeInfo> list;
            if (runtimeTypeMap.TryGetValue(name, out list) == false)
            {
                return null;
            }

            return list.Find(item => item.Namespace == nameSpace);
        }

        /// <summary>
        /// Find runtime types that their name starts with the given string.
        /// </summary>
        /// <param name="str">The string to find.</param>
        /// <returns>A enumerable with matching types.</returns>
        public static IEnumerable<RuntimeTypeInfo> FindRuntimeTypesNameStartWith(string str)
        {
            if (string.IsNullOrEmpty(str) == true)
            {
                return Enumerable.Empty<RuntimeTypeInfo>();
            }

            var cmp = StringComparison.InvariantCultureIgnoreCase;
            return runtimeTypeMap.Where(it => it.Key.StartsWith(str, cmp)).SelectMany(it => it.Value);
        }

        /// <summary>
        /// Parse the types from the full type name the user inputs.
        /// This method does check the registered editor type info,
        /// if the type is not found, false is returned.
        /// </summary>
        /// <param name="fullName">The full type name.</param>
        /// <param name="typeInfo">The type info.</param>
        /// <returns>True on success.</returns>
        public static bool ParseRuntimeType(
            string fullName,
            out RuntimeTypeInfo typeInfo)
        {
            typeInfo = null;

            // Trim the space characters.
            fullName = fullName.Trim();
            if (string.IsNullOrEmpty(fullName) == true)
            {
                return false;
            }

            // Split the namespace and the type name, then search for the type info.
            int lastDotIndex = fullName.LastIndexOf("::");
            if (lastDotIndex < 0)
            {
                // The type name does not include namespace.
                typeInfo = FindRuntimeType(string.Empty, fullName);
            }
            else
            {
                typeInfo = FindRuntimeType(
                    fullName.Substring(0, lastDotIndex),
                    fullName.Substring(lastDotIndex + 2));
            }

            if (typeInfo == null)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Parse the types from the full type name the user inputs.
        /// This method does not check the registered editor type info,
        /// it merely process the string into namespace and type names.
        /// </summary>
        /// <param name="fullName">The full type name.</param>
        /// <param name="typeNamespace">The type namespace.</param>
        /// <param name="typeName">The type name.</param>
        /// <returns>True on success.</returns>
        public static bool ParseRuntimeType(
            string fullName,
            out string typeNamespace,
            out string typeName)
        {
            typeNamespace = string.Empty;
            typeName = string.Empty;

            // Trim the space characters.
            fullName = fullName.Trim();
            if (string.IsNullOrEmpty(fullName) == true)
            {
                return false;
            }

            // Split the namespace and the type name, then search for the type info.
            int lastDotIndex = fullName.LastIndexOf("::");
            if (lastDotIndex < 0)
            {
                // The type name does not include namespace.
                typeNamespace = string.Empty;
                typeName = fullName;
            }
            else
            {
                typeNamespace = fullName.Substring(0, lastDotIndex);
                typeName = fullName.Substring(lastDotIndex + 2);
            }

            return true;
        }

        /// <summary>
        /// Compose the full name of the runtime type.
        /// The method would first look for registered type with the given namespace
        /// and type name, if found, use the full name of the type info, otherwise
        /// compose the full name with the given namespace and name.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type to search for.</param>
        /// <param name="typeName">The type name to search for.</param>
        /// <param name="composedName">The composed type name.</param>
        /// <returns>True if the runtime type info is found.</returns>
        public static bool ComposeRuntimeTypeFullName(
            string nameSpace,
            string typeName,
            out string composedName)
        {
            RuntimeTypeInfo info = FindRuntimeType(nameSpace, typeName);
            if (info != null)
            {
                composedName = info.FullName;
                return true;
            }
            else
            {
                if (string.IsNullOrEmpty(nameSpace) == true)
                {
                    composedName = typeName;
                }
                else
                {
                    composedName = nameSpace + "::" + typeName;
                }

                return false;
            }
        }

        /// <summary>
        /// Compose the name of the runtime type.
        /// The method would first look for registered type with the given namespace
        /// and type name, if found, use the full name of the type info, otherwise
        /// compose the full name with the given type name.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type to search for.</param>
        /// <param name="typeName">The type name to search for.</param>
        /// <param name="composedName">The composed type name.</param>
        /// <returns>True if the runtime type info is found.</returns>
        public static bool ComposeRuntimeTypeName(
            string nameSpace,
            string typeName,
            out string composedName)
        {
            RuntimeTypeInfo info = FindRuntimeType(nameSpace, typeName);
            if (info != null)
            {
                composedName = info.TypeName;
                return true;
            }
            else
            {
                composedName = typeName;
                return false;
            }
        }

        /// <summary>
        /// Add all the referenced .NET namespaces to the editor namespace map.
        /// </summary>
        private static void AddDotNetSystemNamespaces()
        {
            // Do not execute this method at design time. (in Visual Studio Designer)
            if (LicenseManager.UsageMode == LicenseUsageMode.Designtime ||
                CoreSettings.IsWindowMode == false)
            {
                return;
            }

            // Get the application assembly and it's name.
            Assembly entryAsm = Assembly.GetEntryAssembly();
            string appNamespace = entryAsm.GetName().Name;

            // We only want "EffectMaker.DataModelMaker", without the trailing ".Application".
            appNamespace = appNamespace.Remove(appNamespace.LastIndexOf('.'));

            foreach (AssemblyName refAsmName in entryAsm.GetReferencedAssemblies())
            {
                if (refAsmName.Name.StartsWith(appNamespace) == true)
                {
                    continue;
                }

                Assembly refAsm = null;
                try
                {
                    // Load the assembly that is referenced by the entry assembly.
                    // (by the application project)
                    refAsm = Assembly.Load(refAsmName);
                }
                catch
                {
                    Logger.Log(LogLevels.Debug, "Failed loading referenced assembly {0}.", refAsmName.Name);
                    continue;
                }

                if (refAsm == null)
                {
                    continue;
                }

                try
                {
                    // Enumerate all the namespaces in the assembly and add them to our map.
                    Type[] types = refAsm.GetTypes();
                    foreach (string nameSpace in types.Select(tp => tp.Namespace).Distinct())
                    {
                        if (string.IsNullOrEmpty(nameSpace) == false &&
                            editorNamespaceMap.ContainsKey(nameSpace) == false)
                        {
                            editorNamespaceMap.Add(nameSpace, new List<EditorTypeInfo>());
                        }
                    }
                }
                catch
                {
                    Logger.Log(LogLevels.Debug, "Failed enumerating namespaces in the referenced assembly {0}.", refAsmName.Name);
                    continue;
                }
            }
        }
    }
}
