﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

using EffectMaker.DataModelMaker.Core.Definitions;

namespace EffectMaker.DataModelMaker.Core.DataTypes
{
    /// <summary>
    /// Class for storing the information of a runtime-side data type.
    /// </summary>
    public class RuntimeTypeInfo
    {
        /// <summary>
        /// Data size for data types.
        /// The data size for data structures is computed dynamically.
        /// </summary>
        private int staticDataSize = 0;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type.</param>
        /// <param name="name">The name of the type.</param>
        public RuntimeTypeInfo(
            string nameSpace,
            string name)
        {
            this.TypeName = name;
            this.Namespace = nameSpace;
            this.IsStructure = false;
            this.StructureDefinition = null;
        }

        /// <summary>
        /// Get the name of the type.
        /// E.g. System::Collections::Generic::List => List
        /// </summary>
        public string TypeName { get; set; }

        /// <summary>
        /// Get the namespace of the type.
        /// E.g. System::Collections::Generic::List => System::Collections::Generic
        /// </summary>
        public string Namespace { get; set; }

        /// <summary>
        /// Get the full name of the type.
        /// E.g. System::Collections::Generic::List
        /// </summary>
        public string FullName
        {
            get
            {
                if (string.IsNullOrEmpty(this.Namespace) == true)
                {
                    return this.TypeName;
                }
                else
                {
                    return this.Namespace + "::" + this.TypeName;
                }
            }
        }

        /// <summary>
        /// Get or set the size of the type.
        /// </summary>
        public int DataSize
        {
            get
            {
                if (this.IsStructure == false)
                {
                    return this.staticDataSize;
                }

                return 0;
            }

            set
            {
                // Only allow setting the data size if the type is not a structure.
                if (this.IsStructure == false)
                {
                    this.staticDataSize = value;
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether the type is a data structure/class.
        /// </summary>
        public bool IsStructure { get; set; }

        /// <summary>
        /// The data model structure definition.
        /// </summary>
        public RuntimeDataModelDefinition StructureDefinition { get; set; }
    }
}
