﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

using EffectMaker.DataModelMaker.Core.Definitions;

namespace EffectMaker.DataModelMaker.Core.DataTypes
{
    /// <summary>
    /// Class for storing the information of a editor-side data type.
    /// </summary>
    public class EditorTypeInfo
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="nameSpace">The namespace of the type.</param>
        /// <param name="name">The name of the type.</param>
        public EditorTypeInfo(
            string nameSpace,
            string name)
        {
            this.TypeName = name;
            this.Namespace = nameSpace;
            this.IsDataModel = false;
            this.IsEffectMakerPrimitive = false;
            this.IsPrimitive = false;
            this.IsCollection = false;
            this.IsGeneric = false;
            this.IsICloneable = false;
            this.IsISettable = false;
            this.ElementCount = 1;
            this.ElementType = null;
            this.DataModelDefinition = null;
        }

        /// <summary>
        /// Get the name of the type.
        /// E.g. System.Collections.Generic.List => List
        /// </summary>
        public string TypeName { get; set; }

        /// <summary>
        /// Get the namespace of the type.
        /// E.g. System.Collections.Generic.List => System.Collections.Generic
        /// </summary>
        public string Namespace { get; set; }

        /// <summary>
        /// Get the full name of the type.
        /// E.g. System.Collections.Generic.List
        /// </summary>
        public string FullName
        {
            get
            {
                if (string.IsNullOrEmpty(this.Namespace) == true)
                {
                    return this.TypeName;
                }
                else
                {
                    return this.Namespace + "." + this.TypeName;
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether the type is a data model.
        /// </summary>
        public bool IsDataModel { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the type is a EffectMaker primitive.
        /// (Vector2f, Vector3i, ColorRgba, ...etc.)
        /// </summary>
        public bool IsEffectMakerPrimitive { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the type is a primitive.
        /// (int, float, double, bool, ...etc.)
        /// </summary>
        public bool IsPrimitive { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the type is a collection.
        /// </summary>
        public bool IsCollection { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the type is a generic (template) type.
        /// </summary>
        public bool IsGeneric { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the type is an ICloneable.
        /// </summary>
        public bool IsICloneable { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the type is an ISettable.
        /// </summary>
        public bool IsISettable { get; set; }

        /// <summary>
        /// The underlying element count of this type.
        /// </summary>
        /// <remarks>
        /// This property is basically for primitives.
        /// The collection element count is usually variable,
        /// hence the element count is undetermined.
        /// </remarks>
        public int ElementCount { get; set; }

        /// <summary>
        /// The underlying element type of this type.
        /// </summary>
        /// <remarks>
        /// This property is basically for primitives.
        /// The generic or collection element type is determined by the actual definition.
        /// </remarks>
        public EditorTypeInfo ElementType { get; set; }

        /// <summary>
        /// The data model definition.
        /// </summary>
        public EditorDataModelDefinition DataModelDefinition { get; set; }
    }
}
