﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;

using EffectMaker.DataModelMaker.Core.Definitions;

namespace EffectMaker.DataModelMaker.Core.Collections
{
    /// <summary>
    /// List class for storing child definitions.
    /// </summary>
    /// <typeparam name="T">The definition type.</typeparam>
    [Serializable]
    public class DefinitionList<T> : IList<T> where T : DefinitionBase
    {
        /// <summary>The parent definition of this list.</summary>
        private DefinitionBase parent = null;

        /// <summary>The internal list.</summary>
        private List<T> internalList = null;

        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="parent">The parent definition of the list.</param>
        public DefinitionList(DefinitionBase parent)
        {
            this.parent = parent;
            this.internalList = new List<T>();
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="parent">The parent definition of the list.</param>
        /// <param name="collection">The elements to add to the list.</param>
        public DefinitionList(DefinitionBase parent, IEnumerable<T> collection)
        {
            this.parent = parent;
            this.internalList = new List<T>(collection);
        }

        /// <summary>
        /// Constructor with capacity.
        /// </summary>
        /// <param name="parent">The parent definition of the list.</param>
        /// <param name="capacity">The capacity the allocate to the list.</param>
        public DefinitionList(DefinitionBase parent, int capacity)
        {
            this.parent = parent;
            this.internalList = new List<T>(capacity);
        }

        /// <summary>
        /// Get or set the parent definition of the list.
        /// </summary>
        public DefinitionBase Parent
        {
            get
            {
                return this.parent;
            }

            set
            {
                this.parent = value;
                foreach (DefinitionBase item in this.internalList)
                {
                    if (item != null)
                    {
                        item.Parent = this.parent;
                    }
                }
            }
        }

        /// <summary>
        /// Get the count of elements in the list.
        /// </summary>
        public int Count
        {
            get { return this.internalList.Count; }
        }

        /// <summary>
        /// Get the flag indicating whether this list is read only.
        /// </summary>
        public bool IsReadOnly
        {
            get { return false; }
        }

        /// <summary>
        /// Get or set the item at the specified index.
        /// </summary>
        /// <param name="index">The index to the item.</param>
        /// <returns>The item.</returns>
        public T this[int index]
        {
            get { return this.internalList[index]; }
            set { this.internalList[index] = value; }
        }

        /// <summary>
        /// Get enumerator for looping through the list.
        /// </summary>
        /// <returns>The enumerator.</returns>
        public IEnumerator<T> GetEnumerator()
        {
            return this.internalList.GetEnumerator();
        }

        /// <summary>
        /// Get enumerator for looping through the list.
        /// </summary>
        /// <returns>The enumerator.</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return this.internalList.GetEnumerator();
        }

        /// <summary>
        /// Adds an item to the list.
        /// </summary>
        /// <param name="item">The item to add.</param>
        public void Add(T item)
        {
            if (item != null)
            {
                item.Parent = this.parent;
            }

            this.internalList.Add(item);
        }

        /// <summary>
        /// Adds items to the list.
        /// </summary>
        /// <param name="items">The items to add.</param>
        public void AddRange(IEnumerable<T> items)
        {
            if (items != null)
            {
                foreach (T item in items)
                {
                    if (item != null)
                    {
                        item.Parent = this.parent;
                    }
                }
            }

            this.internalList.AddRange(items);
        }

        /// <summary>
        /// Insert the item to the specified index.
        /// </summary>
        /// <param name="index">The index.</param>
        /// <param name="item">The item to insert.</param>
        public void Insert(int index, T item)
        {
            if (item != null)
            {
                item.Parent = this.parent;
            }

            this.internalList.Insert(index, item);
        }

        /// <summary>
        /// Remove the item at the given index.
        /// </summary>
        /// <param name="index">The index.</param>
        public void RemoveAt(int index)
        {
            if (index >= 0 && index < this.internalList.Count)
            {
                this.internalList[index].Parent = null;
            }

            this.internalList.RemoveAt(index);
        }

        /// <summary>
        /// Remove all the items in the list.
        /// </summary>
        public void Clear()
        {
            foreach (DefinitionBase item in this.internalList)
            {
                if (item != null)
                {
                    item.Parent = null;
                }
            }

            this.internalList.Clear();
        }

        /// <summary>
        /// Find the index of the specified item.
        /// </summary>
        /// <param name="item">The item to find.</param>
        /// <returns>The index of the item in the list, or -1 if the item is not found.</returns>
        public int IndexOf(T item)
        {
            return this.internalList.IndexOf(item);
        }

        /// <summary>
        /// Check if the list contains the given item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns>True if the item is in the list.</returns>
        public bool Contains(T item)
        {
            return this.internalList.Contains(item);
        }

        /// <summary>
        /// Check if an element exists in the list.
        /// </summary>
        /// <param name="match">The delegate the defines the conditions of the elements to search for.</param>
        /// <returns>True if the element exists.</returns>
        public bool Exists(Predicate<T> match)
        {
            return this.internalList.Exists(match);
        }

        /// <summary>
        /// Sort the list.
        /// </summary>
        public void Sort()
        {
            this.internalList.Sort();
        }

        /// <summary>
        /// Sort the list with custom comparison delegation.
        /// </summary>
        /// <param name="comparison">The comparison delegation.</param>
        public void Sort(Comparison<T> comparison)
        {
            this.internalList.Sort(comparison);
        }

        /// <summary>
        /// Copy the items to the specified array, starting at the given destination index.
        /// </summary>
        /// <param name="array">The array to copy items to.</param>
        /// <param name="arrayIndex">The index to start copy items at.</param>
        public void CopyTo(T[] array, int arrayIndex)
        {
            this.internalList.CopyTo(array, arrayIndex);
        }

        /// <summary>
        /// Remove the item from the list.
        /// </summary>
        /// <param name="item">The item to remove.</param>
        /// <returns>True if the item is in the list and is successfully removed.</returns>
        public bool Remove(T item)
        {
            bool result = this.internalList.Remove(item);
            if (result == true && item != null)
            {
                item.Parent = null;
            }

            return result;
        }
    }
}
