﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Threading;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.Application.CommandLine;
using EffectMaker.DataModelMaker.Application.Properties;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Log;

using EffectMaker.UIControls.DebugConsole;
using EffectMaker.UIControls.Threading;

namespace EffectMaker.DataModelMaker.Application
{
    /// <summary>
    /// The main entry point for the application.
    /// </summary>
    static class Program
    {
        /// <summary>Constant for attaching console to the parent process.</summary>
        private const int AttachParentProcess = -1;

        /// <summary>
        /// Get the synchronization context for the application.
        /// </summary>
        public static SynchronizationContext SyncContext { get; private set; }

        /// <summary>
        /// Get the flag indicating whether the application is running on command line mode.
        /// </summary>
        public static bool IsCommandLineMode { get; private set; }

        /// <summary>
        /// Get the flag indicating whether to show debug console.
        /// </summary>
        public static bool IsDebugConsoleEnabled { get; private set; }

        /// <summary>
        /// Get the main form.
        /// </summary>
        public static MainForm MainForm { get; private set; }

        /// <summary>
        /// Get the debug console.
        /// </summary>
        public static DebugConsole DebugConsole { get; private set; }

        /// <summary>
        /// Get the flag indicating whether the control tree view dialog
        /// should be enabled or not.
        /// </summary>
        public static bool IsControlTreeViewEnabled { get; private set; }

        /// <summary>
        /// Get or set the workspace definition.
        /// </summary>
        public static WorkspaceDefinition WorkspaceDefinition { get; set; }

        /// <summary>
        /// Get or set the workspace view model.
        /// </summary>
        public static dynamic WorkspaceVM { get; set; }

        #region C++ console functions DLL imports

        /// <summary>
        /// Attach console to the application.
        /// </summary>
        /// <param name="processId">The process ID of the console.</param>
        /// <returns>True on success.</returns>
        [System.Runtime.InteropServices.DllImport("kernel32.dll")]
        private static extern bool AttachConsole(int processId);

        /// <summary>
        /// Allocate console for the application.
        /// </summary>
        /// <returns>True on success.</returns>
        [System.Runtime.InteropServices.DllImport("kernel32.dll")]
        private static extern bool AllocConsole();

        /// <summary>
        /// Release the console the application allocated.
        /// </summary>
        /// <returns>True on success.</returns>
        [System.Runtime.InteropServices.DllImport("kernel32.dll")]
        private static extern bool FreeConsole();

        #endregion

        /// <summary>
        /// The main entry point for the application.
        /// </summary>
        /// <param name="args">The command line arguments.</param>
        [STAThread]
        static void Main(string[] args)
        {
            #if !DEBUG
            System.Windows.Forms.Application.ThreadException += ApplicationThreadException;
            AppDomain.CurrentDomain.UnhandledException += CurrentDomainUnhandledException;
            #endif

            // ロガーを初期化
            Logger.Initialize();

            Program.IsControlTreeViewEnabled = false;
            Program.IsDebugConsoleEnabled = true;

            // Parse command-line arguments if any.
            ICommandLineProcessor[] processors = ArgumentParser.Parse(args);
            Program.IsCommandLineMode = processors != null && processors.Length > 0;
            CoreSettings.IsWindowMode = !Program.IsCommandLineMode;

            if (Program.IsCommandLineMode == true)
            {
                // USD, UDD変化用にコマンドラインモードがあったが、現在は未使用
                Program.RunCommandLineApplication(processors);
            }
            else
            {
                Program.RunWindowApplication();
            }

            // ロガーを破棄
            Logger.Release();
        }

        /// <summary>
        /// Initialize the application.
        /// </summary>
        /// <returns>True on success.</returns>
        private static bool Initialize()
        {
            UISynchronizationContextHolder.SynchronizationContext = Program.SyncContext;

            // Create debug console.
            Program.DebugConsole = null;
            if (Program.IsDebugConsoleEnabled == true && Constants.IsDebug == true)
            {
                Program.DebugConsole = new DebugConsole();

                Program.DebugConsole.Load += OnDebugConsoleLoad;
                Program.DebugConsole.FormClosing += OnDebugConsoleClosing;

                Logger.RegisterLogHandler(Program.DebugConsole);

                Program.DebugConsole.Show();
            }

            return true;
        }

        /// <summary>
        /// Deinitialize the application.
        /// </summary>
        private static void Deinitialize()
        {
            // Save settings
            Settings.Default.Save();

            // Unregister debug console.
            if (Program.DebugConsole != null)
            {
                Logger.UnregisterLogHandler(Program.DebugConsole);
            }
        }

        /// <summary>
        /// Run window application.
        /// </summary>
        private static void RunWindowApplication()
        {
            System.Windows.Forms.Application.EnableVisualStyles();
            System.Windows.Forms.Application.SetCompatibleTextRenderingDefault(false);

            Program.MainForm = new MainForm();
            Program.SyncContext = SynchronizationContext.Current;

            Program.MainForm.FormClosing += OnMainFormClosing;

            // Initialize the application.
            if (Program.Initialize() == true)
            {
                // Run the application.
                System.Windows.Forms.Application.Run(Program.MainForm);

                // Shutdown.
                Program.Deinitialize();
            }
        }

        /// <summary>
        /// Run command line application.
        /// </summary>
        /// <param name="processors">Command line processors.</param>
        private static void RunCommandLineApplication(ICommandLineProcessor[] processors)
        {
            Program.MainForm = null;
            Program.SyncContext = null;

            // Check if silent mode should be enabled.
            bool isSilentMode = processors.Any(p => p is SilentModeProcessor);

            // Do not show debug console on silent mode.
            Program.IsDebugConsoleEnabled = !isSilentMode;

            // Initialize the application.
            if (Program.Initialize() == false)
            {
                return;
            }

            bool shouldFreeConsole = false;

            try
            {
                OperatingSystem myOsInfo = Environment.OSVersion;
                if ((isSilentMode == false) &&
                    (myOsInfo.Platform == PlatformID.Win32NT) &&
                    (myOsInfo.Version.CompareTo(new Version(5, 1, 0)) > 0))
                {
                    // Try to attach the process to parent's console window.
                    if (AttachConsole(AttachParentProcess) == false)
                    {
                        // Failed attaching to parent's console, allocate one.
                        shouldFreeConsole = AllocConsole();
                    }
                    else
                    {
                        Console.WriteLine(string.Empty); // Just an empty new line.
                        shouldFreeConsole = false;
                    }
                }

                // Register the console logger.
                Logger.RegisterLogHandler(new ConsoleLogger());

                // Execute the command-line processors.
                foreach (ICommandLineProcessor processor in processors)
                {
                    processor.Execute();
                }
            }
            finally
            {
                if (shouldFreeConsole == true)
                {
                    Console.ReadLine();
                    FreeConsole();
                }

                Program.Deinitialize();
            }
        }

        /// <summary>
        /// Handle main form FormClosing event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private static void OnMainFormClosing(object sender, FormClosingEventArgs e)
        {
            if (Program.DebugConsole != null && Program.DebugConsole.Visible == true)
            {
                Program.DebugConsole.Close();
            }
        }

        /// <summary>
        /// Handle DebugConsole Load event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private static void OnDebugConsoleLoad(object sender, EventArgs e)
        {
            if (Settings.Default.DebugConsoleLocation != null)
            {
                Program.DebugConsole.Location = Settings.Default.DebugConsoleLocation;
            }

            if (Settings.Default.DebugConsoleSize != null)
            {
                Program.DebugConsole.Size = Settings.Default.DebugConsoleSize;
            }
        }

        /// <summary>
        /// Handle DebugConsole FormClosing event.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private static void OnDebugConsoleClosing(object sender, FormClosingEventArgs e)
        {
            if (Program.DebugConsole != null)
            {
                Settings.Default.DebugConsoleLocation = Program.DebugConsole.Location;

                if (Program.DebugConsole.WindowState == FormWindowState.Normal)
                {
                    Settings.Default.DebugConsoleSize = Program.DebugConsole.Size;
                }
                else
                {
                    Settings.Default.DebugConsoleSize = Program.DebugConsole.RestoreBounds.Size;
                }
            }
        }

        /// <summary>
        /// Handle unhandled exceptions from current application domain.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private static void CurrentDomainUnhandledException(
            object sender,
            UnhandledExceptionEventArgs e)
        {
            var builder = new System.Text.StringBuilder();

            builder.AppendLine("Unhandled exception occurred in the application domain.");

            var now = DateTime.Now;
            builder.AppendLine(now.ToShortDateString() + " " + now.ToShortTimeString());

            builder.AppendLine("================================================================================");

            try
            {
                var ex = e.ExceptionObject as Exception;
                if (ex == null)
                {
                    builder.Append("Application crashed by unknown error.");
                    return;
                }

                builder.AppendLine(ex.ToString());
            }
            finally
            {
                if (builder != null && builder.Length > 0)
                {
                    string logFilePath = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
                    logFilePath = Path.Combine(logFilePath, "crash_log.txt");

                    File.WriteAllText(logFilePath, builder.ToString());

                    MessageBox.Show(
                        "An unhandled exception occurred, please refer to crash_log.txt in the executable folder for further information.",
                        "Unhandled Exception");
                }
                else
                {
                    MessageBox.Show(
                        "An unhandled exception occurred, the application might terminate execution.",
                        "Unhandled Exception");
                }
            }
        }

        /// <summary>
        /// Handle exceptions from the application main thread.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private static void ApplicationThreadException(
            object sender,
            ThreadExceptionEventArgs e)
        {
            var builder = new System.Text.StringBuilder();

            builder.AppendLine("Unhandled exception occurred in the application thread.");

            var now = DateTime.Now;
            builder.AppendLine(now.ToShortDateString() + " " + now.ToShortTimeString());

            builder.AppendLine("================================================================================");

            try
            {
                if (e.Exception == null)
                {
                    builder.Append("Application crashed by unknown error.");
                    return;
                }

                builder.AppendLine(e.Exception.ToString());
            }
            finally
            {
                if (builder != null && builder.Length > 0)
                {
                    string logFilePath = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
                    logFilePath = Path.Combine(logFilePath, "crash_log.txt");

                    File.WriteAllText(logFilePath, builder.ToString());

                    MessageBox.Show(
                        "An unhandled exception occurred, please refer to crash_log.txt in the executable folder for further information.",
                        "Unhandled Exception");
                }
                else
                {
                    MessageBox.Show(
                        "An unhandled exception occurred, the application might terminate execution.",
                        "Unhandled Exception");
                }
            }
        }
    }
}
