﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.Application.CommandLine
{
    /// <summary>
    /// Command line argument parser.
    /// </summary>
    public class ArgumentParser
    {
        /// <summary>The registered commands.</summary>
        private static Dictionary<string, Type> registeredCommands =
            new Dictionary<string, Type>();

        /// <summary>The registered processors.</summary>
        private static HashSet<Type> registeredProcessors = new HashSet<Type>();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static ArgumentParser()
        {
            RegisterCommand("h", typeof(OutputHelpProcessor));
            RegisterCommand("s", typeof(SilentModeProcessor));
        }

        /// <summary>
        /// Parse the command-line arguments.
        /// </summary>
        /// <param name="args">The arguments.</param>
        /// <returns>The parsed command count.</returns>
        public static ICommandLineProcessor[] Parse(string[] args)
        {
            var parsedCommands = new List<KeyValuePair<string, List<string>>>();

            // First parse all the commands and their arguments.
            int currCommandIndex = -1;
            foreach (string arg in args)
            {
                if (arg.StartsWith("-") == true)
                {
                    string currCommand = arg.Substring(1);
                    currCommandIndex = parsedCommands.Count;
                    parsedCommands.Add(
                        new KeyValuePair<string, List<string>>(currCommand, new List<string>()));
                }
                else if (currCommandIndex >= 0)
                {
                    List<string> commandArgs = parsedCommands[currCommandIndex].Value;
                    commandArgs.Add(arg);
                }
            }

            bool shouldOutputHelp = false;

            // Create command-line processors and set parsed commands to them.
            var processors = new List<ICommandLineProcessor>();
            foreach (KeyValuePair<string, List<string>> command in parsedCommands)
            {
                // Get the processor type that is registered with the command.
                Type processorType;
                if (registeredCommands.TryGetValue(command.Key, out processorType) == false)
                {
                    shouldOutputHelp = true;
                    break;
                }

                if (processorType == typeof(OutputHelpProcessor))
                {
                    shouldOutputHelp = true;
                    break;
                }

                // Find the processor of the type.
                ICommandLineProcessor processor =
                    processors.FirstOrDefault(p => p.GetType() == processorType);
                if (processor == null)
                {
                    processor = Activator.CreateInstance(processorType) as ICommandLineProcessor;
                    processors.Add(processor);
                }

                // Set the command to the command-line processor.
                if (processor.SetCommandArguments(command.Key, command.Value) == false)
                {
                    shouldOutputHelp = true;
                    break;
                }
            }

            if (shouldOutputHelp == true)
            {
                processors.Clear();
                processors.Add(new OutputHelpProcessor());
            }

            return processors.ToArray();
        }

        /// <summary>
        /// Output command-line help.
        /// </summary>
        public static void OutputHelp()
        {
            Logger.Log(
                "ConsoleLogger",
                LogLevels.Information,
                "Data model maker, command-line help.");

            Logger.Log(
                "ConsoleLogger",
                LogLevels.Information,
                "Usage:");

            foreach (Type processorType in registeredProcessors)
            {
                var processor = Activator.CreateInstance(processorType) as ICommandLineProcessor;

                IEnumerable<KeyValuePair<string, string>> descs = processor.GetCommandDescriptions();

                foreach (KeyValuePair<string, string> desc in descs)
                {
                    Logger.Log(
                        "ConsoleLogger",
                        LogLevels.Information,
                        "  -{0}    {1}",
                        desc.Key,
                        desc.Value);
                }
            }
        }

        /// <summary>
        /// Register command to the specified type of command-line processor.
        /// </summary>
        /// <param name="command">The command to register.</param>
        /// <param name="processorType">The type of the command-line processor.</param>
        private static void RegisterCommand(string command, Type processorType)
        {
            if (registeredCommands.ContainsKey(command) == true)
            {
                Debug.Assert(false, "The command '" + command + "' has already been registered.");
            }

            if (typeof(ICommandLineProcessor).IsAssignableFrom(processorType) == false)
            {
                Debug.Assert(false, "The type '" + processorType.Name + "' does not implement ICommandLineProcessor interface.");
            }

            registeredCommands.Add(command.ToLower(), processorType);

            if (registeredProcessors.Contains(processorType) == false)
            {
                registeredProcessors.Add(processorType);
            }
        }
    }
}
