﻿// ========================================================================
// <copyright file="EmptyListTrimer.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================
namespace NintendoWare.ToolDevelopmentKit.Xml
{
    using System.Collections;
    using System.Collections.Generic;
    using System.Reflection;
    using NintendoWare.ToolDevelopmentKit.Conversion;

    /// <summary>
    /// 空のリストにnullを設定するユーティリティです。
    /// <remarks>
    /// シリアライズ用クラスに適用し、不要なXML要素を出力しないようにします。
    /// </remarks>
    /// </summary>
    public static class EmptyListTrimer
    {
        /// <summary>
        /// 空のリストを削除します。
        /// </summary>
        /// <param name="target">対象オブジェクトです。</param>
        /// <param name="context">処理状況です。</param>
        public static void Trim(object target, Context context)
        {
            if (target == null || target.GetType().IsPrimitive)
            {
                return;
            }

            if (context.CheckedInstancies.Exists(
                (instance) => object.ReferenceEquals(instance, target)))
            {
                return;
            }

            context.CheckedInstancies.Add(target);

            foreach (PropertyInfo propInfo in
                target.GetType().GetProperties(ConverterUtil.PropartyBindingFlags))
            {
                // パラメータ付きプロパティ(インデクサ等)はスキップ。
                if (propInfo.GetIndexParameters().Length > 0)
                {
                    continue;
                }

                // 読めなけばスキップ。
                PropertyInfo srcPropInfo = ConverterUtil.GetReadableProperty(propInfo);
                if (srcPropInfo == null)
                {
                    continue;
                }

                // 書けなけばスキップ。
                srcPropInfo = ConverterUtil.GetWritableProperty(srcPropInfo);
                if (srcPropInfo == null)
                {
                    continue;
                }

                // リストの場合は...
                object propValue = srcPropInfo.GetValue(target, null);
                IList list = propValue as IList;
                if (list != null)
                {
                    if (CheckListMarkedTobeTrimed(srcPropInfo) && list.Count == 0)
                    {
                        // 空のリストは、コンテキストに記録して、nullを設定します。
                        context.TrimedLists.Add(new Context.TrimedList(target, srcPropInfo, list));
                        srcPropInfo.SetValue(target, null, null);
                    }
                    else
                    {
                        foreach (object element in list)
                        {
                            Trim(element, context);
                        }
                    }
                }
                else
                {
                    // 再帰的に削除します。
                    Trim(propValue, context);
                }
            }
        }

        /// <summary>
        /// 削除対象としてマークされているかを確認します。
        /// </summary>
        /// <param name="propInfo">プロパティ情報です。</param>
        /// <returns>削除対象なら true が返ります。</returns>
        private static bool CheckListMarkedTobeTrimed(PropertyInfo propInfo)
        {
            return ConverterUtil.FindPropertyAttribute<TrimListIfEmptyAttribute>(
                propInfo) != null;
        }

        /// <summary>
        /// 処理コンテキストクラスです。
        /// </summary>
        public class Context
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public Context()
            {
                this.CheckedInstancies = new List<object>();
                this.TrimedLists = new List<TrimedList>();
            }

            /// <summary>
            /// チェック済みインスタンスのリストを取得または設定します。
            /// <remarks>
            /// 再帰処理が無限ループに陥らないために必要です。
            /// </remarks>
            /// </summary>
            public List<object> CheckedInstancies { get; set; }

            /// <summary>
            /// 削除されたリストを取得または設定します。
            /// <remarks>
            /// 復元のために保存してあります。
            /// </remarks>
            /// </summary>
            public List<TrimedList> TrimedLists { get; set; }

            /// <summary>
            /// 復元します。
            /// </summary>
            public void RestoreAll()
            {
                foreach (TrimedList trimedList in this.TrimedLists)
                {
                    trimedList.Resotore();
                }
            }

            /// <summary>
            /// 削除されたリストです。
            /// </summary>
            public class TrimedList
            {
                /// <summary>
                /// 持ち主を取得または設定します。
                /// </summary>
                private readonly object Owner;

                /// <summary>
                /// プロパティ情報を取得または設定します。
                /// </summary>
                private readonly PropertyInfo PropertyInfo;

                /// <summary>
                /// 削除されたリストを取得または設定します。
                /// </summary>
                private readonly IList TrimedListValue;

                /// <summary>
                /// コンストラクタです。
                /// </summary>
                /// <param name="owner">持ち主です。</param>
                /// <param name="propertyInfo">プロパティ情報です。</param>
                /// <param name="trimedListValue">削除されたリストです。</param>
                public TrimedList(object owner, PropertyInfo propertyInfo, IList trimedListValue)
                {
                    this.Owner = owner;
                    this.PropertyInfo = propertyInfo;
                    this.TrimedListValue = trimedListValue;
                }

                /// <summary>
                /// 復元します。
                /// </summary>
                public void Resotore()
                {
                    this.PropertyInfo.SetValue(this.Owner, this.TrimedListValue, null);
                }
            }
        }
    }
}
