﻿// ========================================================================
// <copyright file="MatrixUtility.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;

    /// <summary>
    /// 行列ユーティリティです。
    /// </summary>
    public static class MatrixUtility
    {
        private static readonly string[] propertyNameTable = new string[]
            {
                "M00", "M01", "M02", "M03",
                "M10", "M11", "M12", "M13",
                "M20", "M21", "M22", "M23",
                "M30", "M31", "M32", "M33",
            };

        /// <summary>
        /// カメラ行列をカメラ位置、カメラターゲット、アップベクトルを用いて設定します。
        /// </summary>
        /// <param name="mtx">セット対象となる行列です。
        /// このメソッドを呼ぶ出す前に初期化をしてください。</param>
        /// <param name="eye">カメラ位置です。</param>
        /// <param name="target">カメラターゲットです。</param>
        /// <param name="up">カメラのアップベクトルです。</param>
        public static void LookAt(Matrix34 mtx, Vector3 eye, Vector3 target, Vector3 up)
        {
            // RVL_SDK/buid/libraries/mtx/mtx.cのC_MTXLookAtから移植
            Ensure.Argument.NotNull(eye);
            Ensure.Argument.NotNull(target);
            Ensure.Argument.NotNull(up);
            Ensure.Argument.NotNull(mtx);

            // カメラ座標系のＺ方向
            Vector3 lookReverse = new Vector3(Vector3.Normalize(eye - target));

            // カメラ座標系のＸ方向
            Vector3 right = new Vector3(Vector3.Normalize(Vector3.Cross(up, lookReverse)));

            // カメラ座標系のＹ方向
            Vector3 vUp = new Vector3(Vector3.Cross(lookReverse, right));

            mtx[0, 0] = right.X;
            mtx[0, 1] = right.Y;
            mtx[0, 2] = right.Z;
            mtx[0, 3] = -((eye.X * right.X) + (eye.Y * right.Y) + (eye.Z * right.Z));

            mtx[1, 0] = vUp.X;
            mtx[1, 1] = vUp.Y;
            mtx[1, 2] = vUp.Z;
            mtx[1, 3] = -((eye.X * vUp.X) + (eye.Y * vUp.Y) + (eye.Z * vUp.Z));

            mtx[2, 0] = lookReverse.X;
            mtx[2, 1] = lookReverse.Y;
            mtx[2, 2] = lookReverse.Z;
            mtx[2, 3] =
                -((eye.X * lookReverse.X) + (eye.Y * lookReverse.Y) + (eye.Z * lookReverse.Z));
        }

        /// <summary>
        /// カメラ行列をカメラの回転量を用いて設定します。
        /// </summary>
        /// <param name="mtx">セット対象となる行列です。
        /// このメソッドを呼ぶ出す前に初期化してください。</param>
        /// <param name="eye">カメラ位置です。</param>
        /// <param name="target">カメラターゲットです。</param>
        /// <param name="twist">カメラ座標系でのＺ軸周りの回転量（Radian）です。</param>
        public static void Aim(Matrix34 mtx, Vector3 eye, Vector3 target, float twist)
        {
            // NW4R/Library/build/libraries/g3d/src/g3d_camera.cppから移植
            Ensure.Argument.NotNull(eye);
            Ensure.Argument.NotNull(target);
            Ensure.Argument.NotNull(mtx);

            // カメラ座標系のＺ方向
            Vector3 lookReverse = new Vector3(eye - target);

            if ((lookReverse.X == 0.0f) && (lookReverse.Z == 0.0f))
            {
                // カメラとターゲットのXZ座標が同じ場合ツイストは定義されない
                mtx[0, 0] = 1.0f;
                mtx[0, 1] = 0.0f;
                mtx[0, 2] = 0.0f;
                mtx[0, 3] = -eye.X;

                mtx[1, 0] = 0.0f;
                mtx[1, 1] = 0.0f;

                mtx[2, 0] = 0.0f;
                mtx[2, 2] = 0.0f;

                if (lookReverse.Y <= 0.0f)
                {
                    // 真上を見るとき
                    mtx[1, 2] = 1.0f;
                    mtx[1, 3] = -eye.Z;

                    mtx[2, 1] = -1.0f;
                    mtx[2, 3] = eye.Y;
                }
                else
                {
                    // 真下を見るとき
                    mtx[1, 2] = -1.0f;
                    mtx[1, 3] = eye.Z;

                    mtx[2, 1] = 1.0f;
                    mtx[2, 3] = -eye.Y;
                }
            }
            else
            {
                // カメラ座標系のＸ方向
                Vector3 r = new Vector3(lookReverse.Z, 0.0f, -lookReverse.X);

                lookReverse = Vector3.Normalize(lookReverse);
                r = Vector3.Normalize(r);

                // カメラ座標系のＹ方向
                Vector3 u = Vector3.Cross(lookReverse, r);

                float st = (float)Math.Sin(twist);
                float ct = (float)Math.Cos(twist);

                Vector3 right = new Vector3();
                Vector3 up = new Vector3();

                // r軸, u軸をru平面上でcameraTwistだけ半時計回りに回転させてrightを求める
                // r.y == 0であることに注意
                right.X = (st * u.X) + (ct * r.X);
                right.Y = st * u.Y;
                right.Z = (st * u.Z) + (ct * r.Z);

                up.X = (ct * u.X) - (st * r.X);
                up.Y = ct * u.Y;
                up.Z = (ct * u.Z) - (st * r.Z);

                // right
                mtx[0, 0] = right.X;
                mtx[0, 1] = right.Y;
                mtx[0, 2] = right.Z;
                mtx[0, 3] = -Vector3.Dot(eye, right);

                // up
                mtx[1, 0] = up.X;
                mtx[1, 1] = up.Y;
                mtx[1, 2] = up.Z;
                mtx[1, 3] = -Vector3.Dot(eye, up);

                // look
                mtx[2, 0] = lookReverse.X;
                mtx[2, 1] = lookReverse.Y;
                mtx[2, 2] = lookReverse.Z;
                mtx[2, 3] = -Vector3.Dot(eye, lookReverse);
            }
        }

        /// <summary>
        /// カメラ行列をカメラの回転角度を用いて設定します。
        /// </summary>
        /// <param name="mtx">セット対象となる行列です。
        /// このメソッドを呼ぶ出す前に初期化してください。</param>
        /// <param name="eye">カメラ位置です。</param>
        /// <param name="rotate">カメラのＸＹＺそれぞれのオイラー回転角度（Radian）です。</param>
        public static void Rotate(Matrix34 mtx, Vector3 eye, Vector3 rotate)
        {
            // NW4R/Library/build/libraries/g3d/src/g3d_camera.cppから移植
            Ensure.Argument.NotNull(eye);
            Ensure.Argument.NotNull(rotate);
            Ensure.Argument.NotNull(mtx);

            float sx, sy, sz, cx, cy, cz;
            sx = (float)Math.Sin((double)rotate.X);
            sy = (float)Math.Sin((double)rotate.Y);
            sz = (float)Math.Sin((double)rotate.Z);

            cx = (float)Math.Cos((double)rotate.X);
            cy = (float)Math.Cos((double)rotate.Y);
            cz = (float)Math.Cos((double)rotate.Z);

            // Z軸, X軸, Y軸の順番で回転させている
            Vector3 right = new Vector3();
            Vector3 up = new Vector3();
            Vector3 back = new Vector3();

            right.X = (sx * sy * sz) + (cy * cz);
            right.Y = cx * sz;
            right.Z = (sx * cy * sz) - (sy * cz);

            up.X = (sx * sy * cz) - (cy * sz);
            up.Y = cx * cz;
            up.Z = (sx * cy * cz) + (sy * sz);

            back.X = cx * sy;
            back.Y = -sx;
            back.Z = cx * cy;

            mtx[0, 0] = right.X;
            mtx[0, 1] = right.Y;
            mtx[0, 2] = right.Z;
            mtx[0, 3] = -Vector3.Dot(eye, right);

            mtx[1, 0] = up.X;
            mtx[1, 1] = up.Y;
            mtx[1, 2] = up.Z;
            mtx[1, 3] = -Vector3.Dot(eye, up);

            mtx[2, 0] = back.X;
            mtx[2, 1] = back.Y;
            mtx[2, 2] = back.Z;
            mtx[2, 3] = -Vector3.Dot(eye, back);
        }

        /// <summary>
        /// 射影行列を視錐体を元に作成します。
        /// Z方向の深度は(-1,0)になります。
        /// </summary>
        /// <param name="mtx">セット対象となる行列です。
        /// このメソッドを呼ぶ出す前に初期化してください。</param>
        /// <param name="left">ニアクリップの左辺です。</param>
        /// <param name="right">ニアクリップの右辺です。</param>
        /// <param name="bottom">ニアクリップの下辺です。</param>
        /// <param name="top">ニアクリップの上辺です。</param>
        /// <param name="near">カメラからニアクリップまでの距離です。</param>
        /// <param name="far">カメラからファークリップまでの距離です。</param>
        public static void Frustum(Matrix44 mtx, float left, float right, float bottom, float top, float near, float far)
        {
            // RVL_SDK/buid/libraries/mtx/mtx44.cのC_MTXFrustumから移植
            Ensure.Argument.True(top != bottom);
            Ensure.Argument.True(left != right);
            Ensure.Argument.True(near != far);
            Ensure.Argument.NotNull(mtx);

            float tmp;

            // x
            tmp = 1.0f / (right - left);
            mtx[0, 0] = (2.0f * near) * tmp;
            mtx[0, 1] = 0.0f;
            mtx[0, 2] = (right + left) * tmp;
            mtx[0, 3] = 0.0f;

            // y
            tmp = 1.0f / (top - bottom);
            mtx[1, 0] = 0.0f;
            mtx[1, 1] = (2.0f * near) * tmp;
            mtx[1, 2] = (top + bottom) * tmp;
            mtx[1, 3] = 0.0f;

            // z
            tmp = 1.0f / (far - near);
            mtx[2, 0] = 0.0f;
            mtx[2, 1] = 0.0f;
            mtx[2, 2] = far * tmp;
            mtx[2, 3] = far * near * tmp;

            // w
            mtx[3, 0] = 0.0f;
            mtx[3, 1] = 0.0f;
            mtx[3, 2] = -1.0f;
            mtx[3, 3] = 0.0f;
        }

        /// <summary>
        /// 射影行列を作成します。
        /// Z方向の深度は(-1,0)です。
        /// </summary>
        /// <param name="mtx">セット対象となる行列です。
        /// このメソッドを呼ぶ出す前に初期化してください。</param>
        /// <param name="fovy">垂直視野角です。</param>
        /// <param name="aspect">クリップ面のアスペクト比です。</param>
        /// <param name="near">カメラからニアクリップまでの距離です。</param>
        /// <param name="far">カメラからファークリップまでの距離です。</param>
        public static void Perspective(
            Matrix44 mtx,
            float fovy,
            float aspect,
            float near,
            float far)
        {
            // RVL_SDK/buid/libraries/mtx/mtx44.cのC_MTXPerspectiveから移植
            Ensure.Argument.True((fovy > 0.0f) && (fovy < Math.PI));
            Ensure.Argument.True(aspect != 0.0f);
            Ensure.Argument.True(near != far);
            Ensure.Argument.NotNull(mtx);

            float tmp;

            float angle = fovy * 0.5f;

            float cot = (float)(1.0f / Math.Tan(angle));

            // x
            mtx[0, 0] = cot / aspect;
            mtx[0, 1] = 0.0f;
            mtx[0, 2] = 0.0f;
            mtx[0, 3] = 0.0f;

            // y
            mtx[1, 0] = 0.0f;
            mtx[1, 1] = cot;
            mtx[1, 2] = 0.0f;
            mtx[1, 3] = 0.0f;

            // z
            tmp = 1.0f / (far - near);
            mtx[2, 0] = 0.0f;
            mtx[2, 1] = 0.0f;
            mtx[2, 2] = far * tmp; // -(far + near) * tmp;
            mtx[2, 3] = far * near * tmp; // -2 * far * near * tmp;

            // w
            mtx[3, 0] = 0.0f;
            mtx[3, 1] = 0.0f;
            mtx[3, 2] = -1.0f;
            mtx[3, 3] = 0.0f;
        }

        /// <summary>
        /// 平行射影行列を作成します。
        /// </summary>
        /// <param name="mtx">セット対象となる行列です。
        /// このメソッドを呼ぶ出す前に初期化してください。</param>
        /// <param name="left">ニアクリップの左辺です。</param>
        /// <param name="right">ニアクリップの右辺です。</param>
        /// <param name="bottom">ニアクリップの下辺です。</param>
        /// <param name="top">ニアクリップの上辺です。</param>
        /// <param name="near">カメラからニアクリップまでの距離です。</param>
        /// <param name="far">カメラからファークリップまでの距離です。</param>
        public static void Ortho(Matrix44 mtx, float left, float right, float bottom, float top, float near, float far)
        {
            // RVL_SDK/buid/libraries/mtx/mtx44.cのC_MTXOrthoから移植
            Ensure.Argument.True(top != bottom);
            Ensure.Argument.True(left != right);
            Ensure.Argument.True(near != far);
            Ensure.Argument.NotNull(mtx);

            float tmp;

            // x
            tmp = 1.0f / (right - left);
            mtx[0, 0] = 2.0f * tmp;
            mtx[0, 1] = 0.0f;
            mtx[0, 2] = 0.0f;
            mtx[0, 3] = -(right + left) * tmp;

            // y
            tmp = 1.0f / (top - bottom);
            mtx[1, 0] = 0.0f;
            mtx[1, 1] = 2.0f * tmp;
            mtx[1, 2] = 0.0f;
            mtx[1, 3] = -(top + bottom) * tmp;

            // z
            tmp = 1.0f / (far - near);
            mtx[2, 0] = 0.0f;
            mtx[2, 1] = 0.0f;
            mtx[2, 2] = tmp;
            mtx[2, 3] = near * tmp;

            // w
            mtx[3, 0] = 0.0f;
            mtx[3, 1] = 0.0f;
            mtx[3, 2] = 0.0f;
            mtx[3, 3] = 1.0f;
        }

        /// <summary>
        /// Maya 用のテクスチャ座標変換行列を作成します。
        /// </summary>
        /// <remarks>
        /// [S] [T] [t(0.5,0.5)] [R] [t(-0.5,-0.5)]
        /// [S] = |scaleS      0 0 0|
        ///       |     0 scaleT 0 0|
        ///       |     0      0 1 0|
        ///       |     0      0 0 1|
        /// [T] = t(-translateS, -translateT)
        /// [R] = |rotateCos -rotateSin 0 0|
        ///       |rotateSin  rotateCos 0 0|
        ///       |        0          0 1 0|
        ///       |        0          0 0 1|
        /// </remarks>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="scaleS">S軸のスケール値です。</param>
        /// <param name="scaleT">T軸のスケール値です。</param>
        /// <param name="rotate">回転値です。</param>
        /// <param name="translateS">S軸の移動値です。</param>
        /// <param name="translateT">T軸の移動値です。</param>
        public static void TextureMatrixForMaya(
            Matrix44 mtx,
            float scaleS,
            float scaleT,
            float rotate,
            float translateS,
            float translateT)
        {
            float rotateSin = (float)Math.Sin((double)rotate);
            float rotateCos = (float)Math.Cos((double)rotate);
            mtx[0, 0] = scaleS * rotateCos;
            mtx[0, 1] = -scaleS * rotateSin;
            mtx[0, 3] = scaleS * ((0.5f * rotateSin) - (0.5f * rotateCos) + 0.5f - translateS);

            mtx[1, 0] = scaleT * rotateSin;
            mtx[1, 1] = scaleT * rotateCos;
            mtx[1, 3] = scaleT * ((-0.5f * rotateSin) - (0.5f * rotateCos) + 0.5f - translateT);

            mtx[0, 2] = mtx[1, 2] = 0.0f;
            mtx[2, 0] = mtx[2, 1] = mtx[2, 3] = 0.0f;
            mtx[3, 0] = mtx[3, 1] = mtx[3, 2] = 0.0f;
            mtx[2, 2] = mtx[3, 3] = 1.0f;
        }

        /// <summary>
        /// Softimage 用のテクスチャ座標変換行列を作成します。
        /// </summary>
        /// <remarks>
        /// [S] [R] [T]
        /// [S] = |scaleS      0 0 0|
        ///       |     0 scaleT 0 0|
        ///       |     0      0 1 0|
        ///       |     0      0 0 1|
        /// [R] の回転方向は Maya および Max とは逆方向が正
        /// [R] = | rotateCos rotateSin 0 0|
        ///       |-rotateSin rotateCos 0 0|
        ///       |         0         0 1 0|
        ///       |         0         0 0 1|
        /// [T] = t(-translateS, -translateT)
        /// </remarks>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="scaleS">S軸のスケール値です。</param>
        /// <param name="scaleT">T軸のスケール値です。</param>
        /// <param name="rotate">回転値です。</param>
        /// <param name="translateS">S軸の移動値です。</param>
        /// <param name="translateT">T軸の移動値です。</param>
        public static void TextureMatrixForSoftimage(
            Matrix44 mtx,
            float scaleS,
            float scaleT,
            float rotate,
            float translateS,
            float translateT)
        {
            float rotateSin = (float)Math.Sin((double)rotate);
            float rotateCos = (float)Math.Cos((double)rotate);
            mtx[0, 0] = scaleS * rotateCos;
            mtx[0, 1] = scaleS * rotateSin;
            mtx[0, 3] = scaleS * ((-rotateCos * translateS) - (rotateSin * translateT));

            mtx[1, 0] = -scaleT * rotateSin;
            mtx[1, 1] = scaleT * rotateCos;
            mtx[1, 3] = scaleT * ((rotateSin * translateS) - (rotateCos * translateT));

            mtx[0, 2] = mtx[1, 2] = 0.0f;
            mtx[2, 0] = mtx[2, 1] = mtx[2, 3] = 0.0f;
            mtx[3, 0] = mtx[3, 1] = mtx[3, 2] = 0.0f;
            mtx[2, 2] = mtx[3, 3] = 1.0f;
        }

        /// <summary>
        /// 3ds Max 用のテクスチャ座標変換行列を作成します。
        /// </summary>
        /// <remarks>
        /// [t(0.5,05)] [S] [R] [t(-0.5,-0.5)] [T]
        /// [S] = |scaleS      0 0 0|
        ///       |     0 scaleT 0 0|
        ///       |     0      0 1 0|
        ///       |     0      0 0 1|
        /// [R] = |rotateCos -rotateSin 0 0|
        ///       |rotateSin  rotateCos 0 0|
        ///       |        0          0 1 0|
        ///       |        0          0 0 1|
        /// [T] = t(-translateS, translateT)
        /// </remarks>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="scaleS">S軸のスケール値です。</param>
        /// <param name="scaleT">T軸のスケール値です。</param>
        /// <param name="rotate">回転値です。</param>
        /// <param name="translateS">S軸の移動値です。</param>
        /// <param name="translateT">T軸の移動値です。</param>
        public static void TextureMatrixForMax(
            Matrix44 mtx,
            float scaleS,
            float scaleT,
            float rotate,
            float translateS,
            float translateT)
        {
            float rotateSin = (float)Math.Sin((double)rotate);
            float rotateCos = (float)Math.Cos((double)rotate);
            float scaleSinS = scaleS * rotateSin;
            float scaleCosS = scaleS * rotateCos;
            float scaleSinT = scaleT * rotateSin;
            float scaleCosT = scaleT * rotateCos;
            float ts = -translateS - 0.5f;
            float tt = translateT - 0.5f;

            mtx[0, 0] = scaleCosS;
            mtx[0, 1] = -scaleSinS;
            mtx[0, 3] = (scaleCosS * ts) - (scaleSinS * tt) + 0.5f;

            mtx[1, 0] = scaleSinT;
            mtx[1, 1] = scaleCosT;
            mtx[1, 3] = (scaleSinT * ts) + (scaleCosT * tt) + 0.5f;

            mtx[0, 2] = mtx[1, 2] = 0.0f;
            mtx[2, 0] = mtx[2, 1] = mtx[2, 3] = 0.0f;
            mtx[3, 0] = mtx[3, 1] = mtx[3, 2] = 0.0f;
            mtx[2, 2] = mtx[3, 3] = 1.0f;
        }

        /// <summary>
        /// テクスチャマトリクス用射影行列を視錐体を元に作成します。
        /// </summary>
        /// <param name="mtx">セット対象となる行列です。</param>
        /// <param name="left">ニアクリップの左辺です。</param>
        /// <param name="right">ニアクリップの右辺です。</param>
        /// <param name="bottom">ニアクリップの下辺です。</param>
        /// <param name="top">ニアクリップの上辺です。</param>
        /// <param name="near">カメラからニアクリップまでの距離です。</param>
        /// <param name="scaleS">S軸方向のスケール値です。</param>
        /// <param name="scaleT">T軸方向のスケール値です。</param>
        /// <param name="translateS">S軸方向の移動値です。</param>
        /// <param name="translateT">T軸方向の移動値です。</param>
        public static void TextureProjectionFrustum(
            Matrix34 mtx,
            float left,
            float right,
            float bottom,
            float top,
            float near,
            float scaleS,
            float scaleT,
            float translateS,
            float translateT)
        {
            // RVL_SDK/build/libraries/mtx/src/mtx.c の C_MTXLightFrustum から移植
            Ensure.Argument.True(top != bottom);
            Ensure.Argument.True(left != right);
            Ensure.Argument.NotNull(mtx);

            float reverseWidth = 1.0f / (right - left);
            mtx[0, 0] = ((2.0f * near) * reverseWidth) * scaleS;
            mtx[0, 1] = 0.0f;
            mtx[0, 2] = (((right + left) * reverseWidth) * scaleS) - translateS;
            mtx[0, 3] = 0.0f;

            float reverseHeight = 1.0f / (top - bottom);
            mtx[1, 0] = 0.0f;
            mtx[1, 1] = ((2.0f * near) * reverseHeight) * scaleT;
            mtx[1, 2] = (((top + bottom) * reverseHeight) * scaleT) - translateT;
            mtx[1, 3] = 0.0f;

            mtx[2, 0] = 0.0f;
            mtx[2, 1] = 0.0f;
            mtx[2, 2] = -1.0f;
            mtx[2, 3] = 0.0f;
        }

        /// <summary>
        /// テクスチャマトリクス用射影行列を作成します。
        /// </summary>
        /// <param name="mtx">セット対象となる行列です。</param>
        /// <param name="fovy">垂直視野角です。</param>
        /// <param name="aspect">クリップ面のアスペクト比です。</param>
        /// <param name="scaleS">S軸方向のスケール値です。</param>
        /// <param name="scaleT">T軸方向のスケール値です。</param>
        /// <param name="translateS">S軸方向の移動値です。</param>
        /// <param name="translateT">T軸方向の移動値です。</param>
        public static void TextureProjectionPerspective(
            Matrix34 mtx,
            float fovy,
            float aspect,
            float scaleS,
            float scaleT,
            float translateS,
            float translateT)
        {
            // RVL_SDK/build/libraries/mtx/src/mtx.c の C_MTXLightPerspective から移植
            Ensure.Argument.True((fovy > 0.0f) && (fovy < Math.PI));
            Ensure.Argument.True((aspect != 0.0f));
            Ensure.Argument.NotNull(mtx);

            float angle = fovy * 0.5f;
            float cot = 1.0f / (float)System.Math.Tan(angle);

            mtx[0, 0] = (cot / aspect) * scaleS;
            mtx[0, 1] = 0.0f;
            mtx[0, 2] = -translateS;
            mtx[0, 3] = 0.0f;

            mtx[1, 0] = 0.0f;
            mtx[1, 1] = cot * scaleT;
            mtx[1, 2] = -translateT;
            mtx[1, 3] = 0.0f;

            mtx[2, 0] = 0.0f;
            mtx[2, 1] = 0.0f;
            mtx[2, 2] = -1.0f;
            mtx[2, 3] = 0.0f;
        }

        /// <summary>
        /// 平行射影行列を作成します。
        /// </summary>
        /// <param name="mtx">セット対象となる行列です。</param>
        /// <param name="left">ニアクリップの左辺です。</param>
        /// <param name="right">ニアクリップの右辺です。</param>
        /// <param name="bottom">ニアクリップの下辺です。</param>
        /// <param name="top">ニアクリップの上辺です。</param>
        /// <param name="scaleS">S軸方向のスケール値です。</param>
        /// <param name="scaleT">T軸方向のスケール値です。</param>
        /// <param name="translateS">S軸方向の移動値です。</param>
        /// <param name="translateT">T軸方向の移動値です。</param>
        public static void TextureProjectionOrtho(
            Matrix34 mtx,
            float left,
            float right,
            float bottom,
            float top,
            float scaleS,
            float scaleT,
            float translateS,
            float translateT)
        {
            // RVL_SDK/build/libraries/mtx/src/mtx.c の C_MTXLightPerspective から移植
            Ensure.Argument.True(top != bottom);
            Ensure.Argument.True(left != right);
            Ensure.Argument.NotNull(mtx);

            float reverseWidth = 1.0f / (right - left);
            mtx[0, 0] = 2.0f * reverseWidth * scaleS;
            mtx[0, 1] = 0.0f;
            mtx[0, 2] = 0.0f;
            mtx[0, 3] = ((-(right + left) * reverseWidth) * scaleS) + translateS;

            float reverseHeight = 1.0f / (top - bottom);
            mtx[1, 0] = 0.0f;
            mtx[1, 1] = (2.0f * reverseHeight) * scaleT;
            mtx[1, 2] = 0.0f;
            mtx[1, 3] = ((-(top + bottom) * reverseHeight) * scaleT) + translateT;

            mtx[2, 0] = 0.0f;
            mtx[2, 1] = 0.0f;
            mtx[2, 2] = 0.0f;
            mtx[2, 3] = 1.0f;
        }

        /// <summary>
        /// 指定した行列成分のプロパティー名を作成します
        /// </summary>
        /// <param name="row">行列成分の行の 0 から始まるインデックスです。</param>
        /// <param name="column">行列成分の列の 0 から始まるインデックスです。</param>
        /// <returns>行列成分のプロパティー名です。</returns>
        /// <remarks>
        /// 要素数はrow、columnそれぞれ４未満をサポートします。４以上はアサーションで停止します。
        /// </remarks>
        public static string MakePropertyName(int row, int column)
        {
            Assertion.Argument.True(row < 4);
            Assertion.Argument.True(column < 4);

            int index = (row * 4) + column;
            return propertyNameTable[index];
        }
    }
}
