﻿// ========================================================================
// <copyright file="Matrix2DScaleRotateFunction.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.Diagnostics;

    /// <summary>
    /// 行列の２次元のスケールと回転に関する演算関数です。
    /// </summary>
    /// <typeparam name="TMatrix">テンプレート行列型です。</typeparam>
    internal static class Matrix2DScaleRotateFunction<TMatrix> where TMatrix : IMatrix, new()
    {
        /// <summary>
        /// x 軸、y 軸に沿ってスケーリングする行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="x">x 軸に沿ったスケールです。</param>
        /// <param name="y">y 軸に沿ったスケールです。</param>
        public static void SetScale(TMatrix mtx, float x, float y)
        {
            Ensure.Argument.NotNull(mtx);

            DebugMatrixRowColumnCount(mtx);

            mtx.SetIdentity();
            mtx[0, 0] = x;
            mtx[1, 1] = y;
        }

        /// <summary>
        /// 原点を中心に回転する行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public static void SetRotate(TMatrix mtx, float radian)
        {
            Ensure.Argument.NotNull(mtx);

            DebugMatrixRowColumnCount(mtx);

            float s = (float)Math.Sin(radian);
            float c = (float)Math.Cos(radian);

            mtx.SetIdentity();
            mtx[0, 0] = c;
            mtx[0, 1] = -s;
            mtx[1, 0] = s;
            mtx[1, 1] = c;
        }

        /// <summary>
        /// 行列の行数と列数が演算に対して適切であることを確認します。
        /// </summary>
        /// <param name="mtx">対象の行列です。</param>
        [Conditional("DEBUG")]
        private static void DebugMatrixRowColumnCount(TMatrix mtx)
        {
            Debug.Assert(mtx.RowCount >= 2, "Unexpected case!");
            Debug.Assert(mtx.ColumnCount >= 2, "Unexpected case!");
        }
    }
}
