﻿// ========================================================================
// <copyright file="ArrayMetaData.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.Collections.Generic;
    using System.Xml.Serialization;
    using NintendoWare.ToolDevelopmentKit.Collections;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// データに関する配列型属性情報を表すためのクラスです。
    /// </summary>
    /// <typeparam name="TValue">データの型です。</typeparam>
    public class ArrayMetaData<TValue> : ObservableObject, IMetaData
    {
        private readonly IList<TValue> values = new ObservableList<TValue>();
        private string key = string.Empty;

        //-----------------------------------------------------------------
        // コンストラクタ
        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ArrayMetaData()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="arrayMetaData">コピー元の頂点属性です。</param>
        private ArrayMetaData(ArrayMetaData<TValue> arrayMetaData)
        {
            this.Set(arrayMetaData);
        }

        //-----------------------------------------------------------------
        // プロパティ
        //-----------------------------------------------------------------

        /// <summary>
        /// データを検索する際のキーワードです。
        /// </summary>
        public virtual string Key
        {
            get
            {
                return this.key;
            }

            set
            {
                Ensure.Argument.NotNull(value);

                this.key = value;
                OnPropertyChanged("Key");
            }
        }

        /// <summary>
        /// データ内容のリストを取得します。
        /// </summary>
        public IList<TValue> Values
        {
            get { return this.values; }
        }

        //-----------------------------------------------------------------
        // メソッド
        //-----------------------------------------------------------------

        /// <summary>
        /// 文字列化を行います。
        /// </summary>
        /// <returns>文字列化した結果です。</returns>
        public override string ToString()
        {
            return this.Key;
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 引数 obj と等しいは判定された場合、true が返ります。
        /// </summary>
        /// <param name="obj">比較するオブジェクトです。</param>
        /// <returns>等しい場合 true が返ります。</returns>
        public override bool Equals(object obj)
        {
            return this.GetHashCode().Equals(obj.GetHashCode());
        }

        /// <summary>
        /// ハッシュコードを取得します。
        /// </summary>
        /// <returns>ハッシュコードです。</returns>
        public override int GetHashCode()
        {
            return this.Key.GetHashCode();
        }

        //-----------------------------------------------------------------
        // クローン、コピー
        //-----------------------------------------------------------------

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>このインスタンスのコピーである新しいオブジェクトです。</returns>
        public virtual object Clone()
        {
            return new ArrayMetaData<TValue>(this);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            this.Set(source as ArrayMetaData<TValue>);
        }

        /// <summary>
        /// 配列型属性情報を設定します。
        /// </summary>
        /// <param name="source">設定する配列型属性情報です。</param>
        protected void Set(ArrayMetaData<TValue> source)
        {
            Ensure.Argument.NotNull(source);

            this.Key = source.Key;
            //// TODO: 下記では浅いコピーになるので、Vector型などはちゃんとクローンされるように修正する。
            ListUtility.Copy(source.Values, this.Values);
        }
    }
}
