﻿// ========================================================================
// <copyright file="Log.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.Logs
{
    using System;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// ログです。
    /// </summary>
    public class Log : ObservableObject, ILog
    {
        //-----------------------------------------------------------------
        private ILogLocation location = new NullLogLocation();
        private string message = string.Empty;
        private DateTime timeStamp = DateTime.Now;

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="location">ILogLocationを持つオブジェクトです。</param>
        public Log(ILogLocation location)
        {
            Ensure.Argument.NotNull(location);

            this.location = location;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="log">ILogを持つオブジェクトです。</param>
        public Log(ILog log)
        {
            this.Set(log);
        }

        /// <summary>
        /// ログの出力位置を取得します。
        /// </summary>
        public ILogLocation Location
        {
            get
            {
                return this.location;
            }
        }

        /// <summary>
        /// ログのメッセージを取得または設定します。
        /// </summary>
        public string Message
        {
            get
            {
                return this.message;
            }

            set
            {
                Ensure.Argument.NotNull(value);

                this.message = value;
                OnPropertyChanged("Message");
            }
        }

        /// <summary>
        /// ログが出力された日時を取得します。
        /// </summary>
        public DateTime TimeStamp
        {
            get
            {
                return this.timeStamp;
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public virtual object Clone()
        {
            // ロケーションは new せずに共有する
            return new Log(this);
        }

        //-----------------------------------------------------------------
        // プロパティの取得または設定
        //-----------------------------------------------------------------

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public virtual void Set(object source)
        {
            this.Set(source as ILog);
        }

        /// <summary>
        /// ログを設定します。
        /// </summary>
        /// <param name="log">設定するログです。</param>
        public virtual void Set(ILog log)
        {
            Ensure.Argument.NotNull(log);

            this.location = log.Location;
            this.Message = log.Message;
            this.timeStamp = log.TimeStamp;
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            if (other == this)
            {
                return true;
            }

            if ((other == null) || (other.GetType() != GetType()))
            {
                return false;
            }

            return this.Equals(other as Log);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return
                this.Location.GetHashCode() ^
                this.Message.GetHashCode() ^
                this.TimeStamp.GetHashCode();
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        protected bool Equals(Log other)
        {
            Ensure.Argument.NotNull(other);
            return
                this.Location.Equals(other.Location) &&
                (this.Message == other.Message) &&
                this.TimeStamp.Equals(other.TimeStamp);
        }
    }
}
