﻿// ========================================================================
// <copyright file="CacheManager.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.ComponentModel
{
    using System;
    using System.Collections.Generic;
    using System.Linq;

    /// <summary>
    /// キャッシュを管理します。
    /// </summary>
    public sealed class CacheManager : IDisposable
    {
        private readonly object lockObject = new object();
        private readonly LinkedList<CacheItem> cacheItems = new LinkedList<CacheItem>();
        private readonly int capacity = 0;

        private bool isDisposed = false;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public CacheManager()
            : this(0, true)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="capacity">
        /// 格納できるキャッシュの最大数を指定します。
        /// 0 以下を指定すると、最大数を設定しません。
        /// </param>
        public CacheManager(int capacity)
            : this(capacity, true)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="capacity">
        /// 格納できるキャッシュの最大数を指定します。
        /// 0 以下を指定すると、最大数を設定しません。
        /// </param>
        /// <param name="isGarbageCacheCollectionTarget">
        /// ガベージキャッシュコレクションの対象とするかどうかを指定します。
        /// </param>
        public CacheManager(int capacity, bool isGarbageCacheCollectionTarget)
        {
            this.capacity = capacity;

            if (isGarbageCacheCollectionTarget)
            {
                GarbageCacheCollector.RegisterCacheManager(this);
            }
        }

        /// <summary>
        /// ファイナライザです。
        /// </summary>
        ~CacheManager()
        {
            this.Dispose();
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// キャッシュアイテムを列挙します。
        /// </summary>
        public IEnumerable<CacheItem> CacheItems
        {
            get
            {
                return this.cacheItems;
            }
        }

        /// <summary>
        /// キャッシュ操作のロックオブジェクトを取得します。
        /// </summary>
        internal object LockObject
        {
            get
            {
                return this.lockObject;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// オブジェクトを破棄します。
        /// </summary>
        public void Dispose()
        {
            if (this.isDisposed)
            {
                return;
            }

            this.isDisposed = true;
            GarbageCacheCollector.UnregisterCacheManager(this);

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// キャッシュを取得します。
        /// </summary>
        /// <param name="keyObject">キャッシュのキーとなるオブジェクトを指定します。</param>
        /// <returns>
        /// キャッシュを返します。
        /// キャッシュが見つからない場合は null を返します。
        /// </returns>
        public CacheItem TryGetCache(object keyObject)
        {
            Ensure.Argument.NotNull(keyObject);

            lock (this.lockObject)
            {
                LinkedListNode<CacheItem> node = this.FindNode(keyObject);

                if (node == null)
                {
                    return null;
                }

                // キャッシュアイテムを先頭に移動します。
                this.cacheItems.Remove(node);
                this.cacheItems.AddFirst(node);

                return node.Value;
            }
        }

        /// <summary>
        /// キャッシュを追加します。
        /// 許容数を超えている場合は、最も古いキャッシュを削除します。
        /// </summary>
        /// <param name="keyObject">キャッシュのキーとなるオブジェクトを指定します。</param>
        /// <param name="value">キャッシュ値を指定します。</param>
        public void AddCache(object keyObject, object value)
        {
            Ensure.Argument.NotNull(keyObject);
            Ensure.Argument.NotNull(value);

            lock (this.lockObject)
            {
                if (this.capacity == 0)
                {
                    return;
                }

                // 許容数を超えている場合は、最も古いキャッシュを削除します。
                if (this.capacity > 0 &&
                    this.cacheItems.Count >= this.capacity)
                {
                    this.cacheItems.RemoveLast();
                }

                this.cacheItems.AddFirst(
                    this.CreateItem(keyObject, value));
            }
        }

        /// <summary>
        /// キャッシュを削除します。
        /// </summary>
        /// <param name="keyObject">キャッシュのキーとなるオブジェクトを指定します。</param>
        public void RemoveCache(object keyObject)
        {
            Ensure.Argument.NotNull(keyObject);

            lock (this.lockObject)
            {
                LinkedListNode<CacheItem> node = this.FindNode(keyObject);

                if (node != null)
                {
                    this.cacheItems.Remove(node);
                }
            }
        }

        /// <summary>
        /// キャッシュを削除します。
        /// </summary>
        /// <param name="cacheItem">キャッシュアイテムを指定します。</param>
        public void RemoveCache(CacheItem cacheItem)
        {
            lock (this.lockObject)
            {
                Ensure.Argument.NotNull(cacheItem);
                this.cacheItems.Remove(cacheItem);
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// キャッシュアイテムを作成します。
        /// </summary>
        /// <param name="keyObject">キャッシュのキーとなるオブジェクトを指定します。</param>
        /// <param name="value">キャッシュ値を指定します。</param>
        /// <returns>作成したキャッシュアイテムを返します。</returns>
        private CacheItem CreateItem(object keyObject, object value)
        {
            Assertion.Argument.NotNull(keyObject);
            Assertion.Argument.NotNull(value);
            return new CacheItem(keyObject, value);
        }

        /// <summary>
        /// キャッシュアイテムのノードを検索します。
        /// </summary>
        /// <param name="keyObject">キャッシュのキーとなるオブジェクトを指定します。</param>
        /// <returns>キャッシュアイテムのノードを返します。</returns>
        private LinkedListNode<CacheItem> FindNode(object keyObject)
        {
            Ensure.Argument.NotNull(keyObject);

            LinkedListNode<CacheItem> node = this.cacheItems.First;

            while (node != null)
            {
                if (object.ReferenceEquals(node.Value.KeyObject, keyObject))
                {
                    return node;
                }

                node = node.Next;
            }

            return null;
        }
    }
}
