﻿// ========================================================================
// <copyright file="KeyedListDecorator.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace NintendoWare.ToolDevelopmentKit.Collections
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.ComponentModel;
    using ToolDevelopmentKit;
    using ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// 名前引き可能なリストのデコレータです。
    /// IListはINotifyCollectionChangedを実装していることを必要とします。
    /// </summary>
    /// <typeparam name="TKey">キーのテンプレート型です。</typeparam>
    /// <typeparam name="TValue">要素のテンプレート型です。</typeparam>
    public class KeyedListDecorator<TKey, TValue> : ListDecorator<TValue>, IKeyedList<TKey, TValue>
    {
        private readonly IList<TValue> list;
        private readonly Dictionary<TKey, TValue> dictionary = new Dictionary<TKey, TValue>();
        private GetKeyForItemMethod getKeyForItem;

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="list">デコレート対象のIListを持つリストオブジェクトです。</param>
        /// <param name="method">キーを決定するメソッドです。</param>
        public KeyedListDecorator(IList<TValue> list, GetKeyForItemMethod method)
            : base(list)
        {
            Ensure.Argument.NotNull(list);
            Ensure.Argument.NotNull(method);
            this.list = list;
            this.getKeyForItem = method;

            INotifyCollectionChanged observableList = list as INotifyCollectionChanged;
            Ensure.Operation.ObjectNotNull(observableList);
            observableList.CollectionChanged += this.UpdateDictionary;
        }

        //-----------------------------------------------------------------
        // デリゲート
        //-----------------------------------------------------------------

        /// <summary>
        /// リストの要素からキーを取得します。
        /// </summary>
        /// <param name="value">取得に使用する要素です。</param>
        /// <returns>取得したキーです。</returns>
        public delegate TKey GetKeyForItemMethod(TValue value);

        //-----------------------------------------------------------------
        // 要素の取得または設定
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定されたキーに関連付けられている要素を取得します。
        /// </summary>
        /// <param name="key">キーを指定します。</param>
        /// <returns>指定されたキーに関連付けられている要素を返します。</returns>
        public TValue GetValue(TKey key)
        {
            return this.dictionary[key];
        }

        /// <summary>
        /// 指定されたキーに関連付けられている要素を取得します。
        /// </summary>
        /// <param name="key">キーを指定します。</param>
        /// <param name="value">
        /// 指定されたキーが存在する場合、そのキーに関連付けられている要素を返します。
        /// それ以外の場合は value パラメータの型に対する既定の要素を返します。
        /// </param>
        /// <returns>指定されたキーに関連付けられている要素を返します。</returns>
        public bool TryGetValue(TKey key, out TValue value)
        {
            return this.dictionary.TryGetValue(key, out value);
        }

        /// <summary>
        /// 指定キーがコレクションに含まれているかどうかを調べます。
        /// </summary>
        /// <param name="key">検索するキーです。</param>
        /// <returns>含まれている場合は true、含まれていない場合は falseを返します。</returns>
        public bool ContainsKey(TKey key)
        {
            return this.dictionary.ContainsKey(key);
        }

        /// <summary>
        /// 指定されたすべての要素を辞書に追加します。
        /// </summary>
        /// <param name="items">要素の列挙子を指定します。</param>
        private void AddItemsToDictionary(IEnumerable items)
        {
            foreach (TValue item in items)
            {
                this.dictionary.Add(this.getKeyForItem(item), item);
            }
        }

        /// <summary>
        /// 指定されたすべての要素を辞書から削除します。
        /// </summary>
        /// <param name="items">要素の列挙子を指定します。</param>
        private void RemoveItemsFromDitionary(IEnumerable items)
        {
            foreach (TValue item in items)
            {
                this.dictionary.Remove(this.getKeyForItem(item));
            }
        }

        /// <summary>
        /// 辞書を再構築します。
        /// </summary>
        private void ResetDictionary()
        {
            this.dictionary.Clear();
            this.AddItemsToDictionary(this.list);
        }

        /// <summary>
        /// コレクションの変更に合わせて、辞書の内容を更新します。
        /// </summary>
        /// <param name="sender">イベントの送信者です。</param>
        /// <param name="e">コレクション変更イベントデータです。</param>
        private void UpdateDictionary(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    this.AddItemsToDictionary(e.NewItems);
                    break;

                case NotifyCollectionChangedAction.Remove:
                    this.RemoveItemsFromDitionary(e.OldItems);
                    break;

                case NotifyCollectionChangedAction.Move:
                    // 並びが替わっただけなら何もしません。
                    break;

                default:
                    this.ResetDictionary();
                    break;
            }
        }
    }
}

