﻿// ========================================================================
// <copyright file="ComplexEmitterData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_1.Minor_4.Build_0.Revision_0
{
    /// <summary>
    /// シンプルエミッタデータ
    /// </summary>
    public class ComplexEmitterData : BaseDataModel<ComplexEmitterData>
    {
        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public ComplexEmitterData()
        {
            this.EmitterVelDir = new Vector3();     // エミッタ速度の方向
            this.GravityDir = new Vector3();        // 重力の向き
            this.SpreadVec = new Vector3();         // 拡散ベクトル
            this.VolumeRadius = new Vector3();      // ボリューム半径
            this.UVScroll = new Vector2();          // ＵＶスクロール

            this.TexRes = new TextureRes();
            this.SubTexRes = new TextureRes();

            this.FluctuationFlag = new FluctuationFlag();

            this.FieldFlag = new FieldMask();
            this.FieldFlag.Flag = (uint)0;

            this.TexPatData = new TexPatData();
            this.SubTexPatData = new TexPatData();

            this.AnimEditData = new AnimEditData();
            this.ChildFlag = new ChildFlag();
            this.ChildAssistData = new ChildAssistData();
            this.ChildData = new ChildData();
            this.ChildTexPatData = new TexPatData();
            this.FieldCollisionData = new FieldCollisionData();
            this.FieldConvergenceData = new FieldConvergenceData();
            this.FieldMagnetData = new FieldMagnetData();
            this.FieldPosAddData = new FieldPosAddData();
            this.FieldRandomData = new FieldRandomData();
            this.VolumeRadius = new Vector3();
            this.StripeData = new StripeData();
            this.FluctuationData = new FluctuationData();
            this.FieldSpinData = new FieldSpinData();
            this.UVScroll         = new Vector2( 0.0f, 0.0f );
            this.UVScrollInit     = new Vector2( 0.0f, 0.0f );
            this.UVScrollInitRand = new Vector2( 0.0f, 0.0f );
            this.UVScale          = new Vector2( 0.0f, 0.0f );
            this.UVScaleInit      = new Vector2( 1.0f, 1.0f );
            this.UVScaleInitRand  = new Vector2( 0.0f, 0.0f );
            this.SubTexUVScroll         = new Vector2(0.0f, 0.0f);
            this.SubTexUVScrollInit     = new Vector2(0.0f, 0.0f);
            this.SubTexUVScrollInitRand = new Vector2(0.0f, 0.0f);
            this.SubTexUVScale          = new Vector2(0.0f, 0.0f);
            this.SubTexUVScaleInit      = new Vector2(1.0f, 1.0f);
            this.SubTexUVScaleInitRand  = new Vector2(0.0f, 0.0f);

            this.UserData = new UserData();
            this.UserDataGroups = new EmitterUserData[9];
            for ( int i=0;i<this.UserDataGroups.Length;++i )
            {
                this.UserDataGroups[i] = new EmitterUserData();
                this.UserDataGroups[i].Index = i - 1;
            }

            this.UserShaderParams = new float[32];

            this.UserShaderParamGroups = new UserShaderParameter[8];
            for ( int i=0;i<this.UserShaderParamGroups.Length;++i )
            {
                this.UserShaderParamGroups[i] = new UserShaderParameter();
                this.UserShaderParamGroups[i].Index = i;
            }

            this.AnimTableList = new EmitterAnimTableList();

            this.EmitterColor0 = new RgbaColor( 1.0f, 1.0f, 1.0f, 1.0f );
            this.EmitterColor1 = new RgbaColor( 1.0f, 1.0f, 1.0f, 1.0f );

            this.EmitterFormScale = new Vector3();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public ComplexEmitterData(ComplexEmitterData source)
            : this()
        {
            this.FluctuationFlag = new FluctuationFlag();
            this.FieldFlag = new FieldMask();

            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.VolumeType = VolumeType.Point;

            this.DispParticle = true;

            this.EmitRate = 1.0f;
            this.VolumeLatitudeEnabled = false;
            this.VolumeLatitudeDir = VolumeLatitudeDirection.PlusY;
            this.VolumeLatitude = 180.0f;
            this.VolumeSweepParam = (uint)0xffffffff;
            this.VolumeSweepStart = 0;
            this.VolumeSweepStartRandom = false;
            this.VolumeSurfacePosRand = 0.0f;
            this.VolumeHollow = 0;
            this.VolumeTblIndex = 0;
            this.Volume64TblIndex = 0;

            this.EmitDistEnabled = false;
            this.EmitDistUnit = 1.0f;
            this.EmitDistMin = 0.0f;
            this.EmitDistMax = 10.0f;
            this.EmitDistMargin = 0.0f;

            this.GravityVolume = 0;
            this.GravityDir.Set(0, -1, 0);

            this.FluctuationFlag.ApplyAlpha = true;
            this.FluctuationFlag.ApplyScale = true;

            this.ChildFlag.EmitterFollow = true;
            this.ChildFlag.AlphaInherit = true;
            this.ChildFlag.ScaleInherit = true;
            this.ChildFlag.RotateInherit = true;

            this.FieldFlag.Flag = (uint)0;
            this.StripeFlag = 0;

            this.TexPatData.Initialize();
            this.SubTexPatData.Initialize();
            this.AnimEditData.Initialize();
            this.ChildAssistData.Initialize();
            this.ChildData.Initialize();
            this.ChildTexPatData.Initialize();
            this.FieldMagnetData.Initialize();
            this.FieldRandomData.Initialize();
            this.FluctuationData.Initialize();
            this.FieldSpinData.Initialize();
            this.StripeData.Initialize();

            this.AnimTableList.Initialize();

            for ( int i=0;i<this.UserShaderParamGroups.Length;++i )
                this.UserShaderParamGroups[i].Initialize();

            this.AirRegist = 1.0f;
            this.DynamicsRandom = 0.0f;
            this.LifeStep = 10;
            this.LineCenter = 0.0f;

            this.UseSoftParticle = false;
            this.SoftParticleFadeDistance = 0.0f;
            this.SoftParticleVolumeParam = 0.0f;
            this.ShaderType = ShaderType.Particle;
            this.ShaderParam0 = 0.0f;
            this.ShaderParam1 = 0.0f;
            this.UI_ApplyAlphaToRefraction = true;

            this.ZBufATestType = ZBufATestType.Normal;
            this.DisplaySide = DisplaySideTypes.Both;

            this.UserShaderSwitch = 0x01010101;
            this.UserShaderFlags = 0;
            for (int i = 0; i < 32; ++i)
            {
                this.UserShaderParams[i] = 0.0f;
            }

            this.UserShaderCompileDef1 = NWKernel.DefaultUserShaderCompileDef1;
            this.UserShaderCompileDef2 = NWKernel.DefaultUserShaderCompileDef2;

            this.MeshType = MeshType.Particle;
            this.ParticleType  = BillboardType.Billboard;
            this.StripeType    = StripeType.Stripe;
            this.BillboardPrimitiveFileSource = string.Empty;

            this.EmitterVelDir.Set(0, 1, 0);
            this.EmitterVelDirAngle = 0.0f;
            this.YAxisDiffusionVel = 0.0f;

            this.InitPosRand = 0.0f;

            float worldScale = s_WorldScaleRange / 100.0f;
            {
                this.FigureVelocity = 0.1f * 1.0f * worldScale; // *velScale;
                this.VolumeRadius.Set(1.0f * worldScale, 1.0f * worldScale, 1.0f * worldScale); // *posScale;
            }

            this.UVShiftAnimMode      = 0;
            this.IsUVShiftAnimComplex = false;
            this.UVRot                = 0.0f;
            this.UVRotInit            = 0.0f;
            this.UVRotInitRand        = 0.0f;

            this.UVScroll.Set( 0.0f, 0.0f );
            this.UVScrollInit.Set( 0.0f, 0.0f );
            this.UVScrollInitRand.Set( 0.0f, 0.0f );
            this.UVScale.Set( 0.0f, 0.0f );
            this.UVScaleInit.Set( 1.0f, 1.0f );
            this.UVScaleInitRand.Set( 0.0f, 0.0f );

            this.SubTexUVShiftAnimMode = 0;
            this.SubTexIsUVShiftAnimComplex = false;
            this.SubTexUVRot = 0.0f;
            this.SubTexUVRotInit = 0.0f;
            this.SubTexUVRotInitRand = 0.0f;

            this.SubTexUVScroll.Set(0.0f, 0.0f);
            this.SubTexUVScrollInit.Set(0.0f, 0.0f);
            this.SubTexUVScrollInitRand.Set(0.0f, 0.0f);
            this.SubTexUVScale.Set(0.0f, 0.0f);
            this.SubTexUVScaleInit.Set(1.0f, 1.0f);
            this.SubTexUVScaleInitRand.Set(0.0f, 0.0f);

            this.UserData.Initialize();

            this.UserCallbackID = -1;
            for ( int i=0;i<this.UserDataGroups.Length;++i )
                this.UserDataGroups[i].Initialize();

            this.EmitterColor0.R = 1.0f;
            this.EmitterColor0.G = 1.0f;
            this.EmitterColor0.B = 1.0f;
            this.EmitterColor0.A = 1.0f;

            this.EmitterColor1.R = 1.0f;
            this.EmitterColor1.G = 1.0f;
            this.EmitterColor1.B = 1.0f;
            this.EmitterColor1.A = 1.0f;

            this.EmitterAlpha = 1.0f;

            this.EmitterFormScale.Set( 1.0f, 1.0f, 1.0f );

            this.ParticleSortType = ParticleSortType.Ascending;
            //             this.ChildDataOffset = 9;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(ComplexEmitterData source)
        {
            Ensure.Argument.NotNull(source);

            // SimpleEmitterData
            /*
             *  SimpleEmitterData用
             */
            this.Type = source.Type;
            this.Name = source.Name;

            this.DispParticle = source.DispParticle;

            this.EmitterVelDir.Set(source.EmitterVelDir);
            this.GravityDir.Set(source.GravityDir);
            this.SpreadVec.Set(source.SpreadVec);
            this.VolumeRadius.Set(source.VolumeRadius);
            this.UVScroll.Set(source.UVScroll);

            this.TexRes.Set(source.TexRes);
            this.SubTexRes.Set(source.SubTexRes);

            this.GravityVolume = source.GravityVolume;
            this.AirRegist = source.AirRegist;
            this.MeshType = source.MeshType;
            this.ParticleType = source.ParticleType;
            this.StripeType = source.StripeType;
            this.BillboardPrimitiveFileSource = source.BillboardPrimitiveFileSource;
            this.BillboardZOffset = source.BillboardZOffset;
            this.BlendType = source.BlendType;
            this.EmitRate = source.EmitRate;
            this.EmitterVelDirAngle = source.EmitterVelDirAngle;
            this.YAxisDiffusionVel = source.YAxisDiffusionVel;
            this.EmitterVelocity = source.EmitterVelocity;
            this.InitPosRand = source.InitPosRand;
            this.FigureVelocity = source.FigureVelocity;
            this.IsWorldField = source.IsWorldField;
            this.DynamicsRandom = source.DynamicsRandom;
            this.LifeStep = source.LifeStep;
            this.LifeStepRnd = source.LifeStepRnd;
            this.EmitDistEnabled = source.EmitDistEnabled;
            this.EmitDistUnit = source.EmitDistUnit;
            this.EmitDistMin = source.EmitDistMin;
            this.EmitDistMax = source.EmitDistMax;
            this.EmitDistMargin = source.EmitDistMargin;
            this.StartFrame = source.StartFrame;
            this.VolumeLatitudeEnabled = source.VolumeLatitudeEnabled;
            this.VolumeLatitudeDir = source.VolumeLatitudeDir;
            this.VolumeLatitude = source.VolumeLatitude;
            this.VolumeSweepParam = source.VolumeSweepParam;
            this.VolumeSweepStart = source.VolumeSweepStart;
            this.VolumeSweepStartRandom = source.VolumeSweepStartRandom;
            this.VolumeSurfacePosRand = source.VolumeSurfacePosRand;
            this.VolumeHollow = source.VolumeHollow;
            this.VolumeTblIndex = source.VolumeTblIndex;
            this.Volume64TblIndex = source.Volume64TblIndex;
            this.LineCenter = source.LineCenter;
            this.VolumeType = source.VolumeType;
            this.ZBufATestType = source.ZBufATestType;
            this.DisplaySide = source.DisplaySide;

            /*
             *  Cafe拡張用
             */
            this.UseSoftParticle = source.UseSoftParticle;
            this.SoftParticleFadeDistance = source.SoftParticleFadeDistance;
            this.SoftParticleVolumeParam = source.SoftParticleVolumeParam;
            this.ShaderType = source.ShaderType;
            this.ShaderSetting = source.ShaderSetting;
            this.ShaderParam0 = source.ShaderParam0;
            this.ShaderParam1 = source.ShaderParam1;
            this.UI_ApplyAlphaToRefraction = source.UI_ApplyAlphaToRefraction;

            this.UserShaderFlags = source.UserShaderFlags;
            this.UserShaderSwitch = source.UserShaderSwitch;
            for ( int i = 0; i < 32; ++i )
            {
                this.UserShaderParams[i] = source.UserShaderParams[i];
            }

            for ( int i=0;i<this.UserShaderParamGroups.Length;++i )
            {
                this.UserShaderParamGroups[i].Set( source.UserShaderParamGroups[i] );
            }

            this.UserShaderCompileDef1 = source.UserShaderCompileDef1;
            this.UserShaderCompileDef2 = source.UserShaderCompileDef2;

            this.ChildFlag.Set(source.ChildFlag);
            this.FieldFlag.Set(source.FieldFlag);
            this.FluctuationFlag.Set(source.FluctuationFlag);

            this.StripeFlag = source.StripeFlag;

            this.TexPatData.Set(source.TexPatData);
            this.SubTexPatData.Set(source.SubTexPatData);

            this.AnimEditData.Set(source.AnimEditData);
            this.ChildFlag.Set(source.ChildFlag);
            this.ChildAssistData.Set(source.ChildAssistData);
            this.ChildData.Set(source.ChildData);
            this.ChildTexPatData.Set(source.ChildTexPatData);
            this.FieldCollisionData.Set(source.FieldCollisionData);
            this.FieldConvergenceData.Set(source.FieldConvergenceData);
            this.FieldMagnetData.Set(source.FieldMagnetData);
            this.FieldPosAddData.Set(source.FieldPosAddData);
            this.FieldRandomData.Set(source.FieldRandomData);
            this.VolumeRadius.Set(source.VolumeRadius);
            this.StripeData.Set(source.StripeData);
            this.FluctuationData.Set(source.FluctuationData);
            this.FieldSpinData.Set(source.FieldSpinData);
            this.ParticleSortType = source.ParticleSortType;

            this.UVShiftAnimMode      = source.UVShiftAnimMode;
            this.IsUVShiftAnimComplex = source.IsUVShiftAnimComplex;
            this.UVRot                = source.UVRot;
            this.UVRotInit            = source.UVRotInit;
            this.UVRotInitRand        = source.UVRotInitRand;

            this.UVScroll.Set( source.UVScroll );
            this.UVScrollInit.Set( source.UVScrollInit );
            this.UVScrollInitRand.Set( source.UVScrollInitRand );
            this.UVScale.Set( source.UVScale );
            this.UVScaleInit.Set( source.UVScaleInit );
            this.UVScaleInitRand.Set( source.UVScaleInitRand );

            this.SubTexUVShiftAnimMode = source.SubTexUVShiftAnimMode;
            this.SubTexIsUVShiftAnimComplex = source.SubTexIsUVShiftAnimComplex;
            this.SubTexUVRot = source.SubTexUVRot;
            this.SubTexUVRotInit = source.SubTexUVRotInit;
            this.SubTexUVRotInitRand = source.SubTexUVRotInitRand;

            this.SubTexUVScroll.Set(source.SubTexUVScroll);
            this.SubTexUVScrollInit.Set(source.SubTexUVScrollInit);
            this.SubTexUVScrollInitRand.Set(source.SubTexUVScrollInitRand);
            this.SubTexUVScale.Set(source.SubTexUVScale);
            this.SubTexUVScaleInit.Set(source.SubTexUVScaleInit);
            this.SubTexUVScaleInitRand.Set(source.SubTexUVScaleInitRand);

            this.UserData.Set(source.UserData);

            this.UserCallbackID = source.UserCallbackID;
            for ( int i=0;i<this.UserDataGroups.Length;++i )
            {
                this.UserDataGroups[i].Set( source.UserDataGroups[i] );
            }

            this.AnimTableList.Set(source.AnimTableList);

            this.EmitterColor0.Set( source.EmitterColor0 );
            this.EmitterColor1.Set( source.EmitterColor1 );

            this.EmitterAlpha = source.EmitterAlpha;

            this.EmitterFormScale.Set( source.EmitterFormScale );
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override ComplexEmitterData Clone()
        {
            return new ComplexEmitterData(this);
        }

        #endregion

        #region Properties - SimpleEmitter

        /// <summary>
        /// エミッタ・タイプ
        /// </summary>
        public EmitterType Type { get; set; }

        /// <summary>
        /// アドレス解決後の名前
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// User callback ID ( -1, 0-7 )
        /// </summary>
        public int UserCallbackID { get; set; }

        /// <summary>
        /// ユーザーデータ
        /// </summary>
        public EmitterUserData[] UserDataGroups { get; set; }

        /// <summary>
        /// ユーザーデータ
        /// </summary>
        public UserData UserData { get; set; }

        /// <summary>
        /// テクスチャリソース
        /// </summary>
        public TextureRes TexRes { get; set; }

        /// <summary>
        /// サブ・テクスチャリソース
        /// </summary>
        public TextureRes SubTexRes { get; set; }

        /// <summary>
        /// パーティクルを描画する
        /// </summary>
        public bool DispParticle { get; set; }

        /// <summary>
        /// ワールド座標系でフィールドをかけるか？
        /// </summary>
        public bool IsWorldField { get; set; }

        /// <summary>
        ///  テーブルを使うボリューム形状のインデックス
        /// </summary>
        //TODO:ＧＵＩ接続されていない
        public int VolumeTblIndex { get; set; }

        /// <summary>
        ///  テーブルを使うボリューム形状のインデックス
        /// </summary>
        public int Volume64TblIndex { get; set; }

        /// <summary>
        /// ボリュームのタイプ
        /// </summary>
        public VolumeType VolumeType { get; set; }

        /// <summary>
        /// ボリューム半径
        /// </summary>
        public Vector3 VolumeRadius { get; set; }

        /// <summary>
        /// 中心位置
        /// </summary>
        public float LineCenter { get; set; }

        /// <summary>
        /// 弧の開き方
        /// </summary>
        public bool VolumeLatitudeEnabled { get; set; }

        /// <summary>
        /// 軸
        /// </summary>
        public VolumeLatitudeDirection VolumeLatitudeDir { get; set; }

        /// <summary>
        /// 弧の幅（角度）
        /// </summary>
        public float VolumeLatitude { get; set; }

        /// <summary>
        /// 弧の幅（開始）
        /// </summary>
        public uint VolumeSweepStart { get; set; }

        /// <summary>
        /// 弧の開始角度ランダム
        /// </summary>
        public bool VolumeSweepStartRandom { get; set; }

        /// <summary>
        /// 円（等分割）のランダム位置
        /// </summary>
        public float VolumeSurfacePosRand { get; set; }

        /// <summary>
        /// 弧の計算に使う値（ボリューム形状によって異なります）
        /// </summary>
        public uint VolumeSweepParam { get; set; }

        /// <summary>
        /// 空洞率
        /// </summary>
        public int VolumeHollow { get; set; }

        /// <summary>
        /// ランダム位置
        /// </summary>
        public float InitPosRand { get; set; }

        /// <summary>
        /// エミッタ速度
        /// </summary>
        public float FigureVelocity { get; set; }

        /// <summary>
        /// エミッタ速度の方向
        /// </summary>
        public Vector3 EmitterVelDir { get; set; }

        /// <summary>
        /// 指定方向拡散角度
        /// </summary>
        public float EmitterVelDirAngle { get; set; }

        /// <summary>
        /// Ｙ軸拡散速度
        /// </summary>
        public float YAxisDiffusionVel { get; set; }

        /// <summary>
        /// エミッタ速度
        /// </summary>
        public float EmitterVelocity { get; set; }

        /// <summary>
        /// 拡散ベクトル
        /// </summary>
        public Vector3 SpreadVec { get; set; }

        /// <summary>
        /// 放出開始フレーム
        /// </summary>
        public int StartFrame { get; set; }

        /// <summary>
        /// 運動量ランダム
        /// </summary>
        public float DynamicsRandom { get; set; }

        /// <summary>
        /// ステップ値
        /// </summary>
        public int LifeStep { get; set; }

        /// <summary>
        /// ステップ値のランダム
        /// </summary>
        public int LifeStepRnd { get; set; }

        /// <summary>
        /// 放出レート（数）
        /// </summary>
        public float EmitRate { get; set; }

        /// <summary>
        /// 放出間隔タイプ
        /// </summary>
        public bool EmitDistEnabled { get; set; }

        /// <summary>
        /// 放出間隔
        /// </summary>
        public float EmitDistUnit { get; set; }

        /// <summary>
        /// １フレームあたりの最小放出個数。
        /// </summary>
        public float EmitDistMin { get; set; }

        /// <summary>
        /// １フレームあたりの最大放出個数。
        /// </summary>
        public float EmitDistMax { get; set; }

        /// <summary>
        /// 移動距離切り捨ての閾値。
        /// </summary>
        public float EmitDistMargin { get; set; }

        /// <summary>
        /// 空気抵抗
        /// </summary>
        public float AirRegist { get; set; }

        /// <summary>
        /// ブレンドのタイプ
        /// </summary>
        public BlendType BlendType { get; set; }

        /// <summary>
        /// Mesh type
        /// </summary>
        public MeshType MeshType { get; set; }

        /// <summary>
        /// パーティクルタイプ
        /// </summary>
        public BillboardType ParticleType { get; set; }

        /// <summary>
        /// ストライプタイプ
        /// </summary>
        public StripeType StripeType { get; set; }

        /// <summary>
        /// プリミティブファイルパス
        /// </summary>
        public string BillboardPrimitiveFileSource { get; set; }

        /// <summary>
        /// ビルボードのスクリーンZ方向へのオフセット
        /// </summary>
        public float BillboardZOffset { get; set; }

        /// <summary>
        /// Ｚバッファー、αテストタイプ
        /// </summary>
        public ZBufATestType ZBufATestType { get; set; }

        /// <summary>
        /// カリング
        /// </summary>
        public DisplaySideTypes DisplaySide { get; set; }

        /// <summary>
        /// 重量の大きさ
        /// </summary>
        public float GravityVolume { get; set; }

        /// <summary>
        /// 重力の向き
        /// </summary>
        public Vector3 GravityDir { get; set; }

        /// <summary>
        /// シェーダータイプ：ソフトパーティクルを利用する
        /// </summary>
        public bool UseSoftParticle { get; set; }

        /// <summary>
        /// Fade distance for soft particle.
        /// </summary>
        public float SoftParticleFadeDistance { get; set; }

        /// <summary>
        /// Volume parameter for the soft particle
        /// </summary>
        public float SoftParticleVolumeParam { get; set; }

        /// <summary>
        /// シェーダータイプ
        /// </summary>
        public ShaderType ShaderType { get; set; }

        /// <summary>
        /// シェーダカスタマイズのセッティング
        /// </summary>
        public uint ShaderSetting { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam0 { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam1 { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public Vector2 ShaderParams01
        {
            get
            {
                return new Vector2(ShaderParam0, ShaderParam1);
            }
            set
            {
                ShaderParam0 = value.X;
                ShaderParam1 = value.Y;
            }
        }

        /// <summary>
        /// アルファ値を反映する。
        /// </summary>
        public bool UI_ApplyAlphaToRefraction { get; set; }

        /// <summary>
        /// ユーザーシェーダパラメータ
        /// </summary>
        public UserShaderParameter[] UserShaderParamGroups { get; set; }

        /// <summary>
        /// ユーザーシェーダスイッチ
        /// </summary>
        public uint UserShaderSwitch { get; set; }

        #region Accessing each byte of UserShaderSwitch

        /// <summary>Get or set the byte of the user shader switch.</summary>
        public uint UserShaderSwitch0
        {
            get { return (UserShaderSwitch & ((uint)0x000000FF)); }
            set
            {
                uint iMask = (uint)0x000000FF;
                UserShaderSwitch = (UserShaderSwitch & ~iMask) | (value & iMask);
            }
        }

        /// <summary>Get or set the byte of the user shader switch.</summary>
        public uint UserShaderSwitch1
        {
            get { return (UserShaderSwitch & ((uint)0x0000FF00)); }
            set
            {
                uint iMask = (uint)0x0000FF00;
                UserShaderSwitch = (UserShaderSwitch & ~iMask) | (value & iMask);
            }
        }

        /// <summary>Get or set the byte of the user shader switch.</summary>
        public uint UserShaderSwitch2
        {
            get { return (UserShaderSwitch & ((uint)0x00FF0000)); }
            set
            {
                uint iMask = (uint)0x00FF0000;
                UserShaderSwitch = (UserShaderSwitch & ~iMask) | (value & iMask);
            }
        }

        /// <summary>Get or set the byte of the user shader switch.</summary>
        public uint UserShaderSwitch3
        {
            get { return (UserShaderSwitch & ((uint)0xFF000000)); }
            set
            {
                uint iMask = (uint)0xFF000000;
                UserShaderSwitch = (UserShaderSwitch & ~iMask) | (value & iMask);
            }
        }

        #endregion

        /// <summary>
        /// ユーザーシェーダフラグ
        /// </summary>
        public uint UserShaderFlags { get; set; }

        #region Accessing each bit of UserShaderFlags

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag0
        {
            get { return ((UserShaderFlags & ((uint)1 << 0)) != 0); }
            set
            {
                uint iMask = (uint)1 << 0;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag1
        {
            get { return ((UserShaderFlags & ((uint)1 << 1)) != 0); }
            set
            {
                uint iMask = (uint)1 << 1;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag2
        {
            get { return ((UserShaderFlags & ((uint)1 << 2)) != 0); }
            set
            {
                uint iMask = (uint)1 << 2;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag3
        {
            get { return ((UserShaderFlags & ((uint)1 << 3)) != 0); }
            set
            {
                uint iMask = (uint)1 << 3;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag4
        {
            get { return ((UserShaderFlags & ((uint)1 << 4)) != 0); }
            set
            {
                uint iMask = (uint)1 << 4;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag5
        {
            get { return ((UserShaderFlags & ((uint)1 << 5)) != 0); }
            set
            {
                uint iMask = (uint)1 << 5;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag6
        {
            get { return ((UserShaderFlags & ((uint)1 << 6)) != 0); }
            set
            {
                uint iMask = (uint)1 << 6;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag7
        {
            get { return ((UserShaderFlags & ((uint)1 << 7)) != 0); }
            set
            {
                uint iMask = (uint)1 << 7;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag8
        {
            get { return ((UserShaderFlags & ((uint)1 << 8)) != 0); }
            set
            {
                uint iMask = (uint)1 << 8;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag9
        {
            get { return ((UserShaderFlags & ((uint)1 << 9)) != 0); }
            set
            {
                uint iMask = (uint)1 << 9;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag10
        {
            get { return ((UserShaderFlags & ((uint)1 << 10)) != 0); }
            set
            {
                uint iMask = (uint)1 << 10;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag11
        {
            get { return ((UserShaderFlags & ((uint)1 << 11)) != 0); }
            set
            {
                uint iMask = (uint)1 << 11;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag12
        {
            get { return ((UserShaderFlags & ((uint)1 << 12)) != 0); }
            set
            {
                uint iMask = (uint)1 << 12;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag13
        {
            get { return ((UserShaderFlags & ((uint)1 << 13)) != 0); }
            set
            {
                uint iMask = (uint)1 << 13;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag14
        {
            get { return ((UserShaderFlags & ((uint)1 << 14)) != 0); }
            set
            {
                uint iMask = (uint)1 << 14;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag15
        {
            get { return ((UserShaderFlags & ((uint)1 << 15)) != 0); }
            set
            {
                uint iMask = (uint)1 << 15;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag16
        {
            get { return ((UserShaderFlags & ((uint)1 << 16)) != 0); }
            set
            {
                uint iMask = (uint)1 << 16;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag17
        {
            get { return ((UserShaderFlags & ((uint)1 << 17)) != 0); }
            set
            {
                uint iMask = (uint)1 << 17;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag18
        {
            get { return ((UserShaderFlags & ((uint)1 << 18)) != 0); }
            set
            {
                uint iMask = (uint)1 << 18;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag19
        {
            get { return ((UserShaderFlags & ((uint)1 << 19)) != 0); }
            set
            {
                uint iMask = (uint)1 << 19;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag20
        {
            get { return ((UserShaderFlags & ((uint)1 << 20)) != 0); }
            set
            {
                uint iMask = (uint)1 << 20;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag21
        {
            get { return ((UserShaderFlags & ((uint)1 << 21)) != 0); }
            set
            {
                uint iMask = (uint)1 << 21;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag22
        {
            get { return ((UserShaderFlags & ((uint)1 << 22)) != 0); }
            set
            {
                uint iMask = (uint)1 << 22;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag23
        {
            get { return ((UserShaderFlags & ((uint)1 << 23)) != 0); }
            set
            {
                uint iMask = (uint)1 << 23;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag24
        {
            get { return ((UserShaderFlags & ((uint)1 << 24)) != 0); }
            set
            {
                uint iMask = (uint)1 << 24;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag25
        {
            get { return ((UserShaderFlags & ((uint)1 << 25)) != 0); }
            set
            {
                uint iMask = (uint)1 << 25;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag26
        {
            get { return ((UserShaderFlags & ((uint)1 << 26)) != 0); }
            set
            {
                uint iMask = (uint)1 << 26;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag27
        {
            get { return ((UserShaderFlags & ((uint)1 << 27)) != 0); }
            set
            {
                uint iMask = (uint)1 << 27;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag28
        {
            get { return ((UserShaderFlags & ((uint)1 << 28)) != 0); }
            set
            {
                uint iMask = (uint)1 << 28;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag29
        {
            get { return ((UserShaderFlags & ((uint)1 << 29)) != 0); }
            set
            {
                uint iMask = (uint)1 << 29;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag30
        {
            get { return ((UserShaderFlags & ((uint)1 << 30)) != 0); }
            set
            {
                uint iMask = (uint)1 << 30;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        /// <summary>Get or set the flag bit of the user shader flags.</summary>
        public bool UserShaderFlag31
        {
            get { return ((UserShaderFlags & ((uint)1 << 31)) != 0); }
            set
            {
                uint iMask = (uint)1 << 31;
                UserShaderFlags = (value ? (UserShaderFlags | iMask) : (UserShaderFlags & ~iMask));
            }
        }

        #endregion

        /// <summary>
        /// ユーザーシェーダパラメータ
        /// </summary>
        public float[] UserShaderParams { get; set; }

        /// <summary>
        /// ユーザーシェーダコンパイル定義１
        /// </summary>
        public string UserShaderCompileDef1 { get; set; }

        /// <summary>
        /// ユーザーシェーダコンパイル定義２
        /// </summary>
        public string UserShaderCompileDef2 { get; set; }


        /// <summary>
        /// ＵＶアニメーション
        /// </summary>
        public int UVShiftAnimMode { get; set; }


        /// <summary>
        /// ＵＶアニメーション詳細設定
        /// </summary>
        public bool IsUVShiftAnimComplex { get; set; }


        /// <summary>
        /// ＵＶスクロール加算値
        /// </summary>
        public Vector2 UVScroll { get; set; }


        /// <summary>
        /// ＵＶスクロール初期値
        /// </summary>
        public Vector2 UVScrollInit { get; set; }


        /// <summary>
        /// ＵＶスクロール　ランダム
        /// </summary>
        public Vector2 UVScrollInitRand { get; set; }


        /// <summary>
        /// ＵＶスケール加算値
        /// </summary>
        public Vector2 UVScale { get; set; }


        /// <summary>
        /// ＵＶスケール初期値
        /// </summary>
        public Vector2 UVScaleInit { get; set; }


        /// <summary>
        /// ＵＶスケール　ランダム
        /// </summary>
        public Vector2 UVScaleInitRand { get; set; }


        /// <summary>
        /// ＵＶ回転加算値
        /// </summary>
        public float UVRot { get; set; }


        /// <summary>
        /// ＵＶ回転初期値
        /// </summary>
        public float UVRotInit { get; set; }


        /// <summary>
        /// ＵＶ回転　ランダム
        /// </summary>
        public float UVRotInitRand { get; set; }

        /// <summary>
        /// ＵＶアニメーション
        /// </summary>
        public int SubTexUVShiftAnimMode { get; set; }


        /// <summary>
        /// ＵＶアニメーション詳細設定
        /// </summary>
        public bool SubTexIsUVShiftAnimComplex { get; set; }


        /// <summary>
        /// ＵＶスクロール加算値
        /// </summary>
        public Vector2 SubTexUVScroll { get; set; }


        /// <summary>
        /// ＵＶスクロール初期値
        /// </summary>
        public Vector2 SubTexUVScrollInit { get; set; }


        /// <summary>
        /// ＵＶスクロール　ランダム
        /// </summary>
        public Vector2 SubTexUVScrollInitRand { get; set; }


        /// <summary>
        /// ＵＶスケール加算値
        /// </summary>
        public Vector2 SubTexUVScale { get; set; }


        /// <summary>
        /// ＵＶスケール初期値
        /// </summary>
        public Vector2 SubTexUVScaleInit { get; set; }


        /// <summary>
        /// ＵＶスケール　ランダム
        /// </summary>
        public Vector2 SubTexUVScaleInitRand { get; set; }


        /// <summary>
        /// ＵＶ回転加算値
        /// </summary>
        public float SubTexUVRot { get; set; }


        /// <summary>
        /// ＵＶ回転初期値
        /// </summary>
        public float SubTexUVRotInit { get; set; }


        /// <summary>
        /// ＵＶ回転　ランダム
        /// </summary>
        public float SubTexUVRotInitRand { get; set; }


        /// <summary>
        /// The list of animation tables.
        /// </summary>
        public EmitterAnimTableList AnimTableList { get; set; }


        /// <summary>
        /// グローバル乗算カラー０
        /// </summary>
        public RgbaColor EmitterColor0 { get; set; }


        /// <summary>
        /// グローバル乗算カラー１
        /// </summary>
        public RgbaColor EmitterColor1 { get; set; }


        /// <summary>
        /// グローバル乗算アルファ
        /// </summary>
        public float EmitterAlpha { get; set; }


        /// <summary>
        /// エミッタ形状スケール
        /// </summary>
        public Vector3 EmitterFormScale { get; set; }

        #endregion

        #region Properties - ComplexEmitter

        /// <summary>
        /// エミッタ：テクスチャ用
        /// </summary>
        public TexPatData TexPatData { get; set; }

        /// <summary>
        /// エミッタ：サブ・テクスチャ用
        /// </summary>
        public TexPatData SubTexPatData { get; set; }

        /// <summary>
        /// チャイルド：テクスチャ用
        /// </summary>
        public TexPatData ChildTexPatData { get; set; }

        /// <summary>
        /// チャイルド用フラグ
        /// </summary>
        public ChildFlag ChildFlag { get; set; }

        /// <summary>
        /// フィールドを使用するか？
        /// </summary>
        public FieldMask FieldFlag { get; set; }

        /// <summary>
        /// 揺らぎ用フラグ
        /// </summary>
        public FluctuationFlag FluctuationFlag { get; set; }

        /// <summary>
        /// ストライプ用フラグ
        /// </summary>
        public int StripeFlag
        {
            get
            {
                if ( this.StripeData==null )
                    return 0;

                return ( this.StripeData.FollowEmitter==true ? (int)NWCore.DataModel.Major_1.Minor_4.Build_0.Revision_0.StripeFlag.EmitterCoord : 0 );
            }
            set
            {
                if ( this.StripeData==null )
                    return;

                this.StripeData.FollowEmitter = ( (value & (int)NWCore.DataModel.Major_1.Minor_4.Build_0.Revision_0.StripeFlag.EmitterCoord)!=0 );
            }
        }

        /// <summary>
        ///
        /// </summary>
        public AnimEditData AnimEditData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public ChildAssistData ChildAssistData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public ChildData ChildData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldRandomData FieldRandomData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldMagnetData FieldMagnetData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldSpinData FieldSpinData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldCollisionData FieldCollisionData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldConvergenceData FieldConvergenceData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldPosAddData FieldPosAddData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FluctuationData FluctuationData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public StripeData StripeData { get; set; }

        /// <summary>
        /// ソートオーダー
        /// </summary>
        public ParticleSortType ParticleSortType { get; set; }

        #endregion
    }
}
