﻿// ========================================================================
// <copyright file="PreviewModelData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using NintendoWare.ToolDevelopmentKit;
using NWCore.DataModel.Major_1.Minor_3.Build_0.Revision_0;
using NWCore.Serializer.Major_1.Minor_3.Build_0.Revision_0;

namespace NWCore.DataModel.Major_1.Minor_3.Build_0.Revision_0
{
    /// <summary>
    /// プレビュー・モデル設定データ
    /// </summary>
    [Serializable]
    public sealed class PreviewModelData : BaseDataModel<PreviewModelData>
    {
        List<string> m_boneNameList = new List<string>();

        #region Constructors

        const int iMaxNumCharAnimations = 4;
        const int iMaxNumMatAnimations = 2;
        const int iMaxNumVisibilityAnimations = 2;

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public PreviewModelData()
        {
            Initialize();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public PreviewModelData(PreviewModelData source)
            : this()
        {
            Initialize();
            this.Set(source);
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.DisplayModel = true;
            this.ModelName = String.Empty;
            this.ModelFilePath = String.Empty;
            this.m_boneNameList.Clear();

            UserFunctionList = new List<UserFunctionXML>();

            int i;

            this.CharacterAnimations = new string[iMaxNumCharAnimations];
            for (i = 0; i < iMaxNumCharAnimations; ++i)
            {
                this.CharacterAnimations[i] = string.Empty;
            }

            this.MaterialAnimations = new string[iMaxNumMatAnimations];
            for (i = 0; i < iMaxNumMatAnimations; ++i)
            {
                this.MaterialAnimations[i] = string.Empty;
            }

            this.VisibilityAnimations = new string[iMaxNumVisibilityAnimations];
            for (i = 0; i < iMaxNumVisibilityAnimations; ++i)
            {
                this.VisibilityAnimations[i] = string.Empty;
            }
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(PreviewModelData source)
        {
            Ensure.Argument.NotNull(source);

            this.DisplayModel = source.DisplayModel;
            this.ModelName = string.Copy(source.ModelName);
            this.ModelFilePath = string.Copy(source.ModelFilePath);
            SetBoneNameList(source.GetBoneNameList());

            this.UserFunctionList.Clear();
            foreach (UserFunctionXML src in source.UserFunctionList)
            {
                UserFunctionXML dst = new UserFunctionXML(src.UserFuncName, src.UserFuncID);
                dst.Copy(src);
                this.UserFunctionList.Add(dst);
            }

            int i;
            for (i = 0; i < iMaxNumCharAnimations; ++i)
            {
                this.CharacterAnimations[i] =
                    string.Copy(source.CharacterAnimations[i]);
            }

            for (i = 0; i < iMaxNumMatAnimations; ++i)
            {
                this.MaterialAnimations[i] =
                    string.Copy(source.MaterialAnimations[i]);
            }

            for (i = 0; i < iMaxNumVisibilityAnimations; ++i)
            {
                this.VisibilityAnimations[i] =
                    string.Copy(source.VisibilityAnimations[i]);
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override PreviewModelData Clone()
        {
            return new PreviewModelData(this);
        }

        /// <summary>
        /// ボーン名を更新します。
        /// </summary>
        /// <param name="boneNames"></param>
        public void SetBoneNameList(string[] boneNames)
        {
            m_boneNameList.Clear();
            if (boneNames != null)
            {
                m_boneNameList.AddRange(boneNames);
            }
        }

        /// <summary>
        /// ボーン名リストを取得します。
        /// </summary>
        public string[] GetBoneNameList()
        {
            return m_boneNameList.ToArray();
        }

        /// <summary>
        /// ボーン数
        /// </summary>
        public int BoneCount
        {
            get { return m_boneNameList.Count; }
        }

        /// <summary>
        /// ボーン名を取得
        /// </summary>
        public string GetBoneName(int index)
        {
            return (m_boneNameList.Count > index ? m_boneNameList[index] : String.Empty);
        }

        /// <summary>
        /// ボーン名からインデックス取得
        /// </summary>
        /// <returns>見つからないときは-１</returns>
        public int GetBoneIndex(string boneName)
        {
            for (int i = 0; i < m_boneNameList.Count; ++i)
            {
                if (m_boneNameList[i] == boneName)
                {
                    return i;
                }
            }
            return -1;
        }

        #endregion

        #region Properties

        /// <summary>
        /// 表示
        /// </summary>
        public bool DisplayModel { get; set; }

        /// <summary>
        /// モデル
        /// </summary>
        public string ModelName { get; set; }

        /// <summary>
        /// モデルファイルパス
        /// </summary>
        public string ModelFilePath { get; set; }

        /// <summary>
        /// キャラクタアニメ
        /// </summary>
        public string[] CharacterAnimations { get; set; }

        /// <summary>
        /// マテリヤルアニメ
        /// </summary>
        public string[] MaterialAnimations { get; set; }

        /// <summary>
        /// ビジビリティアニメ
        /// </summary>
        public string[] VisibilityAnimations { get; set; }

        /// <summary>
        /// The list of the user functions.
        /// </summary>
        [XmlArrayItem("UserFunction")]
        public List<UserFunctionXML> UserFunctionList { get; set; }

        #endregion
    }
}
