﻿// ========================================================================
// <copyright file="ComplexEmitterData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_0.Minor_9.Build_0.Revision_0
{
    /// <summary>
    /// シンプルエミッタデータ
    /// </summary>
    public class ComplexEmitterData : BaseDataModel<ComplexEmitterData>
    {
        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public ComplexEmitterData()
        {
            this.EmitterVelDir = new Vector3();     // エミッタ速度の方向
            this.GravityDir = new Vector3();        // 重力の向き
            this.SpreadVec = new Vector3();         // 拡散ベクトル
            this.VolumeRadius = new Vector3();      // ボリューム半径
            this.UVScroll = new Vector2();          // ＵＶスクロール

            this.TexRes = new TextureRes();
            this.SubTexRes = new TextureRes();

            this.FluctuationFlag = new FluctuationFlag();

            this.FieldFlag = new FieldMask();
            this.FieldFlag.Flag = (uint)0;

            this.TexPatData = new TexPatData();
            this.SubTexPatData = new TexPatData();

            this.AnimEditData = new AnimEditData();
            this.ChildFlag = new ChildFlag();
            this.ChildAssistData = new ChildAssistData();
            this.ChildData = new ChildData();
            this.ChildTexPatData = new TexPatData();
            this.EmitterVelDir = new Vector3();
            this.FieldCollisionData = new FieldCollisionData();
            this.FieldConvergenceData = new FieldConvergenceData();
            this.FieldMagnetData = new FieldMagnetData();
            this.FieldPosAddData = new FieldPosAddData();
            this.FieldRandomData = new FieldRandomData();
            this.VolumeRadius = new Vector3();
            this.StripeData = new StripeData();
            this.FluctuationData = new FluctuationData();
            this.FieldSpinData = new FieldSpinData();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public ComplexEmitterData(ComplexEmitterData source)
            : this()
        {
            this.FluctuationFlag = new FluctuationFlag();
            this.FieldFlag = new FieldMask();

            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.VolumeType = VolumeType.Point;

            this.EmitRate = 1;
            this.VolumeSweepParam = (uint)0xffffffff;
            this.VolumeSweepStart = 0;
            this.VolumeTblIndex = 0;

            this.GravityVolume = 0;
            this.GravityDir.Set(0, -1, 0);

            this.FluctuationFlag.ApplyAlpha = true;
            this.FluctuationFlag.ApplyScale = true;

            this.ChildFlag.DispParent = true;
            this.ChildFlag.EmitterFollow = true;
            this.ChildFlag.AlphaInherit = true;
            this.ChildFlag.ScaleInherit = true;
            this.ChildFlag.RotateInherit = true;

            this.FieldFlag.Flag = (uint)0;
            this.StripeFlag = 0;

            this.TexPatData.Initialize();
            this.SubTexPatData.Initialize();
            this.AnimEditData.Initialize();
            this.ChildAssistData.Initialize();
            this.ChildData.Initialize();
            this.ChildTexPatData.Initialize();
            this.FieldMagnetData.Initialize();
            this.FieldRandomData.Initialize();
            this.FluctuationData.Initialize();
            this.FieldSpinData.Initialize();
            this.StripeData.Initialize();

            this.AirRegist = 1.0f;
            this.LifeStep = 10;
            this.LineCenter = 0.0f;

            this.EmitterVelDir.Set(0, 1, 0);

            float worldScale = s_WorldScaleRange / 100.0f;
            {
                this.FigureVelocity = 0.1f * 1.0f * worldScale; // *velScale;
                this.VolumeRadius.Set(1.0f * worldScale, 1.0f * worldScale, 1.0f * worldScale); // *posScale;
            }

            //             this.ChildDataOffset = 9;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(ComplexEmitterData source)
        {
            Ensure.Argument.NotNull(source);

            // SimpleEmitterData
            /*
             *  SimpleEmitterData用
             */
            this.Type = source.Type;
            this.Name = source.Name;

            this.EmitterVelDir.Set(source.EmitterVelDir);
            this.GravityDir.Set(source.GravityDir);
            this.SpreadVec.Set(source.SpreadVec);
            this.VolumeRadius.Set(source.VolumeRadius);
            this.UVScroll.Set(source.UVScroll);

            this.TexRes.Set(source.TexRes);
            this.SubTexRes.Set(source.SubTexRes);

            this.GravityVolume = source.GravityVolume;
            this.AirRegist = source.AirRegist;
            this.BillboardType = source.BillboardType;
            this.BillboardZOffset = source.BillboardZOffset;
            this.BlendType = source.BlendType;
            this.EmitRate = source.EmitRate;
            this.EmitterVelocity = source.EmitterVelocity;
            this.FigureVelocity = source.FigureVelocity;
            this.IsWorldField = source.IsWorldField;
            this.LifeStep = source.LifeStep;
            this.LifeStepRnd = source.LifeStepRnd;
            this.StartFrame = source.StartFrame;
            this.VolumeSweepParam = source.VolumeSweepParam;
            this.VolumeSweepStart = source.VolumeSweepStart;
            this.VolumeTblIndex = source.VolumeTblIndex;
            this.LineCenter = source.LineCenter;
            this.VolumeType = source.VolumeType;
            this.ZBufATestType = source.ZBufATestType;

            /*
             *  Cafe拡張用
             */
            this.UseSortParticle = source.UseSortParticle;
            this.ShaderParam0 = source.ShaderParam0;
            this.ShaderParam1 = source.ShaderParam1;


            this.ChildFlag.Set(source.ChildFlag);
            this.FieldFlag.Set(source.FieldFlag);
            this.FluctuationFlag.Set(source.FluctuationFlag);

            this.StripeFlag = source.StripeFlag;

            this.TexPatData.Set(source.TexPatData);
            this.SubTexPatData.Set(source.SubTexPatData);

            this.AnimEditData.Set(source.AnimEditData);
            this.ChildFlag.Set(source.ChildFlag);
            this.ChildAssistData.Set(source.ChildAssistData);
            this.ChildData.Set(source.ChildData);
            this.ChildTexPatData.Set(source.ChildTexPatData);
            this.EmitterVelDir.Set(source.EmitterVelDir);
            this.FieldCollisionData.Set(source.FieldCollisionData);
            this.FieldConvergenceData.Set(source.FieldConvergenceData);
            this.FieldMagnetData.Set(source.FieldMagnetData);
            this.FieldPosAddData.Set(source.FieldPosAddData);
            this.FieldRandomData.Set(source.FieldRandomData);
            this.VolumeRadius.Set(source.VolumeRadius);
            this.StripeData.Set(source.StripeData);
            this.FluctuationData.Set(source.FluctuationData);
            this.FieldSpinData.Set(source.FieldSpinData);
            this.SortOrder = source.SortOrder;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override ComplexEmitterData Clone()
        {
            return new ComplexEmitterData(this);
        }

        #endregion

        #region Properties - SimpleEmitter

        /// <summary>
        /// エミッタ・タイプ
        /// </summary>
        public EmitterType Type { get; set; }

        /// <summary>
        /// アドレス解決後の名前
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// テクスチャリソース
        /// </summary>
        public TextureRes TexRes { get; set; }

        /// <summary>
        /// サブ・テクスチャリソース
        /// </summary>
        public TextureRes SubTexRes { get; set; }

        /// <summary>
        /// ワールド座標系でフィールドをかけるか？
        /// </summary>
        public bool IsWorldField { get; set; }

        /// <summary>
        ///  テーブルを使うボリューム形状のインデックス
        /// </summary>
        //TODO:ＧＵＩ接続されていない
        public int VolumeTblIndex { get; set; }

        /// <summary>
        /// ボリュームのタイプ
        /// </summary>
        public VolumeType VolumeType { get; set; }

        /// <summary>
        /// ボリューム半径
        /// </summary>
        public Vector3 VolumeRadius { get; set; }

        /// <summary>
        /// 中心位置
        /// </summary>
        public float LineCenter { get; set; }

        /// <summary>
        /// 弧の幅（開始）
        /// </summary>
        public uint VolumeSweepStart { get; set; }

        /// <summary>
        /// 弧の計算に使う値（ボリューム形状によって異なります）
        /// </summary>
        public uint VolumeSweepParam { get; set; }

        /// <summary>
        /// エミッタ速度
        /// </summary>
        public float FigureVelocity { get; set; }

        /// <summary>
        /// エミッタ速度の方向
        /// </summary>
        public Vector3 EmitterVelDir { get; set; }

        /// <summary>
        /// エミッタ速度
        /// </summary>
        public float EmitterVelocity { get; set; }

        /// <summary>
        /// 拡散ベクトル
        /// </summary>
        public Vector3 SpreadVec { get; set; }

        /// <summary>
        /// 放出開始フレーム
        /// </summary>
        public int StartFrame { get; set; }

        /// <summary>
        /// ステップ値
        /// </summary>
        public int LifeStep { get; set; }

        /// <summary>
        /// ステップ値のランダム
        /// </summary>
        public int LifeStepRnd { get; set; }

        /// <summary>
        /// 放出レート（数）
        /// </summary>
        public int EmitRate { get; set; }

        /// <summary>
        /// 空気抵抗
        /// </summary>
        public float AirRegist { get; set; }

        /// <summary>
        /// ブレンドのタイプ
        /// </summary>
        public BlendType BlendType { get; set; }

        /// <summary>
        /// ビルボードのタイプ
        /// </summary>
        public BillboardType BillboardType { get; set; }

        /// <summary>
        /// ビルボードのスクリーンZ方向へのオフセット
        /// </summary>
        public float BillboardZOffset { get; set; }

        /// <summary>
        /// Ｚバッファー、αテストタイプ
        /// </summary>
        public ZBufATestType ZBufATestType { get; set; }

        /// <summary>
        /// 重量の大きさ
        /// </summary>
        public float GravityVolume { get; set; }

        /// <summary>
        /// 重力の向き
        /// </summary>
        public Vector3 GravityDir { get; set; }

        /// <summary>
        /// シェーダータイプ：ソフトパーティクルを利用する
        /// </summary>
        public ShaderType UseSortParticle { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam0 { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam1 { get; set; }


        /// <summary>
        /// ＵＶスクロール
        /// </summary>
        public Vector2 UVScroll { get; set; }

        #endregion

        #region Properties - ComplexEmitter

        /// <summary>
        /// エミッタ：テクスチャ用
        /// </summary>
        public TexPatData TexPatData { get; set; }

        /// <summary>
        /// エミッタ：サブ・テクスチャ用
        /// </summary>
        public TexPatData SubTexPatData { get; set; }

        /// <summary>
        /// チャイルド：テクスチャ用
        /// </summary>
        public TexPatData ChildTexPatData { get; set; }

        /// <summary>
        /// チャイルド用フラグ
        /// </summary>
        public ChildFlag ChildFlag { get; set; }

        /// <summary>
        /// フィールドを使用するか？
        /// </summary>
        public FieldMask FieldFlag { get; set; }

        /// <summary>
        /// 揺らぎ用フラグ
        /// </summary>
        public FluctuationFlag FluctuationFlag { get; set; }

        /// <summary>
        /// ストライプ用フラグ
        /// </summary>
        public StripeFlag StripeFlag { get; set; }

        /// <summary>
        ///
        /// </summary>
        public AnimEditData AnimEditData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public ChildAssistData ChildAssistData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public ChildData ChildData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldRandomData FieldRandomData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldMagnetData FieldMagnetData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldSpinData FieldSpinData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldCollisionData FieldCollisionData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldConvergenceData FieldConvergenceData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldPosAddData FieldPosAddData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FluctuationData FluctuationData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public StripeData StripeData { get; set; }

        /// <summary>
        /// ソートオーダー
        /// </summary>
        public int SortOrder { get; set; }

        #endregion


    }
}
