﻿// ========================================================================
// <copyright file="PreviewConfigData.cs" company="Nintendo">
//      Copyright 2011 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using NintendoWare.ToolDevelopmentKit;
using NWCore.DataModel;

namespace NWCore.DataModel.Major_0.Minor_0.Build_0.Revision_0
{
    /// <summary>
    /// プレビュー設定データ
    /// </summary>
    public sealed class PreviewConfigData : BaseDataModel<PreviewConfigData>
    {
        #region Constructors

        /// <summary>
        /// プレビューモデルの最大数
        /// </summary>
        public const int MaxPreviewModelCount = 10;

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public PreviewConfigData()
        {
            Initialize();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public PreviewConfigData(PreviewConfigData source)
            : this()
        {
            this.Set(source);
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.InfinitePreview = false;
            this.MaxPreviewFrame = 180;
            this.AutoTranslateType = AutoTranslateType.None;
            this.TranslateRadiusXZ = 8.0f;
            this.TranslateVelXZ = 1.0f;
            this.VibrationAmplitudeY = 0.0f;
            this.VibrationVelY = 5.0f;
            this.ZRollVel = 0.0f;
            this.CameraHorizontalDegree = 336.0f;
            this.CameraVerticalDegree = -20.0f;
            this.CameraHorizontalVel = 0.0f;
            this.CameraRotateRadius = 22.0f;
            this.CameraVerticalViewAngle = 90.0f;

            this.BackgroundImageFileName = "";
            this.BackgroundImageVertexColor = new RgbaColor(1.0f, 1.0f, 1.0f, 1.0f);

            this.LightDir = new Vector3(-0.4f, -0.8f, -0.4f);
            this.LightDiffuse = new Vector3(1.0f, 1.0f, 1.0f);
            this.LightAmbient = new Vector3(0.2f, 0.2f, 0.2f);
            this.LightSpecular0 = new Vector3(1.0f, 1.0f, 1.0f);
            this.LightSpecular1 = new Vector3(1.0f, 1.0f, 1.0f);
            this.HLightDir = new Vector3(0.0f, 1.0f, 0.0f);
            this.HLightSkyColor = new Vector3(0.9f, 1.0f, 1.0f);
            this.HLightGroundColor = new Vector3(0.1f, 0.0f, 0.0f);

            this.SkyColorWeightXZ = 0.5f;
            this.FogType = FogType.None;
            this.FogColor = new Vector3(1.0f, 1.0f, 1.0f);
            this.FogBeginDistance = 0.0f;
            this.FogEndDistance = 200.0f;
            this.FogDensity = 2.0f;

            this.PreviewModels = new PreviewModelData[MaxPreviewModelCount];
            for (int i = 0; i < MaxPreviewModelCount; ++i)
            {
                this.PreviewModels[i] = new PreviewModelData();
                this.PreviewModels[i].Initialize();
            }
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source"></param>
        public override void Set(PreviewConfigData source)
        {
            Ensure.Argument.NotNull(source);

            this.InfinitePreview = source.InfinitePreview;
            this.MaxPreviewFrame = source.MaxPreviewFrame;
            this.AutoTranslateType = source.AutoTranslateType;
            this.TranslateRadiusXZ = source.TranslateRadiusXZ;
            this.TranslateVelXZ = source.TranslateVelXZ;
            this.VibrationAmplitudeY = source.VibrationAmplitudeY;
            this.VibrationVelY = source.VibrationVelY;
            this.ZRollVel = source.ZRollVel;
            this.CameraHorizontalDegree = source.CameraHorizontalDegree;
            this.CameraVerticalDegree = source.CameraVerticalDegree;
            this.CameraHorizontalVel = source.CameraHorizontalVel;
            this.CameraRotateRadius = source.CameraRotateRadius;
            this.CameraVerticalViewAngle = source.CameraVerticalViewAngle;

            this.BackgroundImageFileName = source.BackgroundImageFileName;
            this.BackgroundImageVertexColor.Set(source.BackgroundImageVertexColor);

            this.LightDir.Set(source.LightDir);
            this.LightDiffuse.Set(source.LightDiffuse);
            this.LightAmbient.Set(source.LightAmbient);
            this.LightSpecular0.Set(source.LightSpecular0);
            this.LightSpecular1.Set(source.LightSpecular1);
            this.HLightDir.Set(source.HLightDir);
            this.HLightSkyColor.Set(source.HLightSkyColor);
            this.HLightGroundColor.Set(HLightGroundColor);

            this.SkyColorWeightXZ = source.SkyColorWeightXZ;
            this.FogType = source.FogType;
            this.FogColor.Set(source.FogColor);
            this.FogBeginDistance = source.FogBeginDistance;
            this.FogEndDistance = source.FogEndDistance;
            this.FogDensity = source.FogDensity;

            for (int i = 0; i < MaxPreviewModelCount; ++i)
            {
                this.PreviewModels[i].Set(source.PreviewModels[i]);
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override PreviewConfigData Clone()
        {
            return new PreviewConfigData(this);
        }

        #endregion

        #region Properties

        /// <summary>
        /// 無限プレビュー
        /// </summary>
        public bool InfinitePreview { get; set; }

        /// <summary>
        /// プレビュー最大フレーム
        /// </summary>
        public int MaxPreviewFrame { get; set; }

        /// <summary>
        /// 児童移動タイプ
        /// </summary>
        public AutoTranslateType AutoTranslateType { get; set; }

        /// <summary>
        /// XZ移動半径
        /// </summary>
        public float TranslateRadiusXZ { get; set; }

        /// <summary>
        /// XZ移動速度
        /// </summary>
        public float TranslateVelXZ { get; set; }

        /// <summary>
        /// Y揺れ振幅
        /// </summary>
        public float VibrationAmplitudeY { get; set; }

        /// <summary>
        /// Y揺れ速度
        /// </summary>
        public float VibrationVelY { get; set; }

        /// <summary>
        /// Zロール速度
        /// </summary>
        public float ZRollVel { get; set; }

        /// <summary>
        /// 横回転角度
        /// </summary>
        public float CameraHorizontalDegree { get; set; }

        /// <summary>
        /// 縦回転角度
        /// </summary>
        public float CameraVerticalDegree { get; set; }

        /// <summary>
        /// 横回転角速度
        /// </summary>
        public float CameraHorizontalVel { get; set; }

        /// <summary>
        /// 回転半径
        /// </summary>
        public float CameraRotateRadius { get; set; }

        /// <summary>
        /// 垂直画角
        /// </summary>
        public float CameraVerticalViewAngle { get; set; }

        /// <summary>
        /// 背景画像
        /// </summary>
        public string BackgroundImageFileName { get; set; }

        /// <summary>
        /// 背景画像頂点カラー
        /// </summary>
        public RgbaColor BackgroundImageVertexColor { get; set; }

        /// <summary>
        /// ライト方向
        /// </summary>
        public Vector3 LightDir { get; set; }

        /// <summary>
        /// ライトディフューズカラー
        /// </summary>
        public Vector3 LightDiffuse { get; set; }

        /// <summary>
        /// アンビエントカラー
        /// </summary>
        public Vector3 LightAmbient { get; set; }

        /// <summary>
        /// スペキュラカラー0
        /// </summary>
        public Vector3 LightSpecular0 { get; set; }

        /// <summary>
        /// スペキュラカラー1
        /// </summary>
        public Vector3 LightSpecular1 { get; set; }

        /// <summary>
        /// 半球ライト方向
        /// </summary>
        public Vector3 HLightDir { get; set; }

        /// <summary>
        /// 半球ライトスカイカラー
        /// </summary>
        public Vector3 HLightSkyColor { get; set; }

        /// <summary>
        /// 半球ライトグラウンドカラー
        /// </summary>
        public Vector3 HLightGroundColor { get; set; }

        /// <summary>
        /// XZスカイカラーウェイト
        /// </summary>
        public float SkyColorWeightXZ { get; set; }

        /// <summary>
        /// フォグタイプ
        /// </summary>
        public FogType FogType { get; set; }

        /// <summary>
        /// フォグカラー
        /// </summary>
        public Vector3 FogColor { get; set; }

        /// <summary>
        /// フォグ開始距離
        /// </summary>
        public float FogBeginDistance { get; set; }

        /// <summary>
        /// フォグ終了距離
        /// </summary>
        public float FogEndDistance { get; set; }

        /// <summary>
        /// フォグ濃度
        /// </summary>
        public float FogDensity { get; set; }

        /// <summary>
        /// プレビューモデル
        /// </summary>
        public PreviewModelData[] PreviewModels { get; set; }

        #endregion
    }

    /// <summary>
    /// プレビュー・モデル設定データ
    /// </summary>
    public sealed class PreviewModelData : BaseDataModel<PreviewModelData>
    {
        List<string> m_boneNameList = new List<string>();

        #region Constructors

        const int iMaxNumCharAnimations       = 4;
        const int iMaxNumMatAnimations        = 2;
        const int iMaxNumVisibilityAnimations = 2;

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public PreviewModelData()
        {
            Initialize();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public PreviewModelData(PreviewModelData source)
            : this()
        {
            Initialize();
            this.Set(source);
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.DisplayModel = true;
            this.ModelName = String.Empty;
            this.ModelFilePath = String.Empty;
            this.m_boneNameList.Clear();

            int i;

            this.CharacterAnimations = new string[iMaxNumCharAnimations];
            for ( i=0;i<iMaxNumCharAnimations;++i )
            {
                this.CharacterAnimations[i] = string.Empty;
            }

            this.MaterialAnimations = new string[iMaxNumMatAnimations];
            for ( i=0;i<iMaxNumMatAnimations;++i )
            {
                this.MaterialAnimations[i] = string.Empty;
            }

            this.VisibilityAnimations = new string[iMaxNumVisibilityAnimations];
            for ( i=0;i<iMaxNumVisibilityAnimations;++i )
            {
                this.VisibilityAnimations[i] = string.Empty;
            }
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(PreviewModelData source)
        {
            Ensure.Argument.NotNull(source);

            this.DisplayModel = source.DisplayModel;
            this.ModelName = string.Copy(source.ModelName);
            this.ModelFilePath = string.Copy(source.ModelFilePath);
            SetBoneNameList(source.GetBoneNameList());

            int i;
            for ( i=0;i<iMaxNumCharAnimations;++i )
            {
                this.CharacterAnimations[i] =
                    string.Copy( source.CharacterAnimations[i] );
            }

            for ( i=0;i<iMaxNumMatAnimations;++i )
            {
                this.MaterialAnimations[i] =
                    string.Copy( source.MaterialAnimations[i] );
            }

            for ( i=0;i<iMaxNumVisibilityAnimations;++i )
            {
                this.VisibilityAnimations[i] =
                    string.Copy( source.VisibilityAnimations[i] );
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override PreviewModelData Clone()
        {
            return new PreviewModelData(this);
        }

        /// <summary>
        /// ボーン名を更新します。
        /// </summary>
        /// <param name="boneNames"></param>
        public void SetBoneNameList(string[] boneNames)
        {
            m_boneNameList.Clear();
            if (boneNames != null)
            {
                m_boneNameList.AddRange(boneNames);
            }
        }

        /// <summary>
        /// ボーン名リストを取得します。
        /// </summary>
        /// <returns></returns>
        public string[] GetBoneNameList()
        {
            return m_boneNameList.ToArray();
        }

        /// <summary>
        /// ボーン名を取得
        /// </summary>
        public string GetBoneName(int index)
        {
            return (m_boneNameList.Count > index ? m_boneNameList[index] : String.Empty);
        }

        /// <summary>
        /// ボーン名からインデックス取得
        /// </summary>
        /// <returns>見つからないときは-１</returns>
        public int GetBoneIndex(string boneName)
        {
            for (int i = 0; i < m_boneNameList.Count; ++i)
            {
                if (m_boneNameList[i] == boneName)
                {
                    return i;
                }
            }
            return -1;
        }

        #endregion

        #region Properties

        /// <summary>
        /// 表示
        /// </summary>
        public bool DisplayModel { get; set; }

        /// <summary>
        /// モデル
        /// </summary>
        public string ModelName { get; set; }

        /// <summary>
        /// モデルファイルパス
        /// </summary>
        public string ModelFilePath { get; set; }

        /// <summary>
        /// キャラクタアニメ
        /// </summary>
        public string[] CharacterAnimations { get; set; }

        /// <summary>
        /// マテリヤルアニメ
        /// </summary>
        public string[] MaterialAnimations { get; set; }

        /// <summary>
        /// ビジビリティアニメ
        /// </summary>
        public string[] VisibilityAnimations { get; set; }

        #endregion
    }
}
