﻿// ========================================================================
// <copyright file="RectangleUtility.cs" company="Nintendo">
//      Copyright 2009 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using NintendoWare.ToolDevelopmentKit;

namespace App.Utility
{
    /// <summary>
    /// レクタングルユーティリティクラス。
    /// </summary>
    public static class RectangleUtility
    {
        /// <summary>
        /// 左位置を設定。
        /// </summary>
        public static void SetLeft(ref Rectangle rect, int value)
        {
            if (value > rect.Left) { rect.Width -= value - rect.Left; }
            else { rect.Width += rect.Left - value; }
            rect.X = value;
        }

        /// <summary>
        /// 左位置を設定。
        /// </summary>
        public static void SetLeft(ref RectangleF rect, float value)
        {
            if (value > rect.Left) { rect.Width -= value - rect.Left; }
            else { rect.Width += rect.Left - value; }
            rect.X = value;
        }

        /// <summary>
        /// 上位置を設定。
        /// </summary>
        public static void SetTop(ref Rectangle rect, int value)
        {
            if (value > rect.Top) { rect.Height -= value - rect.Top; }
            else { rect.Height += rect.Top - value; }
            rect.Y = value;
        }

        /// <summary>
        /// 上位置を設定。
        /// </summary>
        public static void SetTop(ref RectangleF rect, float value)
        {
            if (value > rect.Top) { rect.Height -= value - rect.Top; }
            else { rect.Height += rect.Top - value; }
            rect.Y = value;
        }

        /// <summary>
        /// 右位置を設定。
        /// </summary>
        public static void SetRight(ref Rectangle rect, int value)
        {
            if (value > rect.Right) { rect.Width += value - rect.Right; }
            else { rect.Width -= rect.Right - value; }
        }

        /// <summary>
        /// 右位置を設定。
        /// </summary>
        public static void SetRight(ref RectangleF rect, float value)
        {
            if (value > rect.Right) { rect.Width += value - rect.Right; }
            else { rect.Width -= rect.Right - value; }
        }

        /// <summary>
        /// 下位置を設定。
        /// </summary>
        public static void SetBottom(ref Rectangle rect, int value)
        {
            if (value > rect.Bottom) { rect.Height += value - rect.Bottom; }
            else { rect.Height -= rect.Bottom - value; }
        }

        /// <summary>
        /// 下位置を設定。
        /// </summary>
        public static void SetBottom(ref RectangleF rect, float value)
        {
            if (value > rect.Bottom) { rect.Height += value - rect.Bottom; }
            else { rect.Height -= rect.Bottom - value; }
        }

        /// <summary>
        /// 左位置を移動。
        /// </summary>
        public static void OffsetLeft(ref Rectangle rect, int delta)
        {
            rect.X += delta;
            rect.Width -= delta;
        }

        /// <summary>
        /// 左位置を移動。
        /// </summary>
        public static void OffsetLeft(ref RectangleF rect, float delta)
        {
            rect.X += delta;
            rect.Width -= delta;
        }

        /// <summary>
        /// 上位置を移動。
        /// </summary>
        public static void OffsetTop(ref Rectangle rect, int delta)
        {
            rect.Y += delta;
            rect.Height -= delta;
        }

        /// <summary>
        /// 上位置を移動。
        /// </summary>
        public static void OffsetTop(ref RectangleF rect, float delta)
        {
            rect.Y += delta;
            rect.Height -= delta;
        }

        /// <summary>
        /// 右位置を移動。
        /// </summary>
        public static void OffsetRight(ref Rectangle rect, int delta)
        {
            rect.Width += delta;
        }

        /// <summary>
        /// 右位置を移動。
        /// </summary>
        public static void OffsetRight(ref RectangleF rect, float delta)
        {
            rect.Width += delta;
        }

        /// <summary>
        /// 下位置を移動。
        /// </summary>
        public static void OffsetBottom(ref Rectangle rect, int delta)
        {
            rect.Height += delta;
        }

        /// <summary>
        /// 下位置を移動。
        /// </summary>
        public static void OffsetBottom(ref RectangleF rect, float delta)
        {
            rect.Height += delta;
        }

        /// <summary>
        /// 中心位置を取得。
        /// </summary>
        public static Point GetCenter(Rectangle rect)
        {
            return new Point(
                (rect.Left + rect.Right) / 2,
                (rect.Top + rect.Bottom) / 2
            );
        }

        /// <summary>
        /// 中心位置を取得。
        /// </summary>
        public static PointF GetCenter(RectangleF rect)
        {
            return new PointF(
                (rect.Left + rect.Right) / 2.0f,
                (rect.Top + rect.Bottom) / 2.0f
            );
        }

        /// <summary>
        /// 左上位置を取得。
        /// </summary>
        public static Point GetTopLeft(Rectangle rect)
        {
            return new Point(rect.Left, rect.Top);
        }

        /// <summary>
        /// 左上位置を取得。
        /// </summary>
        public static PointF GetTopLeft(RectangleF rect)
        {
            return new PointF(rect.Left, rect.Top);
        }

        /// <summary>
        /// 右上位置を取得。
        /// </summary>
        public static Point GetTopRight(Rectangle rect)
        {
            return new Point(rect.Right, rect.Top);
        }

        /// <summary>
        /// 右上位置を取得。
        /// </summary>
        public static PointF GetTopRight(RectangleF rect)
        {
            return new PointF(rect.Right, rect.Top);
        }

        /// <summary>
        /// 左下位置を取得。
        /// </summary>
        public static Point GetBottomLeft(Rectangle rect)
        {
            return new Point(rect.Left, rect.Bottom);
        }

        /// <summary>
        /// 左下位置を取得。
        /// </summary>
        public static PointF GetBottomLeft(RectangleF rect)
        {
            return new PointF(rect.Left, rect.Bottom);
        }

        /// <summary>
        /// 右下位置を取得。
        /// </summary>
        public static Point GetBottomRight(Rectangle rect)
        {
            return new Point(rect.Right, rect.Bottom);
        }

        /// <summary>
        /// 右下位置を取得。
        /// </summary>
        public static PointF GetBottomRight(RectangleF rect)
        {
            return new PointF(rect.Right, rect.Bottom);
        }

        /// <summary>
        /// ２点から作成。
        /// </summary>
        public static Rectangle FromPoints(Point pt1, Point pt2)
        {
            return new Rectangle(
                Math.Min(pt1.X, pt2.X),
                Math.Min(pt1.Y, pt2.Y),
                Math.Abs(pt2.X - pt1.X),
                Math.Abs(pt2.Y - pt1.Y)
            );
        }

        /// <summary>
        /// ２点から作成。
        /// </summary>
        public static RectangleF FromPoints(PointF pt1, PointF pt2)
        {
            return new RectangleF(
                Math.Min(pt1.X, pt2.X),
                Math.Min(pt1.Y, pt2.Y),
                Math.Abs(pt2.X - pt1.X),
                Math.Abs(pt2.Y - pt1.Y)
            );
        }

        /// <summary>
        /// RECT型に変換。
        /// </summary>
        public static Win32.RECT ToRECT(Rectangle rect)
        {
            Win32.RECT result = new Win32.RECT();
            result.left = rect.Left;
            result.top = rect.Top;
            result.right = rect.Right;
            result.bottom = rect.Bottom;
            return result;
        }
    }
}
