﻿/*---------------------------------------------------------------------------*
  Project:  NintendoWare
  File:     eft_StreamOut.vsh

  Copyright (C)2011-2013 Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
*---------------------------------------------------------------------------*/

// 本シェーダコード内では、#ifdef _USE_NN_VFX しか利用できません。

//---------------------------------------------------
// GPUコリジョン調整（仮）
//---------------------------------------------------
#ifndef USE_USR_ADJUSTMENT_GPU_COLLISION
vec3 AdjustmentGpuCollision( vec3 srcVec, vec4 projection, float pz, float tz )
{
    return srcVec;
}
#endif

//---------------------------------------------------
// nn::vfx 専用: OldPos 修正用のカスタム処理
//---------------------------------------------------
#ifndef USE_USR_CUSTOM_OLD_POS_MOD
#ifdef _USE_NN_VFX
void AdjustmentStreamOutOldPos( inout vec3 outOldPos, inout vec3 outPos, inout vec3 outVec, inout float outParticleLife )
{
    return;
}
#endif
#endif

//---------------------------------------------------
// カスタムフィールド
//---------------------------------------------------
#ifndef USE_USR_CUSTOM_FIELD
#ifdef _USE_NN_VFX
void AdjustmentStreamOutPosVec( inout vec3 outPos, inout vec3 outVec, inout float outParticleLife )
#else
void AdjustmentStreamOutPosVec()
#endif
{
    return;
}
#endif

//------------------------------------------------------------------------------
// フィールド：ランダム（旧バージョン）
//------------------------------------------------------------------------------
vec3 CalcOldRandomField( vec3 inVec, float time, float rate )
{
    vec3 ret = vec3( 0, 0,0 );

    #define fieldRandomVecX         rnf0.x          // ランダム速度X
    #define fieldRandomVecY         rnf0.y          // ランダム速度Y
    #define fieldRandomVecZ         rnf0.z          // ランダム速度Z
    #define fieldRandomApplyBlank   rnf0.w          // ランダム適用間隔

    float _2pi = 3.141592 * 2.0f;
    if ( ( int( time ) % int( fieldRandomApplyBlank ) ) == 0 )
    {
        vec3 randomVel = rnf0.xyz;
        float sqTime = time * time;
        ret.x = ( sin( sysPtclRandomZ * sqTime + ( sysPtclRandomZ - 0.5f ) * _2pi ) ) * fieldRandomVecX;
        ret.y = ( sin( sysPtclRandomY * sqTime + ( sysPtclRandomX - 0.5f ) * _2pi ) ) * fieldRandomVecY;
        ret.z = ( sin( sysPtclRandomX * sqTime + ( sysPtclRandomY - 0.5f ) * _2pi ) ) * fieldRandomVecZ;
    }

    return ret;
}

//------------------------------------------------------------------------------
// フィールド：ランダム
//------------------------------------------------------------------------------
vec3 CalcRandomField()
{
    #define airRegist               ubVec.x             // 空気抵抗
    float invRate = 1.0 - sysFrameRate;
    float aRegist = airRegist + (1.0f - airRegist) * invRate;

    // 空気抵抗値を加味した時間位置
    float velTime = sysPtclTime;
    if( rnd3.w == 1 )
    {
        velTime = ( aRegist == 1.0 ) ? sysPtclTime : ( 1 - pow( aRegist, sysPtclTime ) )/( 1 - aRegist );
    }

    vec3 randomAdd = vec3( 0, 0, 0 );   // ランダムによって動く変位
    vec3 randomVel = rnd0.xyz;   // ランダム適用量

    float RandCycle = rnd0.w;    // 乱数の基本周期のフレーム数

    // 位相を統一
    vec4 seed; //乱数の種（初期位相）
    if( rnd3.x == 1 )
    {
        seed.x = 0;
        seed.y = 0.31415f;    // x,y,z=0の時のために適当にずらす
        seed.z = 0.92653f;    // x,y,z=0の時のために適当にずらす
        seed.w = 0.58979f;    // x,y,z=0の時のために適当にずらす
    }
    else
    {
        seed.x = sysPtclRandomX;
        seed.y = sysPtclRandomY;
        seed.z = sysPtclRandomZ;
        seed.w = sysPtclRandomW;
    }

    seed *= RandCycle;

    // 位相を統一
    float speed = ( rnd4.x / 100.0f ) * RandCycle;   // 位相変化速度
    if( rnd3.x == 1 )
    {
        float randWidth = ( rnd4.y / 100.0f ) * RandCycle;   // 位相のばらつき
        seed.x += ( sysEmitterTime * speed ) + randWidth * ((sysPtclRandomX - 0.5f) * 2 );
        seed.y += ( sysEmitterTime * speed ) + randWidth * ((sysPtclRandomY - 0.5f) * 2 );
        seed.z += ( sysEmitterTime * speed ) + randWidth * ((sysPtclRandomZ - 0.5f) * 2 );
        seed.w += ( sysEmitterTime * speed ) + randWidth * ((sysPtclRandomW - 0.5f) * 2 );
    }

    vec3 randomValue = vec3( 0, 0, 0 );         // 乱数（これまでの積算値）
    float deltaVelTime = sysFrameRate * 1.0f;

    float deltaSeed = 0;
    if( rnd3.x == 1 )
    {
        deltaSeed = sysFrameRate * speed;
    }

    // 乱数の積算値（更新差分）を取得
    randomValue.x = GetRand( seed.x, velTime, deltaSeed, deltaVelTime );
    randomValue.y = GetRand( seed.y, velTime, deltaSeed, deltaVelTime );
    randomValue.z = GetRand( seed.z, velTime, deltaSeed, deltaVelTime );

    // 変位量を求める
    randomAdd.x = randomValue.x * randomVel.x;
    randomAdd.y = randomValue.y * randomVel.y;
    randomAdd.z = randomValue.z * randomVel.z;

    return randomAdd;

    #undef airRegist
}

//------------------------------------------------------------------------------
// フィールド：位置に加算
//------------------------------------------------------------------------------
vec3 CalcAddPosField( float rate )
{
    vec3  fieldPosAddVel = pAdd.xyz;        // 加算値

    int   isWorld = int( pAdd.w );          // ワールド座標で適用

    vec3 add = fieldPosAddVel * sysPtclDynamicsRand * rate;
    if ( isWorld == 1 ) { add = ApplyEmitterRotateInvMatrix_StreamOut( add ); }
    return add;
}

//------------------------------------------------------------------------------
// フィールド：スピン
//------------------------------------------------------------------------------
vec3 CalcSpinField( vec3 inPos, float rate )
{
    vec3 ret = inPos;

    #define fieldSpinRotRad         spin.x            // フィールドスピン：スピン回転値
    #define fieldSpinOuter          spin.y            // フィールドスピン：スピン拡散速度
    #define fieldSpinAxis           spin.z            // フィールドスピン：スピン軸

    int axis = int( fieldSpinAxis );

    if ( axis == 0 )
    {
        float sinV = sin( fieldSpinRotRad * sysPtclDynamicsRand * rate );
        float cosV = cos( fieldSpinRotRad * sysPtclDynamicsRand * rate );

        float v0 = ret.y *  cosV  + ret.z * sinV;
        float v1 = ret.y * -sinV  + ret.z * cosV;
        ret.y = v0;
        ret.z = v1;

        // 拡散速度
        if( fieldSpinOuter != 0.0 )
        {
            float length = v0 * v0 + v1 * v1;
            if( length > 0.0 )
            {
                float r = 1.0f / sqrt( length ) * fieldSpinOuter * sysPtclDynamicsRand * rate;
                ret.y += ( v0 * r );
                ret.z += ( v1 * r );
            }
        }
    }
    if ( axis == 1 )
    {
        float sinV = sin( fieldSpinRotRad * sysPtclDynamicsRand * rate );
        float cosV = cos( fieldSpinRotRad * sysPtclDynamicsRand * rate );

        float v0 = ret.z *  cosV  + ret.x * sinV;
        float v1 = ret.z * -sinV  + ret.x * cosV;
        ret.z = v0;
        ret.x = v1;

        // 拡散速度
        if( fieldSpinOuter != 0.0 )
        {
            float length = v0 * v0 + v1 * v1;
            if( length > 0.0 )
            {
                float r = 1.0f / sqrt( length ) * fieldSpinOuter * sysPtclDynamicsRand * rate;
                ret.z += ( v0 * r );
                ret.x += ( v1 * r );
            }
        }
    }
    if ( axis == 2 )
    {
        float sinV = sin( fieldSpinRotRad * sysPtclDynamicsRand * rate );
        float cosV = cos( fieldSpinRotRad * sysPtclDynamicsRand * rate );

        float v0 = ret.x *  cosV  + ret.y * sinV;
        float v1 = ret.x * -sinV  + ret.y * cosV;
        ret.x = v0;
        ret.y = v1;

        // 拡散速度
        if( fieldSpinOuter != 0.0 )
        {
            float length = v0 * v0 + v1 * v1;
            if( length > 0.0 )
            {
                float r = 1.0f / sqrt( length ) * fieldSpinOuter * sysPtclDynamicsRand * rate;
                ret.x += ( v0 * r );
                ret.y += ( v1 * r );
            }
        }
    }

    return ret;
}

//------------------------------------------------------------------------------
// フィールド：磁力
//------------------------------------------------------------------------------
vec3 CalcMagnetField( vec3 inPos, vec3 inVec, float rate )
{
    #define fieldMagnetPos          mgn0.xyz       // 磁力位置
    #define fieldMagnetRatio        mgn0.w         // 磁力
    #define fieldMagnetCrdEmt       mgn1.x         // エミッタ追従
    #define fieldMagnetFlagX        mgn1.y         // 対象座標軸X
    #define fieldMagnetFlagY        mgn1.z         // 対象座標軸Y
    #define fieldMagnetFlagZ        mgn1.w         // 対象座標軸Z

    // エミッタに追従
    int crdEmitter = int( fieldMagnetCrdEmt );

    // エミッタローカル
    vec3 emtLocal = vec3( 0 );

    if ( crdEmitter != 0 )
    {
        // TODO:エミッタローカル値取得
        emtLocal = GetEmitterLocalPosForSO();
    }

    // 軸ごとに設定
    vec3 ret = vec3( 0 );
    if ( fieldMagnetFlagX != 0 )
    {
        ret.x = ( emtLocal.x + fieldMagnetPos.x - inPos.x - inVec.x ) * fieldMagnetRatio * sysPtclDynamicsRand * rate;
    }
    if ( fieldMagnetFlagY != 0 )
    {
        ret.y = ( emtLocal.y + fieldMagnetPos.y - inPos.y - inVec.y ) * fieldMagnetRatio * sysPtclDynamicsRand * rate;
    }
    if ( fieldMagnetFlagZ != 0 )
    {
        ret.z = ( emtLocal.z + fieldMagnetPos.z - inPos.z - inVec.z ) * fieldMagnetRatio * sysPtclDynamicsRand * rate;
    }
    return ret;
}

//------------------------------------------------------------------------------
// フィールド：収束
//------------------------------------------------------------------------------
vec3 CalcConvergenceField( vec3 inPos, float rate )
{
    vec3 ret = inPos;

    #define fieldConvergencePos     cnv0.xyz   // 収束位置（ローカル座標系）
    #define fieldConvergenceRatio   cnv0.w     // 収束力
    #define fieldConvergenceCrdEmt  cnv1.x     // エミッタ追従

    // エミッタに追従
    int crdEmitter = int( fieldConvergenceCrdEmt );

    // エミッタのローカル座標
    vec3 emtLocal = vec3( 0, 0, 0 );

    if ( crdEmitter > 0 )
    {
        // パーティクルが生成された時のエミッタ位置から、
        // 現在のエミッタ位置までの差分が必要（追従させるため）
        emtLocal = GetEmitterLocalPosForSO();
    }

    return ( emtLocal + fieldConvergencePos - inPos ) * fieldConvergenceRatio * sysPtclDynamicsRand * rate;

    return ret;
}

//------------------------------------------------------------------------------
// カールノイズ
//------------------------------------------------------------------------------
vec3 CalcCurlNoiseField( vec3 inPos, vec3 inVec, float rate )
{
    vec3 ret = inVec;
    vec3 tex_coord;

    #define fieldCurlNoisePowX      cur0.x           // パワーX
    #define fieldCurlNoisePowY      cur0.y           // パワーY
    #define fieldCurlNoisePowZ      cur0.z           // パワーZ
    #define fieldCurlNoiseScale     cur0.w           // スケール

    #define fieldCurlNoiseSpeedX    cur1.x           // スピードX
    #define fieldCurlNoiseSpeedY    cur1.y           // スピードY
    #define fieldCurlNoiseSpeedZ    cur1.z           // スピードZ
    #define fieldCurlNoiseBase      cur1.w           // オフセット

    #define fieldCurlNoiseBaseRandom cur2.x          // ノイズのオフセットランダム
    #define fieldCurlNoiseIsWorldCoord cur2.y        // ワールド座標系で適用

    vec3 ptclPos;
    // パーティクル中心世界座標
    if( fieldCurlNoiseIsWorldCoord > 0 )
    {
        ptclPos = TransformToWorldPos_StreamOut( inPos );
    }
    else
    {
        ptclPos = inPos;
    }

    tex_coord.x = ( ptclPos.x * fieldCurlNoiseScale + fieldCurlNoiseSpeedX * sysEmitterTime );
    tex_coord.y = ( ptclPos.y * fieldCurlNoiseScale + fieldCurlNoiseSpeedY * sysEmitterTime );
    tex_coord.z = ( ptclPos.z * fieldCurlNoiseScale + fieldCurlNoiseSpeedZ * sysEmitterTime );
    float noiseOffset = fieldCurlNoiseBase;
    if( fieldCurlNoiseBaseRandom != 0 ) noiseOffset *= sysPtclRandomX;
    tex_coord += noiseOffset;

    if ( tex_coord.x < 0 ) tex_coord.x *= -1;
    if ( tex_coord.y < 0 ) tex_coord.y *= -1;
    if ( tex_coord.z < 0 ) tex_coord.z *= -1;
    tex_coord.xyz *= ( 1.0 / 32.0 );

    vec4 curlNoise = texture( sysCurlNoiseTextureArray, tex_coord );
    curlNoise = curlNoise * 2 - 1;
    curlNoise *= 0.5;
    curlNoise.x *= fieldCurlNoisePowX;
    curlNoise.y *= fieldCurlNoisePowY;
    curlNoise.z *= fieldCurlNoisePowZ;

    if( fieldCurlNoiseIsWorldCoord > 0 )
    {
        curlNoise.xyz = TransformToLocalVec_StreamOut( curlNoise.xyz );
    }

    ret += curlNoise.xyz * rate;

    return ret;
}

//------------------------------------------------------------------------------
// シンプルコリジョン
//------------------------------------------------------------------------------
#ifdef _USE_NN_VFX
void CalcSimpleCollisionField( float rate, inout vec3 outPos, inout vec3 outVec, inout float outParticleLife )
#else
void CalcSimpleCollisionField( float rate )
#endif
{
    #define COLLISION_TYPE_REFLECT_LOCAL    0
    #define COLLISION_TYPE_REFLECT_WORLD    1
    #define COLLISION_TYPE_KILL_LOCAL       2
    #define COLLISION_TYPE_KILL_WORLD       3
    #define COLLISION_TYPE_CUSTOM_LOCAL     4
    #define COLLISION_TYPE_CUSTOM_WORLD     5

#ifndef _USE_NN_VFX
    vec3 outPos = sysOutPos.xyz;
    vec3 outVec = sysOutVec.xyz;
#endif

    // パーティクル中心世界座標
    vec3 ptclWorldPos = ApplyEmitterMatrix_StreamOut( outPos );

    #define fieldCollisionType      coll.x       // タイプ（ワールド座標系の可否も含む）
    #define fieldCollisionCoef      coll.y       // 反射率
    #define fieldCollisionRegist    coll.z       // 摩擦
    #define fieldCollisionCoord     coll.w       // 反射座標

    int ColType = int( fieldCollisionType );

    if ( ColType == COLLISION_TYPE_REFLECT_LOCAL && outPos.y < fieldCollisionCoord )
    {
        // エミッタ座標系で反射
        outPos.y  = fieldCollisionCoord;                      // 反射面まで持ち上げる
        outVec.y *= -fieldCollisionCoef;
        outVec   *= fieldCollisionRegist;
    }
    if ( ColType == COLLISION_TYPE_REFLECT_WORLD && ptclWorldPos.y < fieldCollisionCoord )
    {
        // 世界座標系で反射
        outPos.y += fieldCollisionCoord - ptclWorldPos.y;    // 反射面まで持ち上げる
        outVec.y *= -fieldCollisionCoef;
        outVec   *= fieldCollisionRegist;
    }
    if ( ColType == COLLISION_TYPE_KILL_LOCAL && outPos.y < fieldCollisionCoord )
    {
        // エミッタ座標系で消滅
#ifdef _USE_NN_VFX
        outParticleLife = 0;
#else
        sysOutVec.w = sysEmitterTime;
#endif
    }
    if ( ColType == COLLISION_TYPE_KILL_WORLD && ptclWorldPos.y < fieldCollisionCoord )
    {
        // 世界座標系で消滅
#ifdef _USE_NN_VFX
        outParticleLife = 0;
#else
        sysOutVec.w = sysEmitterTime;
#endif
    }

#ifndef _USE_NN_VFX
    sysOutPos.xyz = outPos;
    sysOutVec.xyz = outVec;
#endif

}

//------------------------------------------------------------------------------
// StreamOutシェーダ main関数
//------------------------------------------------------------------------------
void main()
{
#ifdef _USE_NN_VFX

    invocationIndex     = gl_LocalInvocationIndex + gl_WorkGroupID.x * 32;
    if( sysEmitterParticleCount <= invocationIndex )
    {
        // 使用していない領域はそもそも計算を回さないで返す。
        return;
    }

    vec3 inPos          = sysPos[ invocationIndex ].xyz;
    vec3 inVec          = sysVec[ invocationIndex ].xyz;

    sysPtclLife         = int( sysPos[ invocationIndex ].w );
    sysPtclTime         = sysEmitterTime - sysVec[ invocationIndex ].w;
    sysPtclDynamicsRand = sysScale[ invocationIndex ].w;
    sysPtclRandomX      = sysRandomAttr[ invocationIndex ].x;
    sysPtclRandomY      = sysRandomAttr[ invocationIndex ].y;
    sysPtclRandomZ      = sysRandomAttr[ invocationIndex ].z;
    sysPtclRandomW      = sysRandomAttr[ invocationIndex ].w;

    // 寿命を終えたパーティクルを計算しない
    if ( sysPtclTime > sysPtclLife )
    {
        return;
    }

    if ( sysPtclTime < 0 )
    {
        return;
    }

#else

    sysPtclTime = sysEmitterTime - sysPtclEmitTime;
    sysPtclTime = max( sysPtclTime, 0 );

    vec3 inPos  = sysInPos.xyz;
    vec3 inVec  = sysInVec.xyz;

    // 1フレーム目は、LocalXXから位置と速度をもらう
    if ( sysPtclTime <= 0.001 )
    {
        inPos = sysLocalPosAttr.xyz;
        inVec = sysLocalVecAttr.xyz;
    }

    sysPtclDynamicsRand = sysScaleAttr.w;

#endif

    vec3 outPos;
    vec3 outPosDelta;
    vec3 outVec;
    vec3 oldPos = inPos.xyz;

    outPos          = inPos + inVec * sysFrameRate;
    outVec          = inVec;

    // エミッタRTマトリクス生成
    MakeEmitterRTMatrix_StreamOut();

    // 空気抵抗
    {
        #define airRegistRaw   ubVec.x
        float airRegist = pow( airRegistRaw, sysFrameRate );
        outVec.xyz *= airRegist;
    }

    // 重力
    {
        #define gravityVec    ubGrav.xyz
        #define gravityScale  ubGrav.w

        vec3  gravityRaw = gravityVec * gravityScale;
        vec3  gravity    = gravityRaw;

        // 重力をワールド座標で適用
#ifdef _USE_NN_VFX
        if ( bitFlagWorldGrabityEnabled )
#else
        if ( CHECK_BIT_FLAG( EFT_WORLD_GRAVITY_ENABLED ) )
#endif
        {
            gravity.x = gravityRaw.x * sysEmitterRT[0].x + gravityRaw.y * sysEmitterRT[1].x + gravityRaw.z * sysEmitterRT[2].x;
            gravity.y = gravityRaw.x * sysEmitterRT[0].y + gravityRaw.y * sysEmitterRT[1].y + gravityRaw.z * sysEmitterRT[2].y;
            gravity.z = gravityRaw.x * sysEmitterRT[0].z + gravityRaw.y * sysEmitterRT[1].z + gravityRaw.z * sysEmitterRT[2].z;
        }

        outVec.xyz += ( gravity * sysFrameRate );
    }

    // フィールド
#ifdef _USE_NN_VFX
    if ( bitFlagFieldRandom )
#else
    if ( CHECK_BIT_FLAG_1( BIT_FLAG_FIELD_RANDOM ) )
#endif
    {
        // GPUノイズ（位置に合成）
        outPos.xyz += CalcRandomField();
    }
#ifdef _USE_NN_VFX
    if ( bitFlagFieldRandomFe1 )
#else
    if ( CHECK_BIT_FLAG_1( BIT_FLAG_FIELD_RANDOM_FE1 ) )
#endif
    {
        // FE1ランダム（位置に合成）
        outVec.xyz += CalcOldRandomField( outVec.xyz, sysPtclTime, sysFrameRate );
    }
#ifdef _USE_NN_VFX
    if ( bitFlagFieldPosadd )
#else
    if ( CHECK_BIT_FLAG_1( BIT_FLAG_FIELD_POSADD ) )
#endif
    {
        // 位置に加算（位置に合成）
        outPos.xyz += CalcAddPosField( sysFrameRate );
    }
#ifdef _USE_NN_VFX
    if ( bitFlagFieldMagnet )
#else
    if ( CHECK_BIT_FLAG_1( BIT_FLAG_FIELD_MAGNET ) )
#endif
    {
        // 磁力（速度に合成）
        outVec.xyz += CalcMagnetField( outPos.xyz, outVec.xyz, sysFrameRate );
    }
#ifdef _USE_NN_VFX
    if ( bitFlagFieldConvergence )
#else
    if ( CHECK_BIT_FLAG_1( BIT_FLAG_FIELD_CONVERGENCE ) )
#endif
    {
        // 収束（位置に合成）
        outPos.xyz += CalcConvergenceField( outPos.xyz, sysFrameRate );
    }
#ifdef _USE_NN_VFX
    if ( bitFlagFieldSpin )
#else
    if ( CHECK_BIT_FLAG_1( BIT_FLAG_FIELD_SPIN ) )
#endif
    {
        // スピン（位置を更新）
        outPos.xyz = CalcSpinField( outPos.xyz, sysFrameRate );
    }
#ifdef _USE_NN_VFX
    if ( bitFlagFieldCurlnoise )
#else
    if ( CHECK_BIT_FLAG_1( BIT_FLAG_FIELD_CURLNOISE ) )
#endif
    {
        // カールノイズ（速度を更新）
        outVec.xyz = CalcCurlNoiseField( outPos.xyz, outVec.xyz, sysFrameRate );
    }

#ifdef _USE_NN_VFX
    //------------------------------------------------------------------------------
    //  VFX
    //------------------------------------------------------------------------------
    if( bitFlagFieldCollision )
    {
        // コリジョン（速度と位置を更新）（※変則）
        CalcSimpleCollisionField( sysFrameRate, outPos, outVec, sysPos[ invocationIndex ].w );
    }

    // カスタムフィールド
    AdjustmentStreamOutPosVec( outPos, outVec, sysPos[ invocationIndex ].w );
    AdjustmentStreamOutOldPos( oldPos, outPos, outVec, sysPos[ invocationIndex ].w );

    // 最終結果を保存
    sysPos[ invocationIndex ].xyz = outPos;
    sysVec[ invocationIndex ].xyz = outVec;
    outPosDelta.xyz = sysPos[ invocationIndex ].xyz - oldPos;  // 移動差分を計算
    outPosDelta /= sysFrameRate;
#ifdef _VFX_DEPRECATED_GPU_SO_EFT2_COMPATIBLE
    sysPosDelta[ invocationIndex ].xyz = outVec * sysFrameRate;
#else
    sysPosDelta[ invocationIndex ].xyz = outPosDelta;
#endif

#else
    //------------------------------------------------------------------------------
    //  eft2
    //------------------------------------------------------------------------------
    sysOutPos.xyz   = outPos;
    sysOutPos.w     = sysInPos.w;
    sysOutVec.xyz   = outVec;

    if( CHECK_BIT_FLAG_1( BIT_FLAG_FIELD_COLLISION ) )
    {
        // コリジョン（速度と位置を更新）（※変則）
        CalcSimpleCollisionField( sysFrameRate );
    }

    // カスタムフィールド
    AdjustmentStreamOutPosVec();

    outPosDelta.xyz = sysOutPos.xyz - oldPos;   // 移動差分を計算
    sysOutVec.w = sysPtclTime;
#endif

}
