﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the ViewerCameraViewProjectionViewModel.
    /// </summary>
    public class ViewerCameraViewProjectionViewModel : PropertyGroupViewModel<ViewerCameraViewProjectionData>, IModificationFlagOwner
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public ViewerCameraViewProjectionViewModel(HierarchyViewModel parent, ViewerCameraViewProjectionData dataModel)
            : base(parent, dataModel)
        {
            this.ProjectionTypeItems = new []
            {
                new KeyValuePair<string, object>(Properties.Resources.ViewerProjectionTypePerspective, 0),
                new KeyValuePair<string, object>(Properties.Resources.ViewerProjectionTypeOrtho, 1)
            };

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);

            // カメラ情報の更新メソッドを、CameraControllerのイベントハンドラに追加。
            // このイベントハンドラは、カメラ情報に関するパケットをビューアから受信したときに実行される。
            CameraController.Instance.CameraInformationChanged += this.OnCameraViewProjectionDataChanged;
        }

        /// <summary>
        /// 射影タイプの項目を取得します。
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> ProjectionTypeItems { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// カメラ情報の更新を行います。
        /// </summary>
        /// <param name="sender">カメラ情報</param>
        /// <param name="e">event arguments</param>
        private void OnCameraViewProjectionDataChanged(object sender, Vector3f cameraPosition, Vector3f cameraTarget)
        {
            // ビューアにカメラ情報を送り返さないようにする。
            using (new MessageDropSection())
            {
                // 各プロパティごとに履歴が積まれてしまうことを防ぐために、
                // CommandCombinerクラスでコマンドを連結する。
                using (new CommandCombiner())
                {
                    this.SetDataModelValue(cameraPosition, "CameraPosition");
                    this.SetDataModelValue(cameraTarget, "CameraTarget");
                }
            }
        }
    }
}
