﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//// #define DEBUG_MODIFICATION_FLAG_VIEWMODEL

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Dynamic;
using System.Linq;
using System.Reflection;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// 変更フラグのビューモデル.
    /// </summary>
    public class ModificationFlagViewModel : ViewModelBase
    {
        /// <summary>
        /// 状態をチェックしないプロパティ名のリストです.
        /// </summary>
        protected static readonly HashSet<string> IgnorePropertyNamesStatic =
            new HashSet<string>(new string[]
            {
                "Parent",
                "IsSelected",
                "IsMultiSelected",
                "IsExpanded",
                "IsSingleColumn",
                "IsLinearMode",
                "SelectedPropertyPage",
                "Guid",
                "Displayed",
                "CanSetName",
            });

        /// <summary>
        /// 自分自身、またはチャイルドに変更があるかどうかのフラグ.
        /// </summary>
        private bool isAnyValueModified = false;

        /// <summary>
        /// 初期状態からの変更点があるかどうかのフラグ.
        /// </summary>
        private bool isSomeValuesNotDefault = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="parent">親のビューモデル.</param>
        public ModificationFlagViewModel(ViewModelBase parent) :
            base(null)
        {
            this.Parent = parent;
            this.Parent.PropertyChanged += this.OnParentPropertyChanged;

            this.IgnorePropertyNames = new HashSet<string>();
            this.ModifyFlags = new Dictionary<string, ModificationData>();
        }

        /// <summary>
        /// Get or set the flag indicating whether to ignore
        /// the property change event from parent view model.
        /// </summary>
        public static bool IgnoreParentPropertyChangedEvents { get; set; }

        /// <summary>
        /// 自分自身、またはチャイルドに変更があるかどうか取得します。
        /// </summary>
        public bool IsAnyValueModified
        {
            get { return this.isAnyValueModified; }
            private set { this.SetValue(ref this.isAnyValueModified, value); }
        }

        /// <summary>
        /// 初期状態からの変更点の有無を取得する
        /// </summary>
        public bool IsSomeValuesNotDefault
        {
            get { return this.isSomeValuesNotDefault; }
            private set { this.SetValue(ref this.isSomeValuesNotDefault, value); }
        }

        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストを取得または設定します.
        /// </summary>
        protected HashSet<string> IgnorePropertyNames { get; set; }

        /// <summary>
        /// 親のビューモデルを取得または設定します.
        /// </summary>
        protected ViewModelBase Parent { get; set; }

        /// <summary>
        /// 変更状態を格納するコンテナを取得または設定します.
        /// </summary>
        protected Dictionary<string, ModificationData> ModifyFlags { get; set; }

        /// <summary>
        /// 状態をチェックしないプロパティ名前を追加します.
        /// </summary>
        /// <param name="names">プロパティ名</param>
        public void AddIgnoreProperties(string[] names)
        {
            foreach (string name in names)
            {
                this.IgnorePropertyNames.Add(name);
            }
        }

        /// <summary>
        /// 変更フラグをクリアします。
        /// </summary>
        public virtual void ClearModificationFlags()
        {
            string[] propertyNames = null;
            if (this.ModifyFlags != null && this.ModifyFlags.Count > 0)
            {
                propertyNames = this.ModifyFlags.Keys.ToArray();
                this.ModifyFlags.Clear();
            }

            this.IsAnyValueModified = false;

            if (propertyNames != null)
            {
                propertyNames.ForEach(n => this.OnPropertyChanged(n));
            }
        }

        /// <summary>
        /// チャイルドの変更フラグをクリアします。
        /// </summary>
        /// <param name="ignoreType">
        /// フラグクリアの無視対象とするタイプを指定します。省略した場合は何も無視しません。
        /// </param>
        public void ClearChildModificationFlags(Type ignoreType = null)
        {
            var parent = this.Parent as HierarchyViewModel;
            Debug.Assert(parent != null, "DataModel is wrong.");
            foreach (var child in parent.Children.OfType<HierarchyViewModel>())
            {
                var owner = child as IModificationFlagOwner;

                if (owner == null)
                {
                    continue;
                }

                if (ignoreType != null)
                {
                    if (child.GetType() == ignoreType)
                    {
                        continue;
                    }
                }

                var childMod = owner.ModificationFlagViewModel;

                if (childMod == null)
                {
                    continue;
                }

                // 子要素のフラグが親に作用することがあるため、子を先にクリアする
                childMod.ClearChildModificationFlags(ignoreType);
                childMod.ClearModificationFlags();
            }
        }

        /// <summary>
        /// 初期値から変更されているデータがあるかを探索します。
        /// タブカラーに反映されれば良いので、ノード構成の変化は検出対象外です。
        /// </summary>
        public void UpdateDefaultValues()
        {
            // 深さ優先で探索します
            var parent = this.Parent as HierarchyViewModel;
            Debug.Assert(parent != null, "DataModel is wrong.");
            foreach (var child in parent.Children.OfType<IModificationFlagOwner>())
            {
                child.ModificationFlagViewModel.UpdateDefaultValues();
            }

            // グループビューモデル以外は無視します
            if (!(parent is PropertyGroupViewModel))
            {
                return;
            }

            // グループ内の要素になっているアニメ関連はビューモデルに判定を任せます。
            var animVm = parent as IModificationPropertyOwner;
            if (animVm != null)
            {
                if (!animVm.IsDefault)
                {
                    this.UpdateSomeValuesNotDefault(this, true);
                    #if DEBUG_MODIFICATION_FLAG_VIEWMODEL
                    Logger.Log("ModificationFlagViewModel.UpdateDefaultValues : {0}", animVm.GetType());
                    #endif
                }

                return;
            }

            // 親エミッタの場合は継承関連の設定を無視するようにする
            var inheritVm = parent as EmitterBasicInheritanceViewModel;
            bool shouldIgnoreInheritance = inheritVm != null;

            // CPUを使用する判定は例外的にVMのプロパティのみで確定させる
            var basicVm = parent as EmitterBasicBasicViewModel;
            if (basicVm != null)
            {
                if (basicVm.IsUsingCpu)
                {
                    this.UpdateSomeValuesNotDefault(this, true);
                    return;
                }
            }

            foreach (var propertyName in parent.GetDynamicMemberNames())
            {
                // 名前での無視判定
                if (propertyName == "Children"
                    || IgnorePropertyNamesStatic.Contains(propertyName)
                    || this.IgnorePropertyNames.Contains(propertyName))
                {
                    continue;
                }

                // 親エミッタ時の継承設定の無視判定
                if (shouldIgnoreInheritance && inheritVm.IsIgnoreMemberOnParent(propertyName))
                {
                    continue;
                }

                // 型による無視判定(本当は配列に突っ込んでContainsでやりたかったけど何故かできない)
                var value = parent.GetDefaultValue(propertyName);
                if (value is EmitterAnimationCommonData
                    || value is ArrayCollection<EmitterCustomShaderSettingData>
                    || value is ArrayCollection<CustomActionSettingData>
                    || value is IEnumerable<KeyValuePair<string, object>>)
                {
                    continue;
                }

                // ViewModelにデフォルト値を問い合わせて変化していたらアップデート
                if (!parent.IsPropertyDefaultValue(propertyName))
                {
                    this.UpdateSomeValuesNotDefault(this, true);
                    #if DEBUG_MODIFICATION_FLAG_VIEWMODEL
                    Logger.Log("ModificationFlagViewModel.UpdateDefaultValues : {0}", propertyName);
                    #endif
                    return;
                }
            }
        }

        /// <summary>
        /// Try to get the requested property value from the data model.
        /// </summary>
        /// <param name="binder">The get member binder.</param>
        /// <param name="result">The property value.</param>
        /// <returns>True on success.</returns>
        public override bool TryGetMember(
            GetMemberBinder binder,
            out object result)
        {
            if (this.Parent == null)
            {
                result = ModificationData.Empty;
                return false;
            }

            object currValue = null;

            var myBinder = new EffectMakerGetMemberBinder(binder.Name, true);
            this.Parent.TryGetMember(myBinder, out currValue);

            // Get the modification data for the property name.
            ModificationData data = null;
            if (this.ModifyFlags.TryGetValue(binder.Name, out data) == false)
            {
                // If the modification data is not found, means the property has
                // not been modified, so we only need to check if the property is
                // default value.
                result = new ModificationData()
                {
                    IsDefaultValue = this.Parent.IsPropertyDefaultValue(binder.Name),
                    IsModified = false,
                    OriginalValue = this.Parent.GetOriginalValue(binder.Name),
                    DefaultValue = this.Parent.GetDefaultValue(binder.Name),
                    CurrentValue = currValue,
                };
            }
            else
            {
                result = new ModificationData()
                {
                    IsDefaultValue = data.IsDefaultValue,
                    IsModified = data.IsModified,
                    OriginalValue = this.Parent.GetOriginalValue(binder.Name),
                    DefaultValue = this.Parent.GetDefaultValue(binder.Name),
                    CurrentValue = currValue,
                };
            }

            return true;
        }

        /// <summary>
        /// Update the property descriptor map for the this view model.
        /// </summary>
        protected override void UpdateViewModelPropertyDescriptorMap()
        {
            // Do nothing, the modification flag view model doesn't need this map.
        }

        /// <summary>
        /// IsAnyValueModifiedフラグを更新します。
        /// </summary>
        /// <param name="modifiedFlag">更新元のViewModel</param>
        /// <param name="isModified">新しい変更状態</param>
        protected void UpdateAnyValueModified(
            ModificationFlagViewModel modifiedFlag,
            bool isModified)
        {
            if (this.IsAnyValueModified == isModified)
            {
                // 変更状態の変更がないとき
                // 何もしない
                return;
            }
            else if (this.IsAnyValueModified == false && isModified == true)
            {
                // 今まで変更がなく初めて変更があったとき
                // 変更フラグを立てる
                this.IsAnyValueModified = true;

                // if文を抜けたあと、親の更新を呼び出す
            }
            else
            {
                //// if (this.IsAnyValueModified == true && isModified == false) と同じ
                // 変更が既にいくつかあり変更を戻されたとき

                // 他にも変更があるとき、何もしない
                if (this.HasAnyModifiedProperty() ||
                    this.IsAnyChildModified(modifiedFlag))
                {
                    return;
                }

                // 全ての変更が戻されたとき、変更フラグを下ろす
                this.IsAnyValueModified = false;

                // if文を抜けたあと、親の更新を呼び出す
            }

            // 親のフラグ更新を呼び出す
            var parent = this.Parent as HierarchyViewModel;

            if (parent == null)
            {
                return;
            }

            // プレビューとモデルの場合は親要素を更新しない
            if (parent is PreviewViewModel || parent is ModelViewModel)
            {
                return;
            }

            var owner = parent.Parent as IModificationFlagOwner;

            if (owner == null)
            {
                return;
            }

            var ownerFlag = owner.ModificationFlagViewModel;

            if (ownerFlag == null)
            {
                return;
            }

            ownerFlag.UpdateAnyValueModified(this, isModified);
        }

        /// <summary>
        /// IsSomeValuesNotDefaultフラグを更新します。
        /// </summary>
        /// <param name="modifiedFlag">更新元のViewModel</param>
        /// <param name="isNotDefault">新しいデフォルトからの変更状態</param>
        protected void UpdateSomeValuesNotDefault(
            ModificationFlagViewModel modifiedFlag,
            bool isNotDefault)
        {
            if (this.IsSomeValuesNotDefault == isNotDefault)
            {
                // デフォルト値との差分にの変更がないとき
                // 何もしない
                return;
            }
            else if (this.IsSomeValuesNotDefault == false && isNotDefault == true)
            {
                // 今まで変更がなく、初めてデフォルト値から変更があったとき
                // 変更フラグを立てる
                this.IsSomeValuesNotDefault = true;

                // if文を抜けたあと、親の更新を呼び出す
            }
            else
            {
                //// if (this.IsAnyValueModified == true && isModified == false) と同じ
                // 変更が既にいくつかあり変更を戻されたとき

                // 他にも変更があるとき、何もしない
                if (this.HasAnyNotDefaultProperty() ||
                    this.IsAnyChildNotDefault(modifiedFlag))
                {
                    return;
                }

                // 全ての変更が戻されたとき、変更フラグを下ろす
                this.IsSomeValuesNotDefault = false;

                // if文を抜けたあと、親の更新を呼び出す
            }

            // 親のフラグ更新を呼び出す
            var parent = this.Parent as HierarchyViewModel;

            if (parent == null)
            {
                return;
            }

            // プレビューとモデルの場合は親要素を更新しない
            if (parent is PreviewViewModel || parent is ModelViewModel)
            {
                return;
            }

            var owner = parent.Parent as IModificationFlagOwner;

            if (owner == null)
            {
                return;
            }

            var ownerFlag = owner.ModificationFlagViewModel;

            if (ownerFlag == null)
            {
                return;
            }

            ownerFlag.UpdateSomeValuesNotDefault(this, isNotDefault);
        }

        /// <summary>
        /// Handle PropertyChanged event issues when the properties of the parent is changed.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        protected virtual void OnParentPropertyChanged(
            object sender,
            PropertyChangedEventArgs e)
        {
            if (IgnoreParentPropertyChangedEvents == true
                || IgnorePropertyNamesStatic.Contains(e.PropertyName)
                || this.IgnorePropertyNames.Contains(e.PropertyName))
            {
                return;
            }

            // ユーザーページとペースト時のチャイルド変更は無視する
            if ((this.Parent is UserPageViewModel || ExportableViewModel.IsPasting)
                && e.PropertyName == "Children")
            {
                return;
            }

            // ページビューモデル内にはデータに関するプロパティがないので無視
            if (this.Parent is PropertyPageViewModel)
            {
                return;
            }

            // 親エミッタの場合は継承関連の設定を無視するようにする
            var inheritVm = this.Parent as EmitterBasicInheritanceViewModel;
            if (inheritVm != null && inheritVm.IsIgnoreMemberOnParent(e.PropertyName))
            {
                return;
            }

            // データ型依存の無視判定
            var value = this.Parent.GetDefaultValue(e.PropertyName);
            if (value is IExecutable
                || value is IEnumerable<KeyValuePair<string, object>>
                || value is EmitterAnimationCommonData
                || value is Action
                || value is ArrayCollection<EmitterCustomShaderSettingData>
                || value is ArrayCollection<CustomActionSettingData>)
            {
                return;
            }

            #if DEBUG_MODIFICATION_FLAG_VIEWMODEL
                Logger.Log("ModificationFlagViewModel.OnParentPropertyChanged : {0}", e.PropertyName);
            #endif

            // Get the modification data of the property.
            ModificationData data = null;
            if (this.ModifyFlags.TryGetValue(e.PropertyName, out data) == false)
            {
                // This is the first time the property is modified,
                // create the modification data first.
                data = new ModificationData();
                this.ModifyFlags.Add(e.PropertyName, data);
            }

            // Update the modification flags.
            data.IsDefaultValue = this.Parent.IsPropertyDefaultValue(e.PropertyName);
            data.IsModified = this.Parent.IsPropertyChanged(e.PropertyName);

            if (this.Parent is IModificationPropertyOwner)
            {
                if (value is AnimationTableData)
                {
                    // エミッタ時間アニメのテーブルは初期値からの変更を常にないものとして扱う(有効フラグの時点で非初期値になるから)
                    data.IsDefaultValue = true;
                }
            }

            #if DEBUG_MODIFICATION_FLAG_VIEWMODEL
            if (data.IsModified)
            {
                Logger.Log("Effect for star mark");
            }

            if (!data.IsDefaultValue)
            {
                Logger.Log("Effect for label color");
            }
            #endif

            this.UpdateAnyValueModified(this, data.IsModified);
            this.UpdateSomeValuesNotDefault(this, !data.IsDefaultValue);

            // Issue property changed event for "this" view model.
            // (so that it seems like this view model's property is modified.)
            // The caption labels are listening to this event, instead of the parent's.
            this.OnPropertyChanged(e.PropertyName);
        }

        /// <summary>
        /// 変更があるかどうか取得します。
        /// </summary>
        /// <returns>変更があるときTrueを返します。</returns>
        private bool HasAnyModifiedProperty()
        {
            if (this.ModifyFlags == null || this.ModifyFlags.Count <= 0)
            {
                return false;
            }

            foreach (var flag in this.ModifyFlags)
            {
                if ((flag.Value as ModificationData).IsModified)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// デフォルト値から変更があるかどうか取得します.
        /// </summary>
        /// <returns>デフォルト値から変更があるときにTrueを返します.</returns>
        private bool HasAnyNotDefaultProperty()
        {
            if (this.ModifyFlags == null || this.ModifyFlags.Count <= 0)
            {
                return false;
            }

            foreach (var flag in this.ModifyFlags)
            {
                if (!(flag.Value as ModificationData).IsDefaultValue)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// チャイルドに変更があるかどうか取得します。
        /// </summary>
        /// <param name="ignoreFlag">変更のチェックを無視するオブジェクト</param>
        /// <returns>変更があるときTrueを返します。</returns>
        private bool IsAnyChildModified(ModificationFlagViewModel ignoreFlag)
        {
            var parent = this.Parent as HierarchyViewModel;

            if (parent == null)
            {
                return false;
            }

            foreach (var child in parent.Children.OfType<HierarchyViewModel>())
            {
                var owner = child as IModificationFlagOwner;

                if (owner == null)
                {
                    continue;
                }

                var childMod = owner.ModificationFlagViewModel;

                if (childMod == null || childMod == ignoreFlag)
                {
                    continue;
                }

                if (childMod.IsAnyValueModified)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// チャイルドにデフォルト値との差分があるかどうか取得します。
        /// </summary>
        /// <param name="ignoreFlag">差分チェックを無視するオブジェクト</param>
        /// <returns>差分があるときTrueを返します。</returns>
        private bool IsAnyChildNotDefault(ModificationFlagViewModel ignoreFlag)
        {
            var parent = this.Parent as HierarchyViewModel;

            if (parent == null)
            {
                return false;
            }

            foreach (var child in parent.Children.OfType<HierarchyViewModel>())
            {
                var owner = child as IModificationFlagOwner;

                if (owner == null)
                {
                    continue;
                }

                var childMod = owner.ModificationFlagViewModel;

                if (childMod == null || childMod == ignoreFlag)
                {
                    continue;
                }

                if (childMod.IsSomeValuesNotDefault)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// データの変更状態を保持するクラス.
        /// </summary>
        public class ModificationData
        {
            /// <summary>A static empty modification data.</summary>
            private static ModificationData emptyData = null;

            /// <summary>
            /// Constructor.
            /// </summary>
            public ModificationData()
            {
                this.IsDefaultValue = true;
                this.IsModified = false;
                this.DefaultValue = null;
                this.OriginalValue = null;
                this.CurrentValue = null;
            }

            /// <summary>
            /// Get a static empty instance that indicates the data is not modified
            /// and equals to it's default value.
            /// </summary>
            public static ModificationData Empty
            {
                get
                {
                    if (ModificationData.emptyData == null)
                    {
                        ModificationData.emptyData = new ModificationData();
                    }

                    return ModificationData.emptyData;
                }
            }

            /// <summary>
            /// Get or set the flag indicating whether the data equals to it's default value.
            /// </summary>
            public bool IsDefaultValue { get; set; }

            /// <summary>
            /// Get or set the flag indicating whether the data has been modified.
            /// (since it's created or loaded from file.)
            /// </summary>
            public bool IsModified { get; set; }

            /// <summary>
            /// Get or set the default value.
            /// </summary>
            public object DefaultValue { get; set; }

            /// <summary>
            /// Get or set the original value.
            /// </summary>
            public object OriginalValue { get; set; }

            /// <summary>
            /// Get or set the current value.
            /// </summary>
            public object CurrentValue { get; set; }
        }
    }
}
