﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Collections;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Properties;

using BlAssetTypes = EffectMaker.BusinessLogic.AssetResourceTypes;
using UiAssetTypes = EffectMaker.UILogic.ViewModels.IO.AssetTypes;

namespace EffectMaker.UILogic.ViewModels.IO
{
    /// <summary>
    /// アセットのタイプです。
    /// </summary>
    public enum AssetTypes
    {
        /// <summary>
        /// エミッタのテクスチャ0
        /// </summary>
        EmitterTexture0 = 0x00000101,

        /// <summary>
        /// エミッタのテクスチャ1
        /// </summary>
        EmitterTexture1 = 0x00000102,

        /// <summary>
        /// エミッタのテクスチャ2
        /// </summary>
        EmitterTexture2 = 0x00000103,

        /// <summary>
        /// エミッタのボリュームプリミティブ
        /// </summary>
        EmitterVolumePrimitive = 0x00000204,

        /// <summary>
        /// エミッタのパーティクルプリミティブ
        /// </summary>
        EmitterParticlePrimitive = 0x00000205,

        /// <summary>
        /// エミッタのコンバイナシェーダ
        /// </summary>
        EmitterCombinerShader = 0x00000306,

        /// <summary>
        /// アセットタイプのマスク
        /// </summary>
        AssetTypeMask = 0x0000FF00,

        /// <summary>
        /// テクスチャアセット
        /// </summary>
        TextureAsset = 0x00000100,

        /// <summary>
        /// プリミティブアセット
        /// </summary>
        PrimitiveAsset  = 0x00000200,

        /// <summary>
        /// シェーダアセット
        /// </summary>
        CombinerAsset = 0x00000300,
    }

    /// <summary>
    /// ファイル参照のチェック結果です。
    /// </summary>
    public struct FileReferenceCheckResult
    {
        /// <summary>
        /// ファイル参照のチェック結果です。
        /// </summary>
        public ResultType Result;

        /// <summary>
        /// ソースパスから参照できるファイルのパスです。
        /// </summary>
        public string SrcFilePath;

        /// <summary>
        /// 保存先パスから参照できるファイルのパスです。
        /// </summary>
        public string DstFilePath;

        /// <summary>
        /// ソースパスから参照できるファイルのタイムスタンプです。
        /// </summary>
        public DateTime SrcFileLastWriteTime;

        /// <summary>
        /// 保存先パスから参照できるファイルのタイムスタンプです。
        /// </summary>
        public DateTime DstFileLastWriteTime;

        /// <summary>
        /// ファイル参照のチェック結果です。
        /// </summary>
        public enum ResultType
        {
            /// <summary>
            /// ソースパスと保存先パス、同じタイムスタンプのファイルが見つかった。
            /// </summary>
            FileFound,

            /// <summary>
            /// ソースパスと保存先パスで、違うタイムスタンプのファイルが見つかった。
            /// </summary>
            DifferentFileFound,

            /// <summary>
            /// ソースパスと保存パスでファイルが見つからなかった。
            /// </summary>
            SrcDstFileNotFound,

            /// <summary>
            /// ソースパスではファイルは見つからなかった、
            /// 保存先パスではファイルが見つかった。
            /// </summary>
            SrcFileNotFound,

            /// <summary>
            /// ソースパスではファイルが見つかったが、
            /// 保存先パスではファイルが見つからなかった。
            /// </summary>
            DstFileNotFound,
        }
    }

    /// <summary>
    /// アセットを管理するクラスです。
    /// </summary>
    public static class AssetsManager
    {
        /// <summary>
        /// エミッタセットの保存ダイアログを表示する前に
        /// テクスチャパスが正しく登録されているかチェックします。
        /// </summary>
        /// <param name="esetPath">エミッタセットのファイルパス</param>
        /// <param name="esetName">エミッタセットの名前</param>
        /// <param name="assetInfo">アセットリスト</param>
        /// <param name="showCancel">キャンセルボタンの表示On/Off</param>
        /// <returns>チェック結果を返します。</returns>
        public static DialogResult AreTexturesAssigned(
            string esetPath,
            string esetName,
            List<AssetInfo> assetInfo,
            bool showCancel)
        {
            bool valid = true;

            foreach (AssetInfo info in assetInfo)
            {
                if ((info.Type & UiAssetTypes.AssetTypeMask) == UiAssetTypes.TextureAsset)
                {
                    valid &= CheckTexturePathIsSetAndReachable(true, false, esetPath, esetName, info);
                }
            }

            DialogResult result = DialogResult.Yes;

            // 不正なテクスチャパスが見つかったとき
            if (!valid)
            {
                // エラーメッセージを表示
                string msg = string.Format(Resources.WarningTextureNotFoundCancelSaving, esetName);
                result = WorkspaceRootViewModel.Instance.Dialogs.ShowWarningTextureIsNotAssigned(msg, showCancel);
            }

            return result;
        }

        /// <summary>
        /// エミッタセットのロード処理が一通り終わったあと、
        /// アセットファイルのパスチェックと読み込みを行います。
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットビューモデル</param>
        /// <param name="esetPath">エミッタセットパス</param>
        /// <param name="esetName">エミッタセット名</param>
        public static void CheckAndLoadAssetsFile(
            EmitterSetViewModel emitterSetViewModel,
            string esetPath,
            string esetName)
        {
            var resultCounter = new TextureTypeCheckResult();

            // エミッタの処理を呼び出す
            foreach (IHierarchyObject child in emitterSetViewModel.Children)
            {
                var emitterViewModel = child as EmitterViewModel;

                if (emitterViewModel != null)
                {
                    CheckAndLoadAssetsFile(
                        emitterViewModel,
                        esetPath,
                        esetName,
                        resultCounter);
                }
            }

            if (resultCounter.WarningCountLinear > 0)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowTextureTypeMismatch(Resources.WarningTextureLinear);
            }
            else if (resultCounter.WarningCountNonLinear > 0)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowTextureTypeMismatch(Resources.WarningTextureNonlinear);
            }
        }

        /// <summary>
        /// アセットファイルのパスチェックと読み込みを行います。
        /// </summary>
        /// <param name="emitterViewModel">エミッタビューモデル</param>
        /// <param name="esetPath">エミッタセットパス</param>
        /// <param name="esetName">エミッタセット名</param>
        /// <param name="resultCounter">リザルトカウンタ</param>
        /// <returns>The result.</returns>
        public static int CheckAndLoadAssetsFile(
            EmitterViewModel emitterViewModel,
            string esetPath,
            string esetName,
            TextureTypeCheckResult resultCounter = null)
        {
            // アセットリストを取得
            var assets = GetAssetInfo(emitterViewModel);

            bool isLinearMode = OptionStore.ProjectConfig.LinearMode;
            bool isLinearTexture = false;
            int warningCountLinear = 0;
            int warningCountNonLinear = 0;

            // すべてのテクスチャアセットの読み込み、リニアモードのチェック
            foreach (AssetInfo info in assets)
            {
                if ((info.Type & UiAssetTypes.AssetTypeMask) == UiAssetTypes.TextureAsset)
                {
                    // テクスチャを読み込む
                    var result = AssetPathValidator.LoadTexture(info.FilePath);

                    if (result.ResultCode != LoadTextureResultCode.Success)
                    {
                        continue;
                    }

                    // テクスチャのリニアモードを取得
                    isLinearTexture = isLinearTexture | result.TextureData.Linear.Any(x => x);
                    var flag = emitterViewModel.EmitterTextureGroupViewModel.Texture0.EmitterTextureFileViewModel.CheckExceptionLinearWarning(isLinearTexture);

                    // テクスチャのリニアモードとオプションのリニアモードが違ったとき
                    if (isLinearTexture != isLinearMode && !flag)
                    {
                        // リニアモードに対応したエラーメッセージを表示
                        if (isLinearTexture)
                        {
                            ++warningCountLinear;
                        }
                        else
                        {
                            ++warningCountNonLinear;
                        }
                    }
                }
            }

            int texturesValid = 0;
            int primitivesValid = 0;
            int combinersValid = 0;

            // 設定されたアセットパスにファイルがあるかチェック
            foreach (AssetInfo info in assets)
            {
                var assetCategory = info.Type & UiAssetTypes.AssetTypeMask;
                if (assetCategory == UiAssetTypes.TextureAsset)
                {
                    texturesValid |= CheckTexturePathIsSetAndReachable(false, true, esetPath, esetName, info) ? 0 : 1;
                }
                else if (assetCategory == UiAssetTypes.PrimitiveAsset)
                {
                    primitivesValid |= VerifyPrimitivePathIsReachable(esetPath, esetName, info) ? 0 : 2;
                }
                else if (assetCategory == UiAssetTypes.CombinerAsset)
                {
                    combinersValid |= VerifyCombinerPathIsReachable(esetPath, esetName, info) ? 0 : 4;
                }
            }

            int checkResult = texturesValid | primitivesValid | combinersValid;

            // 子エミッタの処理
            foreach (IHierarchyObject child in emitterViewModel.Children)
            {
                var childEmitterViewModel = child as EmitterViewModel;

                if (childEmitterViewModel != null)
                {
                    var childResultCounter = new TextureTypeCheckResult();
                    checkResult |= CheckAndLoadAssetsFile(
                        childEmitterViewModel,
                        esetPath,
                        esetName,
                        childResultCounter);
                    warningCountLinear += childResultCounter.WarningCountLinear;
                    warningCountNonLinear += childResultCounter.WarningCountNonLinear;
                }
            }

            if (emitterViewModel.Parent is EmitterSetViewModel)
            {
                // テクスチャパスに読めないファイルがあったときエラーを表示
                if ((checkResult & 1) != 0)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowAssetPathProblem(
                        string.Format(Resources.WarningTextureNotInSearchPath, esetName),
                        Resources.WarningCaption);
                }

                // プリミティブパスに読めないファイルがあったときエラーを表示
                if ((checkResult & 2) != 0)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowAssetPathProblem(
                        string.Format(Resources.WarningPrimitiveNotInSearchPath, esetName),
                        Resources.WarningCaption);
                }

                // コンバイナシェーダパスに読めないファイルがあったときエラーを表示
                if ((checkResult & 4) != 0)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowAssetPathProblem(
                        string.Format(Resources.WarningCombinerNotInSearchPath, esetName),
                        Resources.WarningCaption);
                }
            }

            if (resultCounter == null)
            {
                if (warningCountLinear > 0)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowTextureTypeMismatch(Resources.WarningTextureLinear);
                }
                else if (warningCountNonLinear > 0)
                {
                    WorkspaceRootViewModel.Instance.Dialogs.ShowTextureTypeMismatch(Resources.WarningTextureNonlinear);
                }
            }
            else
            {
                resultCounter.WarningCountLinear += warningCountLinear;
                resultCounter.WarningCountNonLinear += warningCountNonLinear;
            }

            // ロケート後にステータスを更新
            emitterViewModel.UpdateAssetStatus(false, false);

            return checkResult;
        }

        /// <summary>
        /// エミッタセットを保存するとき
        /// アセットファイルの参照チェックを行います。
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットビューモデル</param>
        /// <param name="filePath">保存先パス(フルパス)</param>
        /// <param name="assetCopyPaths">コピーすることになったファイルリスト</param>
        /// <param name="pathSettingActs">アセットのパス設定アクションのリスト</param>
        /// <returns>チェックに通過したときtrueを返します。</returns>
        public static bool VerifyAssetsLocation(
            EmitterSetViewModel emitterSetViewModel,
            string filePath,
            List<FileCopyInfo> assetCopyPaths,
            List<Action> pathSettingActs)
        {
            bool allVerified = true;
            bool texturePathVerified = true;
            bool primitivePathVerified = true;
            bool combinerPathVerified = true;

            var overwriteList = new List<FileReferenceCheckResult>();
            var copyList = new List<AssetInfo>();

            var assetInfo = GetAssetInfoRecursive(emitterSetViewModel);

            // 各アセットが設定されているかのフラグ
            bool hasTex = false, hasPri = false, hasEcs = false;

            // アセットの参照チェック
            for (int i = 0; i < assetInfo.Count; ++i)
            {
                if (string.IsNullOrEmpty(assetInfo[i].FilePath))
                {
                    continue;
                }

                var assetType = EffectMaker.BusinessLogic.AssetResourceTypes.Texture;
                var assetCategory = assetInfo[i].Type & AssetTypes.AssetTypeMask;
                if (assetCategory == AssetTypes.TextureAsset)
                {
                    hasTex = true;
                }
                else if (assetCategory == AssetTypes.PrimitiveAsset)
                {
                    hasPri = true;
                    assetType = BlAssetTypes.Primitive;
                }
                else if (assetCategory == AssetTypes.CombinerAsset)
                {
                    hasEcs = true;
                    assetType = BlAssetTypes.CombinerShader;
                }
                else
                {
                    continue;
                }

                FileReferenceCheckResult result = FileReferenceCheck(
                    assetType,
                    assetInfo[i].FilePath,
                    emitterSetViewModel.DataModel.FilePath,
                    filePath);

                if (result.Result == FileReferenceCheckResult.ResultType.SrcDstFileNotFound)
                {
                    // ソースパスと保存先パスでアセットが見つからないとき、エラー
                    allVerified = false;

                    // テクスチャ/プリミティブエラーフラグを立てる
                    switch (assetType)
                    {
                        case BlAssetTypes.Texture:
                            texturePathVerified = false;
                            break;

                        case BlAssetTypes.Primitive:
                            primitivePathVerified = false;
                            break;

                        case BlAssetTypes.CombinerShader:
                            combinerPathVerified = false;
                            break;
                    }

                    // アセットファイルパスをファイル名だけに変更
                    // (いったんローカルにコピーしないとラムダのキャプチャに失敗する)
                    var infoEmitter = assetInfo[i].Emitter;
                    var infoType = assetInfo[i].Type;
                    var infoPath = assetInfo[i].FilePath;
                    pathSettingActs.Add(() => SetAssetPath(infoEmitter, infoType, Path.GetFileName(infoPath)));
                }
                else if (result.Result == FileReferenceCheckResult.ResultType.SrcFileNotFound)
                {
                    // ソースパスでアセットが見つからず保存先パスでアセットが見つかったとき

                    // アセットファイルパスを保存先パスに変更
                    // (いったんローカルにコピーしないとラムダのキャプチャに失敗する)
                    var infoEmitter = assetInfo[i].Emitter;
                    var infoType = assetInfo[i].Type;
                    var infoPath = result.DstFilePath;
                    pathSettingActs.Add(() => SetAssetPath(infoEmitter, infoType, infoPath));
                }
                else if (result.Result == FileReferenceCheckResult.ResultType.DstFileNotFound)
                {
                    // ソースパスでアセットが見つかり、保存先パスでアセットが見つからなかったとき
                    // 後でアセットをコピーする
                    copyList.Add(assetInfo[i]);

                    // ファイルパスを設定
                    assetInfo[i].FilePath = result.SrcFilePath;

                    // アセットファイルパスの更新はファイルをコピーするときに行う
                }
                else if (result.Result == FileReferenceCheckResult.ResultType.DifferentFileFound)
                {
                    // ソースパスと保存先パスで違うタイムスタンプのアセットが見つかったとき
                    // 後でアセットを上書きする
                    overwriteList.Add(result);

                    // ファイルパスを設定
                    assetInfo[i].FilePath = result.SrcFilePath;

                    // アセットファイルパスを更新
                    // (いったんローカルにコピーしないとラムダのキャプチャに失敗する)
                    var infoEmitter = assetInfo[i].Emitter;
                    var infoType = assetInfo[i].Type;
                    var infoPath = result.DstFilePath;
                    pathSettingActs.Add(() => SetAssetPath(infoEmitter, infoType, infoPath));
                }
                else if (result.Result == FileReferenceCheckResult.ResultType.FileFound)
                {
                    // ソースパスと保存先パスで同じタイムスタンプのアセットが見つかったとき

                    // アセットファイルパスを更新
                    // (いったんローカルにコピーしないとラムダのキャプチャに失敗する)
                    var infoEmitter = assetInfo[i].Emitter;
                    var infoType = assetInfo[i].Type;
                    var infoPath = result.DstFilePath;
                    pathSettingActs.Add(() => SetAssetPath(infoEmitter, infoType, infoPath));
                }
            }

            // 違うタイムスタンプのファイルの上書き処理を行う
            foreach (FileReferenceCheckResult result in overwriteList)
            {
                // ダイアログを表示して確認を取る
                DialogResult dialogResult = WorkspaceRootViewModel.Instance.Dialogs.PromptOverWriteAsset(result);

                // Yesが選択されたとき、ファイルを上書きする
                if (dialogResult == DialogResult.Yes)
                {
                    // コピーが確定したリストに追加だけして、実際のコピーは別で実行
                    assetCopyPaths.Add(new FileCopyInfo(result.SrcFilePath, result.DstFilePath));
                }
            }

            // コピー処理
            if (copyList.Count > 0)
            {
                // コピー対象リストからファイル名を取り出してリストにまとめる
                List<string> srcPaths = copyList.Select(info => info.FilePath).ToList();
                List<string> texPaths = null;
                List<string> priPaths = null;
                List<string> ecsPaths = null;

                // テクスチャコピー先候補のリストアップ
                if (hasTex)
                {
                    texPaths = PathUtility.ListValidAssetPaths(
                        IOConstants.GetAssetFolderName(BlAssetTypes.Texture),
                        filePath,
                        true);
                }

                // プリミティブコピー先候補のリストアップ
                if (hasPri)
                {
                    priPaths = PathUtility.ListValidAssetPaths(
                        IOConstants.GetAssetFolderName(BlAssetTypes.Primitive),
                        filePath,
                        true);
                }

                // コンバイナシェーダコピー先候補のリストアップ
                if (hasEcs)
                {
                    ecsPaths = PathUtility.ListValidAssetPaths(
                        IOConstants.GetAssetFolderName(BlAssetTypes.CombinerShader),
                        filePath,
                        true);
                }

                // ラジオボタンで選択したパスを取り出し、コピーを実行
                string textureCopyDestinationPath;
                string primitiveCopyDestinationPath;
                string combinerCopyDestinationPath;

                // ダイアログでユーザに問い合わせ
                DialogResult result = WorkspaceRootViewModel.Instance.Dialogs.ShowCopyAssetDialog(
                    out textureCopyDestinationPath,
                    out primitiveCopyDestinationPath,
                    out combinerCopyDestinationPath,
                    filePath,
                    srcPaths,
                    texPaths,
                    priPaths,
                    ecsPaths);

                if (result == DialogResult.OK)
                {
                    // ダイアログのOKが押されたときコピーを行う
                    foreach (AssetInfo info in copyList)
                    {
                        string fileName = Path.GetFileName(info.FilePath);
                        string targetPath;

                        // アセットタイプに対応した出力ディレクトリを取得
                        switch (info.Type & AssetTypes.AssetTypeMask)
                        {
                            case AssetTypes.TextureAsset:
                                targetPath = textureCopyDestinationPath;
                                break;

                            case AssetTypes.PrimitiveAsset:
                                targetPath = primitiveCopyDestinationPath;
                                break;

                            case AssetTypes.CombinerAsset:
                                targetPath = combinerCopyDestinationPath;
                                break;

                            default:
                                continue;
                        }

                        // ファイルをコピー
                        if (!string.IsNullOrEmpty(targetPath))
                        {
                            string targetFullPath = Path.Combine(targetPath, fileName);

                            // 同じファイルを複数設定していたときに2回以上コピーが発生しないよう
                            // コピー先ファイルの有無をチェックする
                            if (File.Exists(targetFullPath) == false)
                            {
                                assetCopyPaths.Add(new FileCopyInfo(info.FilePath, Path.Combine(targetPath, fileName)));
                            }

                            // アセットファイルパスを更新
                            var infoEmitter = info.Emitter;
                            var infoType = info.Type;
                            pathSettingActs.Add(() => SetAssetPath(infoEmitter, infoType, targetFullPath));
                        }
                    }
                }
                else
                {
                    // ダイアログでOKを選択されなかったとき、ログに警告を表示する
                    allVerified = false;

                    // テクスチャ/プリミティブエラーフラグを立てる
                    foreach (AssetInfo info in copyList)
                    {
                        string strFormat = null;

                        switch (info.Type)
                        {
                            case AssetTypes.EmitterTexture0:
                                strFormat = Resources.LogMessageTexture0NextTimeNotFound;
                                texturePathVerified = false;
                                break;

                            case AssetTypes.EmitterTexture1:
                                strFormat = Resources.LogMessageTexture1NextTimeNotFound;
                                texturePathVerified = false;
                                break;

                            case AssetTypes.EmitterTexture2:
                                strFormat = Resources.LogMessageTexture2NextTimeNotFound;
                                texturePathVerified = false;
                                break;

                            case AssetTypes.EmitterVolumePrimitive:
                                strFormat = Resources.LogMessageVolumePrimitiveNextTimeNotFound;
                                primitivePathVerified = false;
                                break;

                            case AssetTypes.EmitterParticlePrimitive:
                                strFormat = Resources.LogMessagePrimitiveNextTimeNotFound;
                                primitivePathVerified = false;
                                break;

                            case AssetTypes.EmitterCombinerShader:
                                strFormat = Resources.LogMessageCombinerNextTimeNotFound;
                                combinerPathVerified = false;
                                break;
                        }

                        // ログに警告を表示
                        Logger.Log(
                            LogLevels.Warning,
                            strFormat,
                            emitterSetViewModel.DataModel.Name,
                            info.Emitter.DataModel.Name,
                            info.FilePath);

                        // アセットファイルパスを更新
                        var infoEmitter = info.Emitter;
                        var infoType = info.Type;
                        var infoPath = Path.GetFileName(info.FilePath);
                        pathSettingActs.Add(() => SetAssetPath(infoEmitter, infoType, infoPath));
                    }
                }
            }

            // 不正なテクスチャパスがあったとき、エラーメッセージを表示
            if (!texturePathVerified)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowAssetPathProblem(
                    Resources.WarningTextureSaveLocationNotReachable,
                    Resources.WarningCaption);
            }

            // 不正なプリミティブパスがあったとき、エラーメッセージを表示
            if (!primitivePathVerified)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowAssetPathProblem(
                    Resources.WarningPrimitiveSaveLocationNotReachable,
                    Resources.WarningCaption);
            }

            // 不正なコンバイナパスがあったとき、エラーメッセージを表示
            if (!combinerPathVerified)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowAssetPathProblem(
                    Resources.WarningCombinerSaveLocationNotReachable,
                    Resources.WarningCaption);
            }

            return allVerified;
        }

        /// <summary>
        /// エミッタセット内のアセットパスをロケートします。
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットのビューモデル</param>
        public static void LocateAssetPaths(EmitterSetViewModel emitterSetViewModel)
        {
            if (emitterSetViewModel == null)
            {
                return;
            }

            // 全てのアセットパスをロケート
            List<AssetInfo> assets = GetAssetInfoRecursive(emitterSetViewModel);

            foreach (AssetInfo info in assets)
            {
                var assetType = BlAssetTypes.Texture;
                switch (info.Type & UiAssetTypes.AssetTypeMask)
                {
                    case UiAssetTypes.TextureAsset:
                        break;
                    case UiAssetTypes.PrimitiveAsset:
                        assetType = BlAssetTypes.Primitive;
                        break;
                    case UiAssetTypes.CombinerAsset:
                        assetType = BlAssetTypes.CombinerShader;
                        break;
                    default:
                        continue;
                }

                string fullPath;

                bool located = AssetPathValidator.LocateAssetPath(
                    assetType,
                    emitterSetViewModel.FilePath,
                    info.FilePath,
                    out fullPath);

                if (located)
                {
                    SetAssetPath(info.Emitter, info.Type, fullPath);
                }
            }
        }

        /// <summary>
        /// エミッタセットをデシリアライズする前に
        /// アセットパスをアセットファイル名に書き換えます。
        /// </summary>
        /// <param name="emitterSet">エミッタセットのデータモデル</param>
        /// <param name="emitterSetFilePath">エミッタセットのファイルパス</param>
        public static void ReplaceAssetsPathToFileName(EmitterSetData emitterSet, string emitterSetFilePath)
        {
            foreach (EmitterData emitter in emitterSet.EmitterList)
            {
                ReplaceAssetsPathToFileName(emitter, emitterSetFilePath);
            }
        }

        /// <summary>
        /// アセットパスの情報を取得します.
        /// </summary>
        /// <param name="filePath">アセットのファイルパス</param>
        /// <param name="esetFilePath">エミッタセットのファイルパス</param>
        /// <param name="assetType">アセットタイプ</param>
        /// <param name="reachable">ファイルがあるときtrue</param>
        /// <param name="locatable">ロケートできるときtrue</param>
        public static void GetAssetStatus(
            string filePath,
            string esetFilePath,
            BusinessLogic.AssetResourceTypes assetType,
            out bool reachable,
            out bool locatable)
        {
            reachable = true;
            locatable = true;

            // ファイルパスがないときはエラーなし
            if (string.IsNullOrEmpty(filePath))
            {
                return;
            }

            // ファイルパスが絶対パスでないとき、ファイルなしを返す
            // ロケートできるかどうかは不定で、UIに合わせてtrueを返す
            if (Path.IsPathRooted(filePath) == false)
            {
                reachable = false;
                return;
            }

            // ファイルがないとき、ファイルなし
            if (File.Exists(filePath) == false)
            {
                reachable = false;
            }

            // エミッタセットのファイルパスが有効なとき
            if (string.IsNullOrEmpty(esetFilePath) == false)
            {
                locatable = AssetPathValidator.CanLocateAssetPath(
                    assetType,
                    esetFilePath,
                    filePath);
            }
        }

        #region Fuction AssetInfoの取得, 設定

        /// <summary>
        /// ビューモデルを再帰的に辿ってAssetInfoを取得します。
        /// </summary>
        /// <param name="emitterSetViewModel">エミッタセットのビューモデル</param>
        /// <returns>AssetInfoを返します。</returns>
        public static List<AssetInfo> GetAssetInfoRecursive(EmitterSetViewModel emitterSetViewModel)
        {
            Debug.Assert(emitterSetViewModel != null, "エミッタセットビューモデルの指定が不正");

            List<AssetInfo> assetInfoList = new List<AssetInfo>();

            // 子エミッタのアセット情報を取得
            foreach (EmitterViewModel emitterViewModel in emitterSetViewModel.Emitters)
            {
                // アセット情報を取得
                IEnumerable<AssetInfo> info = GetAssetInfo(emitterViewModel);

                // アセット情報を登録
                assetInfoList.AddRange(info);
            }

            return assetInfoList;
        }

        /// <summary>
        /// エミッタのAssetInfoを取得します。
        /// </summary>
        /// <param name="emitterViewModel">エミッタビューモデル</param>
        /// <returns>エミッタのAssetInfoを返します。</returns>
        public static List<AssetInfo> GetAssetInfo(EmitterViewModel emitterViewModel)
        {
            Debug.Assert(emitterViewModel != null, "エミッタビューモデルの指定が不正");

            var assetInfoList = new List<AssetInfo>(5);
            AssetInfo info;

            var textureGroupViewModel = emitterViewModel.GetChild<EmitterTextureGroupViewModel>();
            Debug.Assert(textureGroupViewModel != null, "エミッタビューモデルの階層データが不正");

            // テクスチャ0
            info = new AssetInfo()
            {
                Emitter = emitterViewModel,
                Enable = true,
                FilePath = textureGroupViewModel.Texture0.EmitterTextureFileViewModel.FilePath,
                Type = AssetTypes.EmitterTexture0
            };

            assetInfoList.Add(info);

            // テクスチャ1
            info = new AssetInfo()
            {
                Emitter = emitterViewModel,
                Enable = true,
                FilePath = textureGroupViewModel.Texture1.EmitterTextureFileViewModel.FilePath,
                Type = AssetTypes.EmitterTexture1
            };

            assetInfoList.Add(info);

            // テクスチャ2
            info = new AssetInfo()
            {
                Emitter = emitterViewModel,
                Enable = true,
                FilePath = textureGroupViewModel.Texture2.EmitterTextureFileViewModel.FilePath,
                Type = AssetTypes.EmitterTexture2
            };

            assetInfoList.Add(info);

            // ボリューム(エミッタ)プリミティブ
            var emitterEmitterViewModel = emitterViewModel.GetChild<EmitterEmitterViewModel>();
            Debug.Assert(emitterEmitterViewModel != null, "エミッタビューモデルの階層データが不正");

            var emitterEmitterShapeViewModel = emitterEmitterViewModel.GetChild<EmitterEmitterShapeViewModel>();
            Debug.Assert((EmitterEmitterShapeViewModel)emitterEmitterShapeViewModel != null, "エミッタビューモデルの階層データが不正");

            // 形状がプリミティブに設定されている場合のみ有効なアセットとして登録
            if (emitterEmitterShapeViewModel.EmitterType == 15)
            {
                info = new AssetInfo()
                {
                    Emitter = emitterViewModel,
                    Enable = true,
                    FilePath = emitterEmitterShapeViewModel.PrimitiveFilePath,
                    Type = AssetTypes.EmitterVolumePrimitive
                };

                assetInfoList.Add(info);
            }

            // パーティクルプリミティブ
            var emitterParticleViewModel = emitterViewModel.GetChild<EmitterParticleViewModel>();
            Debug.Assert(emitterParticleViewModel != null, "エミッタビューモデルの階層データが不正");

            var emitterParticleShapeViewModel = emitterParticleViewModel.GetChild<EmitterParticleShapeViewModel>();
            Debug.Assert((EmitterParticleShapeViewModel)emitterParticleShapeViewModel != null, "エミッタビューモデルの階層データが不正");

            // 形状がプリミティブに設定されている場合のみ有効なアセットとして登録
            if (emitterParticleShapeViewModel.ShapeType == 1)
            {
                info = new AssetInfo()
                {
                    Emitter = emitterViewModel,
                    Enable = true,
                    FilePath = emitterParticleShapeViewModel.PrimitiveFilePath,
                    Type = AssetTypes.EmitterParticlePrimitive
                };

                assetInfoList.Add(info);
            }

            // コンバイナシェーダ
            var emitterCombinerViewModel = emitterViewModel.GetChild<EmitterCombinerViewModel>();
            Debug.Assert(emitterCombinerViewModel != null, "エミッタビューモデルの階層データが不正");

            var emitterCombinerEditorViewModel = emitterCombinerViewModel.GetChild<EmitterCombinerEditorViewModel>();
            Debug.Assert(emitterCombinerEditorViewModel != null, "エミッタビューモデルの階層データが不正");

            if (emitterCombinerEditorViewModel.IsEffectCombinerEnabled)
            {
                if (!string.IsNullOrEmpty(emitterCombinerEditorViewModel.CombinerEditorProjectPath))
                {
                    info = new AssetInfo()
                    {
                        Emitter = emitterViewModel,
                        Enable = true,
                        FilePath = emitterCombinerEditorViewModel.CombinerEditorProjectPath,
                        Type = AssetTypes.EmitterCombinerShader
                    };
                }

                assetInfoList.Add(info);
            }

            return assetInfoList;
        }

        /// <summary>
        /// エミッタのアセットファイルパスを設定します。
        /// </summary>
        /// <param name="emitterViewModel">エミッタビューモデル</param>
        /// <param name="type">アセットタイプ</param>
        /// <param name="path">ファイルパス</param>
        public static void SetAssetPath(EmitterViewModel emitterViewModel, AssetTypes type, string path)
        {
            switch (type)
            {
                // テクスチャ0
                case AssetTypes.EmitterTexture0:
                {
                    var textureGroupViewModel = emitterViewModel.GetChild<EmitterTextureGroupViewModel>();
                    Debug.Assert(textureGroupViewModel != null, "エミッタビューモデルの階層データが不正");

                    // 警告ウィンドウを表示せずに、テクスチャファイルのパスを更新する.
                    textureGroupViewModel.Texture0.EmitterTextureFileViewModel.SetFilePathWithWarningWindow(path, false);
                    break;
                }

                // テクスチャ1
                case AssetTypes.EmitterTexture1:
                {
                    var textureGroupViewModel = emitterViewModel.GetChild<EmitterTextureGroupViewModel>();
                    Debug.Assert(textureGroupViewModel != null, "エミッタビューモデルの階層データが不正");

                    // 警告ウィンドウを表示せずに、テクスチャファイルのパスを更新する.
                    textureGroupViewModel.Texture1.EmitterTextureFileViewModel.SetFilePathWithWarningWindow(path, false);
                    break;
                }

                // テクスチャ2
                case AssetTypes.EmitterTexture2:
                {
                    var textureGroupViewModel = emitterViewModel.GetChild<EmitterTextureGroupViewModel>();
                    Debug.Assert(textureGroupViewModel != null, "エミッタビューモデルの階層データが不正");

                    // 警告ウィンドウを表示せずに、テクスチャファイルのパスを更新する.
                    textureGroupViewModel.Texture2.EmitterTextureFileViewModel.SetFilePathWithWarningWindow(path, false);
                    break;
                }

                // ボリュームプリミティブ
                case AssetTypes.EmitterVolumePrimitive:
                {
                    EmitterEmitterViewModel emitterEmitterViewModel = emitterViewModel.GetChild<EmitterEmitterViewModel>();
                    Debug.Assert(emitterEmitterViewModel != null, "エミッタビューモデルの階層データが不正");

                    var emitterEmitterShapeViewModel = emitterEmitterViewModel.GetChild<EmitterEmitterShapeViewModel>();
                    Debug.Assert((EmitterEmitterShapeViewModel)emitterEmitterShapeViewModel != null, "エミッタビューモデルの階層データが不正");

                    // 警告ウィンドウを表示せずにプリミティブモデルのファイルパスを更新する
                    emitterEmitterShapeViewModel.SetPrimitiveFilePathWithWarningWindow(path, false);
                    break;
                }

                // パーティクルプリミティブ
                case AssetTypes.EmitterParticlePrimitive:
                {
                    EmitterParticleViewModel emitterParticleViewModel = emitterViewModel.GetChild<EmitterParticleViewModel>();
                    Debug.Assert(emitterParticleViewModel != null, "エミッタビューモデルの階層データが不正");

                    var emitterParticleShapeViewModel = emitterParticleViewModel.GetChild<EmitterParticleShapeViewModel>();
                    Debug.Assert((EmitterParticleShapeViewModel)emitterParticleShapeViewModel != null, "エミッタビューモデルの階層データが不正");

                    // 警告ウィンドウを表示せずにプリミティブモデルのファイルパスを更新する
                    emitterParticleShapeViewModel.SetPrimitiveFilePathWithWarningWindow(path, false);
                    break;
                }

                // コンバイナシェーダ
                case AssetTypes.EmitterCombinerShader:
                {
                    var combinerViewModel = emitterViewModel.GetChild<EmitterCombinerViewModel>();
                    Debug.Assert(combinerViewModel != null, "エミッタビューモデルの階層データが不正");

                    combinerViewModel.EmitterCombinerEditorViewModel.SetCombinerEditorProjectPath(path, false);

                    break;
                }

                // その他
                default:
                {
                    Debug.Assert(true, "アセットタイプの指定が不正");
                    break;
                }
            }
        }

        #endregion

        /// <summary>
        /// アセットファイルの参照チェックを行います。
        /// </summary>
        /// <param name="assetType">アセットタイプ</param>
        /// <param name="assetPath">アセットファイルパス</param>
        /// <param name="srcEsetPath">ソースパス</param>
        /// <param name="dstEsetPath">保存先パス</param>
        /// <returns>チェック結果を返します。</returns>
        private static FileReferenceCheckResult FileReferenceCheck(
            BlAssetTypes assetType,
            string assetPath,
            string srcEsetPath,
            string dstEsetPath)
        {
            FileReferenceCheckResult result = new FileReferenceCheckResult();

            // ソースパス, 保存先パスのフォルダパスを取得
            string srcEsetDir = PathUtility.GetDirectoryName(srcEsetPath);
            string dstEsetDir = PathUtility.GetDirectoryName(dstEsetPath);

            bool srcFound;  // ソースパスでの参照チェック結果
            bool dstFound;  // 保存先パスでの参照チェック結果

            // アセットファイルの直接参照をチェック
            if (File.Exists(assetPath))
            {
                srcFound = true;
                result.SrcFilePath = assetPath;
            }
            else
            {
                // ソースパスからの参照をチェック
                srcFound = AssetPathValidator.LocateAssetPath(
                    assetType,
                    srcEsetDir,
                    Path.GetFileName(assetPath),
                    out result.SrcFilePath);
            }

            // 保存先パスからの参照をチェック
            dstFound = AssetPathValidator.LocateAssetPath(
                assetType,
                dstEsetDir,
                Path.GetFileName(assetPath),
                out result.DstFilePath);

            // ソースパスと保存先パスから参照できるファイルが見つからなかった
            if (!srcFound && !dstFound)
            {
                result.Result = FileReferenceCheckResult.ResultType.SrcDstFileNotFound;
                return result;
            }

            // ソースパスから参照できるファイルはあったが保存先パスから参照できるファイルは見つからなかった
            if (!srcFound && dstFound)
            {
                result.Result = FileReferenceCheckResult.ResultType.SrcFileNotFound;
                return result;
            }

            // 保存先パスから参照できるファイルが見つからなかった
            if (srcFound && !dstFound)
            {
                result.Result = FileReferenceCheckResult.ResultType.DstFileNotFound;
                return result;
            }

            // ソースパスと保存先パスで、違うパスのファイルが見つかった
            if (dstFound && result.DstFilePath.ToLower() != result.SrcFilePath.ToLower())
            {
                // それぞれのタイムスタンプを取得
                result.SrcFileLastWriteTime = File.GetLastWriteTime(result.SrcFilePath);
                result.DstFileLastWriteTime = File.GetLastWriteTime(result.DstFilePath);

                // タイムスタンプが違うファイルだった
                if (result.SrcFileLastWriteTime.CompareTo(result.DstFileLastWriteTime) != 0)
                {
                    result.Result = FileReferenceCheckResult.ResultType.DifferentFileFound;
                    return result;
                }
            }

            // ソースパスと保存先パスで、同じタイムスタンプのファイルが見つかった
            result.Result = FileReferenceCheckResult.ResultType.FileFound;
            return result;
        }

        /// <summary>
        /// エミッタをデシリアライズする前に
        /// アセットパスをアセットファイル名に書き換えます。
        /// </summary>
        /// <param name="emitter">エミッタのデータモデル</param>
        /// <param name="emitterSetFilePath">エミッタセットのファイルパス</param>
        private static void ReplaceAssetsPathToFileName(EmitterData emitter, string emitterSetFilePath)
        {
            string path;

            path = emitter.Textures.Texture0.EmitterTextureFileData.FilePath;
            if (string.IsNullOrEmpty(path) == false)
            {
                emitter.Textures.Texture0.EmitterTextureFileData.FilePath = Path.GetFileName(path);
            }

            path = emitter.Textures.Texture1.EmitterTextureFileData.FilePath;
            if (string.IsNullOrEmpty(path) == false)
            {
                emitter.Textures.Texture1.EmitterTextureFileData.FilePath = Path.GetFileName(path);
            }

            path = emitter.Textures.Texture2.EmitterTextureFileData.FilePath;
            if (string.IsNullOrEmpty(path) == false)
            {
                emitter.Textures.Texture2.EmitterTextureFileData.FilePath = Path.GetFileName(path);
            }

            path = emitter.EmitterParticleData.EmitterParticleShapeData.PrimitiveFilePath;
            if (string.IsNullOrEmpty(path) == false)
            {
                emitter.EmitterParticleData.EmitterParticleShapeData.PrimitiveFilePath = Path.GetFileName(path);
            }

            path = emitter.EmitterEmitterData.EmitterEmitterShapeData.PrimitiveFilePath;
            if (string.IsNullOrEmpty(path) == false)
            {
                emitter.EmitterEmitterData.EmitterEmitterShapeData.PrimitiveFilePath = Path.GetFileName(path);
            }

            // コンバイナエディタがOnになっている場合のみ、コンバイナシェーダのパスも処理する.
            if (OptionStore.ProjectConfig.IsEftCombinerEditorEnabled == true)
            {
                path = emitter.EmitterCombinerData.EmitterCombinerEditorData.CombinerEditorProjectPath;
                if (string.IsNullOrEmpty(path) == false)
                {
                    emitter.EmitterCombinerData.EmitterCombinerEditorData.CombinerEditorProjectPath =
                        Path.GetFileName(path);
                }
            }

            // チャイルド呼び出し
            foreach (EmitterData childEmitter in emitter.EmitterList)
            {
                ReplaceAssetsPathToFileName(childEmitter, emitterSetFilePath);
            }
        }

        /// <summary>
        /// テクスチャパスが正しく登録されているかチェックします。
        /// </summary>
        /// <param name="checkSet">テクスチャパスが空でないかチェックします</param>
        /// <param name="checkReachable">テクスチャパスにファイルがあるかチェックします</param>
        /// <param name="esetPath">エミッタセットのファイルパス</param>
        /// <param name="esetName">エミッタセットの名前</param>
        /// <param name="textureInfo">テクスチャの情報</param>
        /// <returns>テクスチャが指定されていないときfalseを返します。</returns>
        private static bool CheckTexturePathIsSetAndReachable(
            bool checkSet,
            bool checkReachable,
            string esetPath,
            string esetName,
            AssetInfo textureInfo)
        {
            string notFoundMessage = null;
            string notSetMessage = null;

            switch (textureInfo.Type)
            {
                case UiAssetTypes.EmitterTexture0:
                    notFoundMessage = Resources.LogMessageTexture0NotFound;
                    notSetMessage = Resources.LogMessageTexture0NotSet;
                    break;

                case UiAssetTypes.EmitterTexture1:
                    notFoundMessage = Resources.LogMessageTexture1NotFound;
                    break;

                case UiAssetTypes.EmitterTexture2:
                    notFoundMessage = Resources.LogMessageTexture2NotFound;
                    break;
            }

            // テクスチャパスが空のとき
            if (string.IsNullOrEmpty(textureInfo.FilePath))
            {
                if (checkSet && textureInfo.Type == UiAssetTypes.EmitterTexture0)
                {
                    Logger.Log(
                        "LogView",
                        LogLevels.Warning,
                        notSetMessage,
                        esetName,
                        textureInfo.Emitter.DataModel.Name);

                    return false;
                }
                else
                {
                    return true;
                }
            }

            // エミッタセットパスが有効なとき
            if (checkReachable && !string.IsNullOrEmpty(esetPath))
            {
                string fullPath;

                bool found = AssetPathValidator.LocateAssetPath(
                    BlAssetTypes.Texture,
                    PathUtility.GetDirectoryName(esetPath),
                    Path.GetFileName(textureInfo.FilePath),
                    out fullPath);

                if (!found)
                {
                    Logger.Log(
                        "LogView",
                        LogLevels.Warning,
                        notFoundMessage,
                        esetName,
                        textureInfo.Emitter.DataModel.Name,
                        textureInfo.FilePath);

                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// プリミティブのパスが正しく登録されているかチェックします。
        /// </summary>
        /// <param name="esetPath">エミッタセットのファイルパス</param>
        /// <param name="esetName">エミッタセットの名前</param>
        /// <param name="primitiveInfo">プリミティブの情報</param>
        /// <returns>プリミティブパスが不正なときfalseを返します。</returns>
        private static bool VerifyPrimitivePathIsReachable(
            string esetPath,
            string esetName,
            AssetInfo primitiveInfo)
        {
            if (!primitiveInfo.Enable)
            {
                return true;
            }

            if (string.IsNullOrEmpty(primitiveInfo.FilePath) ||
                string.IsNullOrEmpty(esetPath))
            {
                return true;
            }

            // アセットタイプに対応したログメッセージを取得
            string notFoundMessage = null;

            switch (primitiveInfo.Type)
            {
                case UiAssetTypes.EmitterParticlePrimitive:
                    notFoundMessage = Resources.LogMessagePrimitiveNotFound;
                    break;

                case UiAssetTypes.EmitterVolumePrimitive:
                    notFoundMessage = Resources.LogMessagePrimitiveNotFound;
                    break;
            }

            if (!Path.IsPathRooted(primitiveInfo.FilePath))
            {
                Logger.Log(
                    "LogView",
                    LogLevels.Warning,
                    notFoundMessage,
                    esetName,
                    primitiveInfo.Emitter.DataModel.Name,
                    primitiveInfo.FilePath);

                return false;
            }

            return true;
        }

        /// <summary>
        /// コンバイナシェーダのパスが正しく登録されているかチェックします。
        /// </summary>
        /// <param name="esetPath">エミッタセットのファイルパス</param>
        /// <param name="esetName">エミッタセットの名前</param>
        /// <param name="combinerInfo">コンバイナシェーダの情報</param>
        /// <returns>コンバイナシェーダパスが不正なときfalseを返します。</returns>
        private static bool VerifyCombinerPathIsReachable(
            string esetPath,
            string esetName,
            AssetInfo combinerInfo)
        {
            if (!combinerInfo.Enable)
            {
                return true;
            }

            if (string.IsNullOrEmpty(combinerInfo.FilePath) ||
                string.IsNullOrEmpty(esetPath))
            {
                return true;
            }

            // アセットタイプに対応したログメッセージを取得
            string notFoundMessage = Resources.LogMessageCombinerNotFound;

            if (!Path.IsPathRooted(combinerInfo.FilePath))
            {
                Logger.Log(
                    "LogView",
                    LogLevels.Warning,
                    notFoundMessage,
                    esetName,
                    combinerInfo.Emitter.DataModel.Name,
                    combinerInfo.FilePath);

                return false;
            }

            return true;
        }
    }

    /// <summary>
    /// アセットの情報です。
    /// </summary>
    public class AssetInfo
    {
        /// <summary>
        /// エミッタビューモデルを取得または設定します。
        /// </summary>
        public EmitterViewModel Emitter { get; set; }

        /// <summary>
        /// アセットの有効/無効を取得または設定します。
        /// </summary>
        public bool Enable { get; set; }

        /// <summary>
        /// アセットのファイルパスを取得または設定します。
        /// </summary>
        public string FilePath { get; set; }

        /// <summary>
        /// アセットのタイプを取得または設定します。
        /// </summary>
        public AssetTypes Type { get; set; }
    }

    /// <summary>
    /// テクスチャタイプのチェックを行い、警告回数を記録するためのクラス
    /// </summary>
    public class TextureTypeCheckResult
    {
        /// <summary>
        /// リニアなテクスチャだった回数
        /// </summary>
        public int WarningCountLinear { get; set; }

        /// <summary>
        /// ノンリニアなテクスチャだった回数
        /// </summary>
        public int WarningCountNonLinear { get; set; }
    }
}
