﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// チャイルドエミッタを親と同階層に移動するコマンドを定義します。
    /// </summary>
    [Alias("UnparentEmitter")]
    public class UnparentEmitterCommand : CommandBase
    {
        /// <summary>
        /// Stores the command stack target.
        /// </summary>
        private object commandStackTarget;

        /// <summary>
        /// 操作対象の子エミッタビューモデルです。
        /// </summary>
        private EmitterViewModel node;

        /// <summary>
        /// 階層化解除前の親エミッタビューモデルです。
        /// </summary>
        private EmitterViewModel parent;

        /// <summary>
        /// 共通の親となるエミッタセットビューモデルです。
        /// </summary>
        private EmitterSetViewModel emitterSet;

        /// <summary>
        /// 操作対象の子エミッタのインデックスです。
        /// </summary>
        private int childIndexVm, childIndexDm;

        /// <summary>
        /// 親エミッタのインデックスです。
        /// </summary>
        private int parentIndexVm, parentIndexDm;

        /// <summary>
        /// 親エミッタで上書きされる前の値のテーブル
        /// </summary>
        private Dictionary<string, bool> overrideDictionary;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UnparentEmitterCommand()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="node">チャイルドエミッタビューモデル</param>
        public UnparentEmitterCommand(EmitterViewModel node)
        {
            object[] args = new object[] { node };

            this.Initialize(args);
        }

        /// <summary>
        /// Initializes the command.
        /// </summary>
        /// <param name="args">Command arguments, two HierarchyViewModel instances.</param>
        /// <returns>Returns true on success, false otherwise.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 1)
            {
                return false;
            }

            this.node = args[0] as EmitterViewModel;
            if (this.node == null)
            {
                return false;
            }

            this.parent = this.node.Parent as EmitterViewModel;
            if (this.parent == null)
            {
                return false;
            }

            this.emitterSet = this.parent.Parent as EmitterSetViewModel;
            if (this.emitterSet == null)
            {
                return false;
            }

            this.parent.GetEmitterIndex(this.node, out this.childIndexVm, out this.childIndexDm);
            this.emitterSet.GetEmitterIndex(this.parent, out this.parentIndexVm, out this.parentIndexDm);

            var dm = this.node.EmitterBasicViewModel.EmitterBasicInheritanceViewModel.DataModel;
            this.overrideDictionary = new Dictionary<string, bool>
            {
                { "EnableDrawPath", dm.EnableDrawPath },
                { "EnableColor0", dm.EnableColor0 },
                { "EnableColor1", dm.EnableColor1 },
                { "EnableAlpha0", dm.EnableAlpha0 },
                { "EnableAlpha1", dm.EnableAlpha1 },
                { "EnableScale", dm.EnableScale },
                { "EnableRotation", dm.EnableRotation }
            };

            this.commandStackTarget =
                CommandBase.FindCommandStackTargetFromHierarchyObject(this.node);

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// Gets the command target.
        /// </summary>
        /// <returns>Returns the command target.</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Adds the child to the parent.
        /// </summary>
        /// <returns>Returns true or throw an exception.</returns>
        public override bool Do()
        {
            using (new MessageBlockerWithSendBinaryOnce(this.emitterSet.DataModel))
            {
                if (!this.parent.RemoveEmitter(this.node))
                {
                    return false;
                }

                // 親エミッタで上書きされるべき値にパッチを当てる
                var dm = this.node.EmitterBasicViewModel.EmitterBasicInheritanceViewModel.DataModel;
                dm.EnableDrawPath = false;
                dm.EnableColor0 = false;
                dm.EnableColor1 = false;
                dm.EnableAlpha0 = false;
                dm.EnableAlpha1 = false;
                dm.EnableScale = false;
                dm.EnableRotation = false;

                if (!this.emitterSet.InsertEmitter(this.node, this.parentIndexVm + 1, this.parentIndexDm + 1))
                {
                    return false;
                }

                this.parent.EmitterBasicViewModel.EmitterBasicBasicViewModel.IsUsingCpu = false;
            }

            this.node.IsSelected = false;
            this.node.IsSelected = true;

            WorkspaceRootViewModel.Instance.UpdateUIStates();

            return true;
        }

        /// <summary>
        /// Removes the child from the parent.
        /// </summary>
        /// <returns>Returns true or throw an exception.</returns>
        public override bool Undo()
        {
            using (new MessageBlockerWithSendBinaryOnce(this.emitterSet.DataModel))
            {
                if (!this.emitterSet.RemoveEmitter(this.node))
                {
                    return false;
                }

                // 親エミッタで上書きされるべき値にパッチを当てたものを復元する
                var dm = this.node.EmitterBasicViewModel.EmitterBasicInheritanceViewModel.DataModel;
                dm.EnableDrawPath = overrideDictionary["EnableDrawPath"];
                dm.EnableColor0 = overrideDictionary["EnableColor0"];
                dm.EnableColor1 = overrideDictionary["EnableColor1"];
                dm.EnableAlpha0 = overrideDictionary["EnableAlpha0"];
                dm.EnableAlpha1 = overrideDictionary["EnableAlpha1"];
                dm.EnableScale = overrideDictionary["EnableScale"];
                dm.EnableRotation = overrideDictionary["EnableRotation"];

                if (!this.parent.InsertEmitter(this.node, this.childIndexVm, this.childIndexDm))
                {
                    return false;
                }

                this.parent.EmitterBasicViewModel.EmitterBasicBasicViewModel.IsUsingCpu = true;
            }

            this.node.IsSelected = false;
            this.node.IsSelected = true;

            WorkspaceRootViewModel.Instance.UpdateUIStates();

            return true;
        }
    }
}
