﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// プレビューを保存するコマンドです。
    /// </summary>
    [Alias("SavePreview")]
    public class SavePreviewCommand : CommandBase
    {
        /// <summary>
        /// ワークスペースのビューモデルです.
        /// </summary>
        private WorkspaceViewModel workspaceViewModel;

        /// <summary>
        /// プレビューのビューモデルです.
        /// </summary>
        private PreviewViewModel previewViewModel;

        /// <summary>
        /// 保存するファイルの名前です.
        /// </summary>
        private bool showNameInputDialog;

        /// <summary>The command stack target.</summary>
        private object commandStackTarget = null;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public SavePreviewCommand()
        {
            this.CanUndo = false;
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="workspaceViewModel">ワークスペースビューモデル</param>
        /// <param name="previewViewModel">プレビュービューモデル</param>
        /// <param name="showNameInputDialog">名前入力ダイアログの表示フラグ</param>
        public SavePreviewCommand(WorkspaceViewModel workspaceViewModel, PreviewViewModel previewViewModel, bool showNameInputDialog)
            : this()
        {
            object[] args = new object[] { workspaceViewModel, previewViewModel, showNameInputDialog };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">The arguments for the command.</param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 3)
            {
                return false;
            }

            // ワークスペースビューモデルを記録
            this.workspaceViewModel = args[0] as WorkspaceViewModel;

            if (this.workspaceViewModel == null)
            {
                return false;
            }

            // プレビュービューモデルを記録
            this.previewViewModel = args[1] as PreviewViewModel;

            if (this.previewViewModel == null)
            {
                return false;
            }

            // 名前入力ダイアログの表示フラグを記録
            if (args[2] is bool == false)
            {
                return false;
            }

            this.showNameInputDialog = (bool)args[2];

            // 遡ってCommandStackOwnerAttributeを持つインスタンスをチェックする
            var cmdStackTarget = this.previewViewModel.Parent as IHierarchyObject;
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(cmdStackTarget) as IHierarchyObject;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            bool result = this.workspaceViewModel.SavePreviewFile(
                this.previewViewModel,
                this.showNameInputDialog);

            return result;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            return false;
        }

        /// <summary>
        /// Redo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Redo()
        {
            return false;
        }
    }
}
