﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.ViewModels;
using EffectMaker.UILogic.ViewModels.IO;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// エミッタセットを保存するコマンドです。
    /// </summary>
    [Alias("SaveEmitterSet")]
    public class SaveEmitterSetCommand : CommandBase
    {
        /// <summary>
        /// ワークスペースのビューモデルです.
        /// </summary>
        private WorkspaceViewModel workspaceViewModel;

        /// <summary>
        /// エミッタセットのビューモデルです.
        /// </summary>
        private EmitterSetViewModel emitterSetViewModel;

        /// <summary>
        /// 名前入力ダイアログの表示フラグです.
        /// </summary>
        private bool showNameInputDialog;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public SaveEmitterSetCommand()
        {
            this.CanUndo = false;
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="workspaceViewModel">ワークスペースのビューモデル</param>
        /// <param name="emitterSetViewModel">エミッタセットのビューモデル</param>
        /// <param name="showNameInputDialog">名前入力ダイアログの表示フラグを記録</param>
        public SaveEmitterSetCommand(WorkspaceViewModel workspaceViewModel, EmitterSetViewModel emitterSetViewModel, bool showNameInputDialog )
            : this()
        {
            object[] args = new object[] { workspaceViewModel, emitterSetViewModel, showNameInputDialog };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">The arguments for the command.</param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 3)
            {
                return false;
            }

            // ワークスペースのビューモデルを記録
            this.workspaceViewModel = args[0] as WorkspaceViewModel;

            if (this.workspaceViewModel == null)
            {
                return false;
            }

            // エミッタセットのビューモデルを記録
            this.emitterSetViewModel = args[1] as EmitterSetViewModel;

            if (this.emitterSetViewModel == null)
            {
                return false;
            }

            // 名前入力ダイアログの表示フラグを記録
            if (args[2] is bool == false)
            {
                return false;
            }

            this.showNameInputDialog = (bool)args[2];

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return null;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            bool result = this.workspaceViewModel.SaveEmitterSetFile(
                this.emitterSetViewModel,
                this.showNameInputDialog);

            // メニューの状態を更新
            WorkspaceRootViewModel.Instance.UpdateUIStates();

            return result;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            return false;
        }

        /// <summary>
        /// Redo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Redo()
        {
            return false;
        }
    }
}
