﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Extensions;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.UILogic.ViewModels;
using EffectMaker.UILogic.ViewModels.IO;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// エミッタセットを開き直すコマンドです。
    /// </summary>
    [Alias("ReopenEmitterSet")]
    public class ReopenEmitterSetCommand : CommandBase
    {
        /// <summary>
        /// Stores the command stack target.
        /// </summary>
        private object commandStackTarget;

        /// <summary>
        /// Stores the ViewModel given as parameter.
        /// </summary>
        private EmitterSetViewModel viewModel;

        /// <summary>
        /// 再読み込み元のファイルパス
        /// </summary>
        private string filePath;

        /// <summary>
        /// Stores a clone of the DataModel of the ViewModel given as parameter.
        /// </summary>
        private EmitterSetData currentDataModel;

        /// <summary>
        /// Stores the new DataModel given as parameter.
        /// </summary>
        private EmitterSetData newDataModel;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public ReopenEmitterSetCommand()
        {
            // アンドゥ禁止
            this.CanUndo = false;
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="viewModel">エミッタセットのビューモデル</param>
        /// <param name="filePath">再読み込み元のファイルパス</param>
        public ReopenEmitterSetCommand(EmitterSetViewModel viewModel, string filePath)
            : this()
        {
            object[] args = new object[] { viewModel, filePath };

            this.Initialize(args);
        }

        /// <summary>
        /// Initializes the SetDataModel command.
        /// </summary>
        /// <param name="args">Command arguments.</param>
        /// <returns>Returns true on success, false otherwise.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 2)
            {
                return false;
            }

            this.viewModel = args[0] as EmitterSetViewModel;
            if (this.viewModel == null)
            {
                return false;
            }

            this.filePath = args[1] as string;
            if (string.IsNullOrEmpty(this.filePath))
            {
                return false;
            }

            // 実際にロードするファイルのパスを取得
            // FE1からアップデートしたファイルを読み込むための例外的な処理で、
            // entityFilePathはデシリアライズにのみ使用する
            string entityFilePath = AncientEsetUpdater.Instance.GetFe2FilePath(this.filePath);
            if (string.IsNullOrEmpty(entityFilePath))
            {
                return false;
            }

            EmitterSetDataSerializer serializer = new EmitterSetDataSerializer();

            // デシリアライズを実行
            var result = serializer.DeserializeFromFile(entityFilePath);
            this.newDataModel = result.DataModel;

            // エラー処理
            if (result.IsSuccess == false)
            {
                return false;
            }

            using (new EnableDataModelCloneSetter())
            {
                this.currentDataModel = (EmitterSetData)this.viewModel.DataModel.Clone();
            }

            // エミッタセットの名前を設定
            this.newDataModel.Name = Path.GetFileNameWithoutExtension(this.filePath);
            this.newDataModel.FilePath = this.currentDataModel.FilePath;

            // ユーザー定義データが置き換えられたとき、警告ダイアログを表示する
            if (result.AnyUserDefinitionDataReplaced)
            {
                WorkspaceRootViewModel.Instance.Dialogs.ShowEmitterUserDataDefinitionReplaced(this.newDataModel.Name);
            }

            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(this.viewModel);

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// Gets the command target.
        /// </summary>
        /// <returns>Returns the command target.</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Sets the new DataModel into the ViewModel and sends PropertyChange requests.
        /// </summary>
        /// <returns>Returns true on success or throw an exception.</returns>
        public override bool Do()
        {
            this.viewModel.IsSelected = true;

            using (new MessageDropSection())
            {
                ExportableViewModel.IsPasting = true;

                // DataModelの差し替え
                DataModelBase dataModel = this.viewModel.DataModel;

                using (new EnableDataModelCloneSetter())
                {
                    dataModel.SetWithoutGuid(this.newDataModel);
                }

                // Update the child view models with the new data model.
                this.viewModel.UpdateChildViewModels();

                foreach (var previewData in this.currentDataModel.PreviewList)
                {
                    PreviewData clonedPreviewData;
                    using (new EnableDataModelCloneSetter())
                    {
                        clonedPreviewData = (PreviewData)previewData.Clone();
                    }

                    var previewViewModel = new PreviewViewModel(this.viewModel, clonedPreviewData);
                    this.viewModel.Proxy.AddPreview(clonedPreviewData);
                    this.viewModel.AddPreviewViewModel(previewViewModel);
                }

                // プロパティ変更を通知
                this.viewModel.FirePropertyChanges();

                // メニューの状態を更新
                WorkspaceRootViewModel.Instance.UpdateUIStates();

                // カーブエディタウィンドウの更新
                WorkspaceRootViewModel.Instance.UpdateCurveEditorWindow(this.viewModel);

                ExportableViewModel.IsPasting = false;
            }

            // アセットのロケートし直し
            this.viewModel.FilePath = this.currentDataModel.FilePath;

            // Send the whole binary again.
            using (new ForceRefreshBinary())
            {
                ViewerMessageHelper.FindPropertyParentDataAndSend(this.viewModel.DataModel);
            }

            // フラグをクリア
            this.viewModel.ModificationFlagViewModel.ClearChildModificationFlags();
            this.viewModel.ModificationFlagViewModel.ClearModificationFlags();

            // コマンドスタックをクリア
            CommandManager.ClearCommandStack();

            return true;
        }

        /// <summary>
        /// Sets the previous DataModel into the ViewModel and sends PropertyChange requests.
        /// </summary>
        /// <returns>Returns true on success or throw an exception.</returns>
        public override bool Undo()
        {
            return false;
        }
    }
}
