﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// LoadPreviewCommand.
    /// </summary>
    [Alias("LoadPreview")]
    public class LoadPreviewCommand : CommandBase
    {
        /// <summary>
        /// ワークスペースのビューモデルです.
        /// </summary>
        private WorkspaceViewModel workspaceViewModel;

        /// <summary>
        /// プレビューと一緒に読み込んだエミッタセットのリストです.
        /// </summary>
        private List<string> loadedEmitterSetList;

        /// <summary>
        /// プレビューのファイルパスです.
        /// </summary>
        private string filePath;

        /// <summary>
        /// 追加先のエミッタセットビューモデルです。
        /// </summary>
        private EmitterSetViewModel emitterSetViewModel;

        /// <summary>
        /// 読み込み先のプレビュービューモデルです。
        /// </summary>
        private PreviewViewModel previewViewModel;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public LoadPreviewCommand()
        {
            this.CanUndo = false;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="workspaceViewModel">ワークスペースのビューモデル</param>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="loadedEmitterSetList">プレビューと一緒に読み込んだエミッタセットのリスト</param>
        /// <param name="emitterSetViewModel">追加先のエミッタセットビューモデル</param>
        /// <param name="previewViewModel">読み込み先のプレビュービューモデル</param>
        public LoadPreviewCommand(WorkspaceViewModel workspaceViewModel, string filePath, List<string> loadedEmitterSetList, EmitterSetViewModel emitterSetViewModel, PreviewViewModel previewViewModel)
        {
            object[] args = new object[] { workspaceViewModel, filePath, loadedEmitterSetList, emitterSetViewModel, previewViewModel };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">The arguments for the command.</param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 5)
            {
                return false;
            }

            // ワークスペースのビューモデル
            this.workspaceViewModel = args[0] as WorkspaceViewModel;

            if (this.workspaceViewModel == null)
            {
                return false;
            }

            // ファイルパス
            this.filePath = args[1] as string;

            if (string.IsNullOrEmpty(this.filePath))
            {
                return false;
            }

            // プレビューと一緒に読み込んだエミッタセットのリスト
            this.loadedEmitterSetList = args[2] as List<string>;

            // 追加先のエミッタセットビューモデル
            this.emitterSetViewModel = args[3] as EmitterSetViewModel;

            // 読み込み先のプレビュービューモデル
            this.previewViewModel = args[4] as PreviewViewModel;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return null;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            // プレビュービューモデルを読み込む
            var prevVM = this.workspaceViewModel.LoadPreviewFile(
                this.filePath,
                this.loadedEmitterSetList,
                this.emitterSetViewModel,
                this.previewViewModel);

            if (prevVM == null)
            {
                return false;
            }

            // ビューアへ転送
            using (new ForceRefreshBinary())
            {
                ViewerMessageHelper.FindPropertyParentDataAndSend(prevVM.DataModel);
            }

            // 作成したプレビューを選択状態に.
            if (!prevVM.IsSelected)
            {
                prevVM.IsSelected = true;
            }

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            return false;
        }

        /// <summary>
        /// Redo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Redo()
        {
            return false;
        }
    }
}
