﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Text;

namespace EffectMaker.SpecGeneric.Shader
{
    /// <summary>
    /// シェーダコンバータのダンプ情報です。
    /// </summary>
    public class ShaderConverterDumpInfo
    {
        /// <summary>
        /// シェーダ情報のタイトルです。
        /// </summary>
        public const string ShaderInformationTitle = "------------------------Shader Information------------------------";

        /// <summary>
        /// 統計情報のタイトルです。
        /// </summary>
        public const string ShaderStatisticsTitle = "------------------------Shader Statistics------------------------";

        /// <summary>
        /// ダンプ情報のタイトルです。
        /// </summary>
        public const string DumpTitle = "------------------------Dump------------------------";

        /// <summary>
        /// アセンブリデータのタイトルです。
        /// </summary>
        public const string AssemblyTitle = "------------------------Assembly------------------------";

        /// <summary>
        /// GLSLCオプションフラグリストのタイトルを取得します。
        /// </summary>
        public const string GlslcOptionFlagsTitle = "------------------------GLSLC Option Flags------------------------";

        /// <summary>
        /// 情報ログのタイトルを取得します。
        /// </summary>
        public const string InfoLogTitle = "------------------------Info Log------------------------";

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderConverterDumpInfo()
        {
        }

        /// <summary>
        /// ダンプファイルを開きます。
        /// </summary>
        /// <param name="dumpFilePath">ダンプファイルパス</param>
        /// <returns>ダンプファイルを開いたときtrue、開けなかったときfalseを返します。</returns>
        public bool Open(string dumpFilePath)
        {
            string dumpText;

            try
            {
                dumpText = File.ReadAllText(dumpFilePath);
            }
            catch
            {
                return false;
            }

            dumpText = dumpText.Replace("\r\n", "\n");
            dumpText = dumpText.Replace("\r", "\n");
            dumpText = dumpText.Replace("\n", "\r\n");

            // ブロック情報リストを作成
            BlockInfo[] blockInfoList = new BlockInfo[]
            {
                new BlockInfo(ShaderInformationTitle),
                new BlockInfo(ShaderStatisticsTitle ),
                new BlockInfo(DumpTitle             ),
                new BlockInfo(AssemblyTitle         ),
                new BlockInfo(GlslcOptionFlagsTitle ),
                new BlockInfo(InfoLogTitle          ),
            };

            // ブロックごとのインデックスを取得
            foreach (BlockInfo info in blockInfoList)
            {
                info.Index = dumpText.IndexOf(info.Title + "\r\n");
            }

            // ブロック情報をインデックス順にソート
            blockInfoList = blockInfoList.OrderBy(x => x.Index).ToArray();

            // ブロックごとにテキストを切り出す
            for (int i = 0; i < blockInfoList.Length; ++i)
            {
                BlockInfo info = blockInfoList[i];

                if (info.Index != -1)
                {
                    int index = info.Index + info.Title.Length + 2;
                    int nextIndex = (i + 1 < blockInfoList.Length ? blockInfoList[i + 1].Index : -1);

                    if (nextIndex != -1)
                    {
                        info.Text = dumpText.Substring(index, nextIndex - index).Trim();
                    }
                    else
                    {
                        info.Text = dumpText.Substring(index).Trim();
                    }
                }
            }

            // タイトルに対応したテキストを取得するローカル関数
            Func<string, string> getText = (string title) =>
            {
                return blockInfoList.First(x => x.Title == title).Text;
            };

            // ブロック情報からテキストを取得
            this.ShaderInformation = getText(ShaderInformationTitle);
            this.ShaderStatistics  = getText(ShaderStatisticsTitle);
            this.Dump              = getText(DumpTitle);
            this.Assembly          = getText(AssemblyTitle);
            this.GlslcOptionFlags  = getText(GlslcOptionFlagsTitle);
            this.InfoLog           = getText(InfoLogTitle);

            return true;
        }

        /// <summary>
        /// シェーダソースコードを取得します。
        /// </summary>
        /// <returns>シェーダソースコードを返します。</returns>
        public string GetShaderSourceCode()
        {
            return this.Dump + "\r\n";
        }

        /// <summary>
        /// アセンブリ情報を取得します。
        /// </summary>
        /// <returns>アセンブリ情報を返します。</returns>
        public string GetAssemblyInfo()
        {
            StringBuilder sb = new StringBuilder();

            sb.AppendLine(AssemblyTitle);
            sb.AppendLine(this.Assembly);
            sb.AppendLine();

            sb.AppendLine(ShaderInformationTitle);
            sb.AppendLine(this.ShaderInformation);
            sb.AppendLine();

            sb.AppendLine(ShaderStatisticsTitle);
            sb.AppendLine(this.ShaderStatistics);
            sb.AppendLine();

            sb.AppendLine(GlslcOptionFlagsTitle);
            sb.AppendLine(this.GlslcOptionFlags);
            sb.AppendLine();

            sb.AppendLine(InfoLogTitle);
            sb.AppendLine(this.InfoLog);

            return sb.ToString();
        }

        /// <summary>
        /// シェーダ情報を取得します。
        /// </summary>
        public string ShaderInformation { get; private set; } = string.Empty;

        /// <summary>
        /// 統計情報を取得します。
        /// </summary>
        public string ShaderStatistics { get; private set; } = string.Empty;

        /// <summary>
        /// ダンプ情報を取得します。
        /// </summary>
        public string Dump { get; private set; } = string.Empty;

        /// <summary>
        /// アセンブリデータを取得します。
        /// </summary>
        public string Assembly { get; private set; } = string.Empty;

        /// <summary>
        /// GLSLCオプションフラグリストを取得します。
        /// </summary>
        public string GlslcOptionFlags { get; private set; } = string.Empty;

        /// <summary>
        /// 情報ログを取得します。
        /// </summary>
        public string InfoLog { get; private set; } = string.Empty;

        /// <summary>
        /// ダンプファイルのブロック情報です。
        /// ダンプファイルをパースするためのデータです。
        /// </summary>
        private class BlockInfo
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="title">タイトル</param>
            public BlockInfo(string title)
            {
                this.Title = title;
                this.Index = -1;
                this.Text  = string.Empty;
            }

            /// <summary>
            /// タイトルを取得します。
            /// </summary>
            public string Title { get; private set; }

            /// <summary>
            /// インデックスを取得または設定します。
            /// </summary>
            public int Index { get; set; }

            /// <summary>
            /// テキストデータを取得または設定します。
            /// </summary>
            public string Text { get; set; }
        }
    }
}
