﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.BinaryHeaders.Helpers;
using EffectMaker.BusinessLogic.EffectCombinerEditor;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.RuntimeOptions;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic;
using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.BinaryData;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.ShaderConverterWrapper;
using ConvertShaderBinaryCallback = System.Action<bool, System.Collections.Generic.List<EffectMaker.ShaderConverterWrapper.ShaderCompileErrorInfo>, byte[], int, System.Collections.Generic.List<EffectMaker.ShaderConverterWrapper.ShaderConversionInputData>>;
using CompileShaderAssemblyCallback = System.Action<EffectMaker.DataModelLogic.ShaderCompileResult, string, string, string, string>;
using CompileShaderCompleteCallback = System.Action<System.Collections.Generic.List<long>, System.Action<bool>, EffectMaker.Foundation.Debugging.Profiling.ProfileTimer, EffectMaker.Foundation.Debugging.Profiling.ProfileTimer, bool, bool, byte[], long, int, long, int, System.Action<System.IO.Stream>>;

namespace EffectMaker.ShaderManager
{
    public class ShaderManagerCafe : IShaderManager
    {
        /// <summary>The converter folder path.</summary>
        private static readonly string ConverterFolderPath;

        /// <summary>The semaphore to prevent multiple threads to access the shader converter together.</summary>
        private static readonly SemaphoreSlim AsyncSemaphore = new SemaphoreSlim(1, 1);

        /// <summary>シェーダコンバートのキュー管理オブジェクト</summary>
        private static readonly AsyncTaskQueueJustOne AsyncAdjuster = new AsyncTaskQueueJustOne();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static ShaderManagerCafe()
        {
            ConverterFolderPath =
                Path.Combine(IOConstants.ExecutableFolderPath, @"Converter\");
        }

        /// <summary>
        /// ユーザー定義文字列の何番目を利用するかを設定・取得
        /// </summary>
        public uint UserDefineIndex { get; set; }

        /// <summary>
        /// Generate shader source code in the emitter.
        /// </summary>
        /// <param name="emitter">The emitter.</param>
        /// <param name="vertexShader">The generated vertex shader.</param>
        /// <param name="fragmentShader">The generated fragment shader.</param>
        /// <param name="computeShader">The generated compute shader.</param>
        /// <returns>True on success.</returns>
        public bool GenerateShaderSource(
            EmitterData emitter,
            out string vertexShader,
            out string fragmentShader,
            out string conputeShader)
        {
            vertexShader = null;
            fragmentShader = null;
            conputeShader = null;

            var conversionInputDataList = PrepareEmitterForShaderConversion(emitter);
            if (conversionInputDataList == null || conversionInputDataList.Count <= 0)
            {
                return false;
            }

            var context = this.GenerateShaderSourceInternal(
                conversionInputDataList[0],
                OptionStore.RuntimeOptions.ShaderConvertOption);

            vertexShader = context.VertexShaderSource;
            fragmentShader = context.FragmentShaderSource;
            conputeShader = context.ComputeShaderSource;

            return context.IsSuccess;
        }

        /// <summary>
        /// Compile shader in the emitter and generate shader assembly.
        /// </summary>
        /// <param name="emitter">The emitter.</param>
        /// <param name="completeCallback">The callback function when completed.</param>
        public async void GenrateShaderAssemblyAsync(
            EmitterData emitter,
            CompileShaderAssemblyCallback completeCallback)
        {
            if (completeCallback == null)
            {
                return;
            }

            var conversionInputDataList = PrepareEmitterForShaderConversion(emitter);
            if (conversionInputDataList == null || conversionInputDataList.Count <= 0)
            {
                completeCallback(ShaderCompileResult.Error, null, null, null, null);
                return;
            }

            // There can only be one thread that uses the shader converter at one time.
            await AsyncSemaphore.WaitAsync();

            var context = await this.AsyncCompileShaderAssemblyWrapper(
                conversionInputDataList[0],
                OptionStore.RuntimeOptions.ShaderConvertOption);

            AsyncSemaphore.Release();

            ShowShaderCompileError(context.CompileErrors);
            if (context.IsSuccess == false)
            {
                completeCallback(
                    ShaderCompileResult.Error,
                    context.VertexShaderSource,
                    context.FragmentShaderSource,
                    context.ComputeShaderSource,
                    context.LatencyInfo);
            }
            else if (context.CompileErrors.Count > 0)
            {
                completeCallback(
                    ShaderCompileResult.Warning,
                    context.VertexShaderSource,
                    context.FragmentShaderSource,
                    context.ComputeShaderSource,
                    context.LatencyInfo);
            }
            else
            {
                completeCallback(
                    ShaderCompileResult.Success,
                    context.VertexShaderSource,
                    context.FragmentShaderSource,
                    context.ComputeShaderSource,
                    context.LatencyInfo);
            }
        }

        /// <summary>
        /// Write shader binary data.
        /// </summary>
        /// <param name="binaryName">バイナリー名</param>
        /// <param name="stream">バイナリーストリーム</param>
        /// <param name="enableAsync">True to compile shader asynchronously.</param>
        /// <param name="completeCallback">The callback to execute when finished.</param>
        /// <param name="emitterContexts">エミッタのバイナリーコンテキスト</param>
        /// <param name="handleShaderCompileComplete"></param>
        /// <returns>True on success.</returns>
        public bool WriteShaderResourcesImpl(
            string binaryName,
            Stream stream,
            string shaderBinaryOutputFilePath,
            string computeShaderBinaryOutputFilePath,
            bool enableAsync,
            Action<bool> completeCallback,
            IEnumerable<WriteBinaryDataContext> emitterContexts,
            CompileShaderCompleteCallback handleShaderCompileComplete)
        {
            var profiler = new ProfileTimer("ShaderSrc");

            // Prepare the information needed for shader binary conversion.
            var conversionInputDataList = MakeShaderConversionInputDataList(stream, emitterContexts);

            // Do we have any emitter?
            if (conversionInputDataList.Count <= 0)
            {
                return true;
            }

            var emitterOffsets = new List<long>();
            foreach (WriteBinaryDataContext context in emitterContexts)
            {
                var emitter = context.BinaryStruct.DataModel;
                var helper = BinaryHeaderHelperSelector.GetHelper(emitter);
                var offset = helper.GetOffset(emitter, (uint)context.WritePosition);
                emitterOffsets.Add(context.WritePosition + offset);
            }

            if (enableAsync == true)
            {
                // シェーダコンパイルが完了したときに呼び出されるコールバックアクションを作成
                var callback = new Action<bool, List<ShaderCompileErrorInfo>, byte[], int, List<ShaderConversionInputData>>((rr, ee, bb, cc, ii) =>
                {
                    ShowShaderCompileError(ee);

                    Action<Stream> writeData = (st) =>
                    {
                        int emitterIndex = 0;
                        foreach (long emitterOffset in emitterOffsets)
                        {
                            ShaderConversionInputData data = ii[emitterIndex];
                            var shaderIndexOffset = emitterOffset + data.ShaderIndexOffset;

                            // 現在のバイナリはUserPixelShaderIndex1, UserPixelShaderIndex2が削除されているため、
                            // Cafe向けには正しいバイナリが出力されません
                            st.Seek(shaderIndexOffset, SeekOrigin.Begin);
                            BinaryConversionUtility.ForResource.WriteStream(st, data.VertexShaderIndex);
                            BinaryConversionUtility.ForResource.WriteStream(st, data.PixelShaderIndex);
                            BinaryConversionUtility.ForResource.WriteStream(st, data.UserVertexShaderIndex1);
                            BinaryConversionUtility.ForResource.WriteStream(st, data.UserPixelShaderIndex1);
                            BinaryConversionUtility.ForResource.WriteStream(st, data.UserVertexShaderIndex2);
                            BinaryConversionUtility.ForResource.WriteStream(st, data.UserPixelShaderIndex2);

                            ++emitterIndex;
                        }
                    };

                    // エラーはsuccessで、警告はレポートの有無で判定
                    bool shaderCompileFailed = ee.Any();
                    long bbLength = (long)(bb != null ? bb.Length : 0);

                    handleShaderCompileComplete(
                        emitterOffsets,
                        completeCallback,
                        profiler,
                        null,
                        rr,
                        shaderCompileFailed,
                        bb,
                        bbLength,
                        cc,
                        0,
                        0,
                        writeData);
                });

                // シェーダコンパイルを非同期で実行
                this.ConvertShaderBinaryAsync(
                    binaryName,
                    conversionInputDataList,
                    OptionStore.RuntimeOptions.ShaderConvertOption,
                    OptionStore.RuntimeOptions.IsCommandLineMode,
                    OptionStore.RuntimeOptions.JobsNumber,
                    SpecManager.CurrentSpec.ShaderConversionOption.BinaryAlignment,
                    (int)stream.Length,
                    OptionStore.RuntimeOptions.IsSubBinaryConverting,
                    callback);

                return true;
            }
            else
            {
                List<ShaderCompileErrorInfo> errorList;
                byte[] shaderBinary;
                int shaderCount;

                // Compile shader in the same thread.
                bool success = this.ConvertShaderBinary(
                    conversionInputDataList,
                    OptionStore.RuntimeOptions.ShaderConvertOption,
                    OptionStore.RuntimeOptions.IsCommandLineMode,
                    OptionStore.RuntimeOptions.JobsNumber,
                    SpecManager.CurrentSpec.ShaderConversionOption.BinaryAlignment,
                    (int)stream.Length,
                    OptionStore.RuntimeOptions.IsSubBinaryConverting,
                    out errorList,
                    out shaderBinary,
                    out shaderCount);

                // Handle the completion of the compile.
                ShowShaderCompileError(errorList);

                Action<Stream> writeData = (st) =>
                {
                    int emitterIndex = 0;
                    foreach (long emitterOffset in emitterOffsets)
                    {
                        ShaderConversionInputData data = conversionInputDataList[emitterIndex];
                        var shaderIndexOffset = emitterOffset + data.ShaderIndexOffset;

                        // 現在のバイナリはUserPixelShaderIndex1, UserPixelShaderIndex2が削除されているため、
                        // Cafe向けには正しいバイナリが出力されません
                        st.Seek(shaderIndexOffset, SeekOrigin.Begin);
                        BinaryConversionUtility.ForResource.WriteStream(st, data.VertexShaderIndex);
                        BinaryConversionUtility.ForResource.WriteStream(st, data.PixelShaderIndex);
                        BinaryConversionUtility.ForResource.WriteStream(st, data.UserVertexShaderIndex1);
                        BinaryConversionUtility.ForResource.WriteStream(st, data.UserPixelShaderIndex1);
                        BinaryConversionUtility.ForResource.WriteStream(st, data.UserVertexShaderIndex2);
                        BinaryConversionUtility.ForResource.WriteStream(st, data.UserPixelShaderIndex2);

                        ++emitterIndex;
                    }
                };

                // エラーはsuccessで、警告はレポートの有無で判定
                bool shaderCompileFailed = errorList.Any();

                handleShaderCompileComplete(
                    emitterOffsets,
                    completeCallback,
                    profiler,
                    null,
                    success,
                    shaderCompileFailed,
                    shaderBinary,
                    (long)shaderBinary.Length,
                    shaderCount,
                    0,
                    0,
                    writeData);

                return success;
            }
        }

        /// <summary>
        /// Display error log and shader code when shader compile failed.
        /// </summary>
        /// <param name="errorList">The error information list.</param>
        private static void ShowShaderCompileError(List<ShaderCompileErrorInfo> errorList)
        {
            if (errorList == null || errorList.Count <= 0)
            {
                return;
            }

            // 30個以下になるように調整
            PathUtility.DeleteSubDirectoriesOverNum(IOConstants.ShaderErrorLogFolderPath, 30);

            // Save files in [Executable Folder]/Converter/shader_error/PID
            string outputFolderPath = Path.Combine(
                IOConstants.ShaderErrorLogFolderPath, Process.GetCurrentProcess().Id.ToString());

            // 空のフォルダを作成
            if (IOUtility.SafeCreateEmptyDirectory(outputFolderPath) == false)
            {
                return;
            }

            List<string> filePathsForTextEditor = new List<string>();

            // Save error log and shader source code to file.
            int index = 0;
            foreach (ShaderCompileErrorInfo error in errorList)
            {
                string errorLog = error.EmitterName + Environment.NewLine + error.ErrorLog;

                int retryCount = 0;
                bool retry = false;
                do
                {
                    string filePath = Path.Combine(
                        outputFolderPath,
                        string.Format("{0}_log{1}", error.EmitterName, index));

                    retry = false;
                    try
                    {
                        File.WriteAllText(filePath + ".log", errorLog);
                        File.WriteAllText(filePath + ".vsh", error.VertexShader);
                        File.WriteAllText(filePath + ".fsh", error.FragmentShader);
                    }
                    catch
                    {
                        // Failed to save the files as the file name, try
                        retry = true;
                        ++retryCount;
                        ++index;
                    }

                    // Do not show the files in the text editor if the message is just an warning.
                    if (error.IsWarning == false &&
                        retry == false)
                    {
                        // Remember the error log file path, so we can open it
                        // with external text editor later.
                        filePathsForTextEditor.Add(filePath + ".log");
                    }
                }
                while (retry == true && retryCount < 10);

                ++index;
            }

            if (OptionStore.RuntimeOptions.IsCommandLineMode == false)
            {
                // Open the error log files with external text editor.
                foreach (string filePath in filePathsForTextEditor)
                {
                    ApplicationIOManager.OpenTextFileWithExternalTextEditor(filePath);
                }
            }
        }

        /// <summary>
        /// Prepare emitter binary data for shader conversion.
        /// </summary>
        /// <param name="emitter">The emitter data.</param>
        /// <returns>The data for shader conversion.</returns>
        private static List<ShaderConversionInputData> PrepareEmitterForShaderConversion(EmitterData emitter)
        {
            using (var stream = new MemoryStream())
            {
                // Find the binary data for the data model.
                DataModelBase dataModel = emitter;

                BinaryStructInstance binaryData =
                    BinaryConversionInfoManager.FindBinaryData(dataModel);

                while (binaryData == null && dataModel.Parent != null)
                {
                    dataModel = dataModel.Parent;
                    binaryData = BinaryConversionInfoManager.FindBinaryData(dataModel);
                }

                if (binaryData == null)
                {
                    return null;
                }

                Stream childStream;
                IEnumerable<WriteBinaryDataContext> emitterContexts;
                if (!WriteBinaryDataSession.WriteBinaryForPrepareEmitter(
                    emitter,
                    stream,
                    binaryData,
                    out childStream,
                    out emitterContexts))
                {
                    return null;
                }

                // Make input data list for shader conversion.
                var conversionInputDataList =
                    MakeShaderConversionInputDataList(childStream, emitterContexts);

                // Do we have any emitter?
                if (conversionInputDataList.Count <= 0)
                {
                    return null;
                }

                return conversionInputDataList;
            }
        }

        /// <summary>
        /// Make a list of input data for shader conversion.
        /// </summary>
        /// <param name="stream">セッションのストリーム</param>
        /// <param name="emitterContexts">WriteBinaryDataContexts of emitters.</param>
        /// <returns>The list of input data for shader conversion.</returns>
        private static List<ShaderConversionInputData> MakeShaderConversionInputDataList(
            Stream stream,
            IEnumerable<WriteBinaryDataContext> emitterContexts)
        {
            uint binHeaderSize = BinaryStructHeader.Size;

            var conversionInputDataList = new List<ShaderConversionInputData>();

            // Prepare the information needed for shader binary conversion.
            foreach (WriteBinaryDataContext context in emitterContexts)
            {
                var tmpHeader = new byte[binHeaderSize];
                var tmpBuffer = new byte[context.WriteSize - binHeaderSize];

                stream.Seek(context.WritePosition, SeekOrigin.Begin);
                stream.Read(tmpHeader, 0, tmpHeader.Length);

                stream.Seek(context.WritePosition + (long)binHeaderSize, SeekOrigin.Begin);
                stream.Read(tmpBuffer, 0, tmpBuffer.Length);

                // Get the effect combiner project path set in the emitter data.
                var emitter = context.BinaryStruct.DataModel as EmitterData;

                // Get the ID of the reserved shader the emitter uses.
                int reservedShaderIndex = -1;
                if (emitter.ActiveReservedShader != null &&
                    emitter.ActiveReservedShader.PageData != null &&
                    emitter.ActiveReservedShader.PageData.ContentsData != null)
                {
                    reservedShaderIndex = ReservedShaderUserDataManager.FindReservedShaderIndexByUserData(
                        emitter.ActiveReservedShader.PageData.ContentsData);
                }

                // Get the effect combiner project path set in the emitter data.
                string combinerShaderCode = string.Empty;
                string combinerProjectPath = emitter.EmitterCombinerData.EmitterCombinerEditorData.CombinerEditorProjectPath;

                if (OptionStore.ProjectConfig.IsEftCombinerEditorEnabled == true &&
                    string.IsNullOrEmpty(combinerProjectPath) == false)
                {
                    var combiner = EffectCombinerCommunicationManager.CommunicationBridge;
                    if (combiner != null)
                    {
                        combinerShaderCode = combiner.GetShaderSourceCode(combinerProjectPath);
                    }
                }

                // Create the shader conversion input data.
                var data = new ShaderConversionInputData()
                {
                    BinaryHeader = tmpHeader,
                    EmitterBinary = tmpBuffer,
                    OverrideShaderCode = combinerShaderCode,
                    ReservedShaderIndex = reservedShaderIndex,
                };

                conversionInputDataList.Add(data);
            }

            return conversionInputDataList;
        }

        /// <summary>
        /// Asynchronously create shader binary data with the shader converter.
        /// (executed at another thread, without blocking the current execution)
        /// </summary>
        /// <param name="binaryName">バイナリー名</param>
        /// <param name="conversionInputDataList">The input data for the conversion.</param>
        /// <param name="option">The option for the shader conversion.</param>
        /// <param name="isCommandLineMode">コマンドラインモードかどうか</param>
        /// <param name="jobsNumber">ジョブ数</param>
        /// <param name="shaderAlignment">シェーダアライメント</param>
        /// <param name="binaryPosition">バイナリ書き込み位置</param>
        /// <param name="completeCallback">
        /// The callback method to be called when complete.
        /// The parameters are :
        /// - bool : True on success; otherwise, false.
        /// - List[ShaderCompileErrorInfo] : A list of shader compile errors.
        /// - byte[] : The generated shader assemblies.
        /// - List[ShaderConversionInputData] : The same input data sequence that is passed in.
        /// </param>
        private void ConvertShaderBinaryAsync(
            string binaryName,
            List<ShaderConversionInputData> conversionInputDataList,
            ShaderConvertOption option,
            bool isCommandLineMode,
            int jobsNumber,
            int shaderAlignment,
            int binaryPosition,
            bool isSubBinaryConverting,
            ConvertShaderBinaryCallback completeCallback)
        {
            // Cafeのシェーダアライメントは256バイトにすべし
            if (shaderAlignment != 256)
            {
                Logger.Log("LogView", LogLevels.Error, Properties.Resources.ShaderManagerMsg_Alignment);
            }

            AsyncAdjuster.RunOrQueue(
                binaryName,
                () =>
                {
                    // Compile shader binary!
                    ShaderCompileContext context = this.ConvertShaderBinaryInternal(
                        conversionInputDataList,
                        option,
                        isCommandLineMode,
                        jobsNumber,
                        shaderAlignment,
                        binaryPosition,
                        isSubBinaryConverting);

                    if (completeCallback != null)
                    {
                        // Execute the callback function.
                        completeCallback(
                            context.IsSuccess,
                            context.CompileErrors,
                            context.ShaderBinary,
                            context.ShaderCount,
                            context.InputDataList);
                    }
                },
                () => completeCallback(
                    false,
                    new List<ShaderCompileErrorInfo>(),
                    null,
                    1,
                    null));
        }

        /// <summary>
        /// Synchronously compile shader assembly with the shader converter.
        /// (executed at the same thread, blocks the current execution)
        /// </summary>
        /// <param name="conversionInputDataList">The input data for the conversion.</param>
        /// <param name="option">The option for the shader conversion.</param>
        /// <param name="isCommandLineMode">コマンドラインモードかどうか</param>
        /// <param name="shaderAlignment">シェーダアライメント</param>
        /// <param name="binaryPosition">バイナリ書き込み位置</param>
        /// <param name="errorList">The error messages generated when compiling shader.</param>
        /// <param name="binaryData">The compiled shader binary.</param>
        /// <param name="shaderCount">The number of shaders being compiled.</param>
        /// <param name="jobsNumber">ジョブ数</param>
        /// <returns>True on success.</returns>
        private bool ConvertShaderBinary(
            List<ShaderConversionInputData> conversionInputDataList,
            ShaderConvertOption option,
            bool isCommandLineMode,
            int jobsNumber,
            int shaderAlignment,
            int binaryPosition,
            bool isSubBinaryConverting,
            out List<ShaderCompileErrorInfo> errorList,
            out byte[] binaryData,
            out int shaderCount)
        {
            // Cafeのシェーダアライメントは256バイトにすべし
            if (shaderAlignment != 256)
            {
                Logger.Log("LogView", LogLevels.Error, Properties.Resources.ShaderManagerMsg_Alignment);
            }

            // Compile shader binary!
            ShaderCompileContext context = this.ConvertShaderBinaryInternal(
                conversionInputDataList,
                option,
                isCommandLineMode,
                jobsNumber,
                shaderAlignment,
                binaryPosition,
                isSubBinaryConverting);

            errorList = context.CompileErrors;
            binaryData = context.ShaderBinary;
            shaderCount = context.ShaderCount;

            return context.IsSuccess;
        }

        /// <summary>
        /// Wraps CompileShaderAssemblyInternal() to make it run on another thread. (asychronously)
        /// </summary>
        /// <param name="conversionInputData">The input data for the conversion.</param>
        /// <param name="option">The option for the shader conversion.</param>
        /// <returns>The compile result.</returns>
        private Task<ShaderCompileContext> AsyncCompileShaderAssemblyWrapper(
            ShaderConversionInputData conversionInputData,
            ShaderConvertOption option)
        {
            return Task.Run<ShaderCompileContext>(() => this.CompileShaderAssemblyInternal(
                conversionInputData,
                option));
        }

        /// <summary>
        /// The internal method for converting shader binary.
        /// </summary>
        /// <param name="conversionInputDataList">The input data for the conversion.</param>
        /// <param name="option">The option for the shader conversion.</param>
        /// <param name="isCommandLineMode">コマンドラインモードかどうか</param>
        /// <param name="jobsNumber">ジョブ数</param>
        /// <param name="shaderAlignment">シェーダアライメント</param>
        /// <param name="binaryPosition">バイナリ書き込み位置</param>
        /// <returns>The compile result.</returns>
        private ShaderCompileContext ConvertShaderBinaryInternal(
            List<ShaderConversionInputData> conversionInputDataList,
            ShaderConvertOption option,
            bool isCommandLineMode,
            int jobsNumber,
            int shaderAlignment,
            int binaryPosition,
            bool isSubBinaryConverting)
        {
            bool result;
            int shaderCount;
            byte[] shaderBinary;
            List<ShaderCompileErrorInfo> errorList;

            using (var converter = new ShaderConverterHelper(
                ConverterFolderPath,
                IOConstants.DefaultShaderFolderPath,
                new ShaderConverterMessageForwarder(),
                (int)option,
                isCommandLineMode,
                jobsNumber,
                shaderAlignment,
                binaryPosition,
                isSubBinaryConverting))
            {
                // Set shader source code to the shader converter.
                converter.SetShaderCodes(
                    this.CopyShaderCodeForShaderConversion());

                result = converter.Convert(
                    conversionInputDataList.ToArray(),
                    out shaderBinary,
                    out errorList,
                    out shaderCount);
            }

            // C++側のConvert関数は、シェーダ出力用のバッファが確保できてさえいればtrueを返す。
            // なので、警告ではないエラーレポートの有無で成否を判断する必要がある。
            if (errorList != null && errorList.Any(e => !e.IsWarning))
            {
                result = false;
            }

            return new ShaderCompileContext(
                result,
                errorList,
                shaderBinary,
                shaderCount,
                conversionInputDataList);
        }

        /// <summary>
        /// The internal method for generating shader source code.
        /// </summary>
        /// <param name="conversionInputData">The input data for the conversion.</param>
        /// <param name="option">The option for the shader conversion.</param>
        /// <returns>The result of the source code generation.</returns>
        private ShaderCompileContext GenerateShaderSourceInternal(
            ShaderConversionInputData conversionInputData,
            ShaderConvertOption option)
        {
            bool result;
            string vertexShader;
            string fragmentShader;
            string computeShader;

            using (var converter = new ShaderConverterHelper(
                ConverterFolderPath,
                IOConstants.DefaultShaderFolderPath,
                new ShaderConverterMessageForwarder(),
                (int)option,
                false,
                1,
                1,
                0,
                false))
            {
                // Set shader source code to the shader converter.
                converter.SetShaderCodes(
                    this.CopyShaderCodeForShaderConversion());

                conversionInputData.UserDefineIndex = this.UserDefineIndex;

                result = converter.GenerateShaderCode(
                    conversionInputData,
                    out vertexShader,
                    out fragmentShader,
                    out computeShader);
            }

            return new ShaderCompileContext(
                result,
                new List<ShaderCompileErrorInfo>(),
                vertexShader,
                fragmentShader,
                computeShader,
                null,
                conversionInputData);
        }

        /// <summary>
        /// The internal method for compiling shader assembly.
        /// </summary>
        /// <param name="conversionInputData">The input data for the conversion.</param>
        /// <param name="option">The option for the shader conversion.</param>
        /// <returns>The compile result.</returns>
        private ShaderCompileContext CompileShaderAssemblyInternal(
            ShaderConversionInputData conversionInputData,
            ShaderConvertOption option)
        {
            bool result;
            string vertexShader;
            string fragmentShader;
            string computeShader;
            List<ShaderCompileErrorInfo> errorList;

            using (var converter = new ShaderConverterHelper(
                ConverterFolderPath,
                IOConstants.DefaultShaderFolderPath,
                new ShaderConverterMessageForwarder(),
                (int)option,
                false,
                1,
                1,
                0,
                false))
            {
                // Set shader source code to the shader converter.
                converter.SetShaderCodes(
                    this.CopyShaderCodeForShaderConversion());

                conversionInputData.UserDefineIndex = this.UserDefineIndex;

                result = converter.GenerateShaderAssembly(
                    conversionInputData,
                    out vertexShader,
                    out fragmentShader,
                    out computeShader,
                    out errorList);
            }

            // アセンブリダンプからおおよその命令数を抽出する

            int vLatency = -1;

            if (string.IsNullOrEmpty(vertexShader) == false)
            {
                var vLines = vertexShader.Split(new[] { Environment.NewLine }, StringSplitOptions.None);
                foreach (var vLine in vLines)
                {
                    if (vLine.Contains("END_OF_PROGRAM"))
                    {
                        vLatency = vLines.FindIndex(vLine) - 1;
                        break;
                    }
                }
            }

            int pLatency = -1;

            if (string.IsNullOrEmpty(fragmentShader) == false)
            {
                var pLines = fragmentShader.Split(new[] { Environment.NewLine }, StringSplitOptions.None);
                foreach (var pLine in pLines)
                {
                    if (pLine.Contains("END_OF_PROGRAM"))
                    {
                        pLatency = pLines.FindIndex(pLine) - 1;
                        break;
                    }
                }
            }

            string latencyInfo = string.Format("Steps: VS({0}) / PS({1})", vLatency, pLatency);

            return new ShaderCompileContext(
                result,
                errorList,
                vertexShader,
                computeShader,
                fragmentShader,
                latencyInfo,
                conversionInputData);
        }

        /// <summary>
        /// Copy shader source code for shader conversion.
        /// </summary>
        /// <returns>The shader code for conversion.</returns>
        private ShaderCodeListCs CopyShaderCodeForShaderConversion()
        {
            ShaderCodeListCs shaderCodes = new ShaderCodeListCs();

            shaderCodes.SpecDecSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.SpecDeclarationTargetShader);

            shaderCodes.ParticleGlslSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.ParticleDeclarationShader);

            shaderCodes.ParticleDecVshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.ParticleDeclarationVertexShader);

            shaderCodes.ParticleDecFshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.ParticleDeclarationFragmentShader);

            shaderCodes.StreamOutDecVshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.StreamOutDeclarationVertexShader);

            shaderCodes.VertexShaderSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.ParticleVertexShader);

            shaderCodes.FragShaderSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.ParticleFragmentShader);

            shaderCodes.StreamOutVshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.StreamOutVertexShader);

            shaderCodes.GeneralVshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.GeneralVertexShader);

            shaderCodes.GeneralFshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.GeneralFragmentShader);

            shaderCodes.GeneralCshSrc = CustomShaderUserDataManager.GetShaderSourceCode(
                ShaderTypes.GeneralComputeShader);

            shaderCodes.CustomVshSrc = CustomShaderUserDataManager.GetCustomShaderSourceCodeArray(
                ShaderTypes.CustomVertexShader);

            shaderCodes.CustomFshSrc = CustomShaderUserDataManager.GetCustomShaderSourceCodeArray(
                ShaderTypes.CustomFragmentShader);

            shaderCodes.ReservedVshSrc = CustomShaderUserDataManager.GetReservedShaderSourceCodeArray(
                ShaderTypes.ReservedVertexShader);

            shaderCodes.ReservedFshSrc = CustomShaderUserDataManager.GetReservedShaderSourceCodeArray(
                ShaderTypes.ReservedFragmentShader);

            return shaderCodes;
        }

        /// <summary>
        /// Holds result and data generated when the shader is compiled.
        /// </summary>
        private class ShaderCompileContext
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="isSuccess">The result of the shader compilation.</param>
            /// <param name="errors">The error messages returned from the compiler.</param>
            /// <param name="shaderBinary">The compiled shader binary.</param>
            /// <param name="shaderCount">The number of shaders being compiled.</param>
            /// <param name="inputDataList">The shader conversion input data list.</param>
            public ShaderCompileContext(
                bool isSuccess,
                List<ShaderCompileErrorInfo> errors,
                byte[] shaderBinary,
                int shaderCount,
                List<ShaderConversionInputData> inputDataList)
            {
                this.IsSuccess = isSuccess;
                this.CompileErrors = errors;
                this.ShaderBinary = shaderBinary;
                this.ShaderCount = shaderCount;
                this.InputDataList = inputDataList;
                this.VertexShaderSource = string.Empty;
                this.FragmentShaderSource = string.Empty;
                this.LatencyInfo = string.Empty;
            }

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="isSuccess">The result of the shader compilation.</param>
            /// <param name="errors">The error messages returned from the compiler.</param>
            /// <param name="vertexSource">The source code of the vertex shader.</param>
            /// <param name="fragmentSource">The source code of the fragment shader.</param>
            /// <param name="computeSource">The source code of the compute shader.</param>
            /// <param name="latencyInfo">シェーダの負荷情報</param>
            /// <param name="inputData">The shader conversion input data.</param>
            public ShaderCompileContext(
                bool isSuccess,
                List<ShaderCompileErrorInfo> errors,
                string vertexSource,
                string fragmentSource,
                string computeShader,
                string latencyInfo,
                ShaderConversionInputData inputData)
            {
                this.IsSuccess = isSuccess;
                this.CompileErrors = errors;
                this.ShaderBinary = null;
                this.ShaderCount = 0;
                this.InputDataList = new List<ShaderConversionInputData>() { inputData };
                this.VertexShaderSource = vertexSource;
                this.FragmentShaderSource = fragmentSource;
                this.ComputeShaderSource = computeShader;
                this.LatencyInfo = latencyInfo;
            }

            /// <summary>
            /// Get the flag indicating whether the shader being compiled successfully.
            /// </summary>
            public bool IsSuccess { get; private set; }

            /// <summary>
            /// Get the error messages returned from the shader compiler.
            /// </summary>
            public List<ShaderCompileErrorInfo> CompileErrors { get; private set; }

            /// <summary>
            /// Get the generated shader binary.
            /// </summary>
            public byte[] ShaderBinary { get; private set; }

            /// <summary>
            /// Get the number of shaders being compiled.
            /// </summary>
            public int ShaderCount { get; private set; }

            /// <summary>
            /// Get the shader conversion input data list.
            /// </summary>
            public List<ShaderConversionInputData> InputDataList { get; private set; }

            /// <summary>
            /// Get the source code of the vertex shader.
            /// </summary>
            public string VertexShaderSource { get; private set; }

            /// <summary>
            /// Get the source code of the fragment shader.
            /// </summary>
            public string FragmentShaderSource { get; private set; }

            /// <summary>
            /// Get the source code of the compute shader.
            /// </summary>
            public string ComputeShaderSource { get; private set; }

            /// <summary>
            /// シェーダ負荷の指標となる情報を取得します。
            /// </summary>
            public string LatencyInfo { get; private set; }
        }
    }
}
