﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using BinaryConvertUtils;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Model;
using EffectMaker.Foundation.Model.Types;
using EffectMaker.PrimitiveManager.Cache;
using EffectMaker.PrimitiveManager.Loader;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using G3dIfLib;

namespace EffectMaker.PrimitiveManager
{
    /// <summary>
    /// プリミティブマネージャ
    /// </summary>
    public class PrimitiveManager
    {
        /// <summary>
        /// 出力するG3Dのバイナリモードです。
        /// </summary>
        public enum G3DBinaryMode
        {
            NnBinary,
            NwBinary,
        }

        /// <summary>
        /// モデルデータキャッシュ.
        /// </summary>
        private readonly ModelDataCache modelDataCache = new ModelDataCache();

        /// <summary>
        /// トリミングプリミティブデータキャッシュ.
        /// </summary>
        private readonly ModelDataCache trimmingPrimitiveDataCache = new ModelDataCache();

        /// <summary>
        /// モデルローダー
        /// </summary>
        private readonly IModelLoader[] modelLoaders = { new FmdModelLoader() };

        /// <summary>
        /// バイナリコンバータ
        /// </summary>
        private BinaryConverter converter;

        /// <summary>
        /// BNTX出力用のテクスチャリスト
        /// </summary>
        private readonly List<string> texturePathList = new List<string>();

        /// <summary>
        /// ファイルリロード後イベント
        /// </summary>
        public event EventHandler<FileReloadedEventArgs> FileReloaded;

        /// <summary>
        /// アプリケーションのフォルダパスを取得または設定します.
        /// BinaryConvertUtilsの初期化に使います.
        /// </summary>
        public string AppFolderPath { get; set; }

        /// <summary>
        /// モデルノードで転送するバイナリのモードを取得または設定します。
        /// </summary>
        public G3DBinaryMode BinaryMode { get; set; }

        /// <summary>
        /// BNTXを生成する処理を取得または設定します。
        /// </summary>
        public Func<IEnumerable<string>, string> ConvertBntx { get; set; }

        /// <summary>
        /// バイナリコンバータを取得します.
        /// </summary>
        internal BinaryConverter Converter
        {
            get
            {
                // 初回コール時はコンバータを初期化
                if (this.converter == null)
                {
                    this.converter = new BinaryConverter();

                    if (this.converter.Initialize(this.AppFolderPath, false) == false)
                    {
                        this.converter = null;
                    }
                }

                return this.converter;
            }
        }

        /// <summary>
        /// ファイルが読み込めるか調査する
        /// </summary>
        /// <param name="filePath">調査するファイルパス</param>
        /// <returns>読み込めるかどうか？</returns>
        public bool CanLoad(string filePath)
        {
            if (string.IsNullOrEmpty(filePath))
            {
                return false;
            }

            if (File.Exists(filePath) == false)
            {
                return false;
            }

            return this.modelLoaders.Any(x => x.CanLoad(filePath));
        }

        /// <summary>
        /// キャッシュされているか
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>キャッシュされているか.</returns>
        public bool IsCached(string filePath)
        {
            return this.modelDataCache.Get(filePath) != null;
        }

        /// <summary>
        /// キャッシュをフラッシュする.
        /// </summary>
        /// <param name="filePath">ファイルパス.</param>
        public void FlushCache(string filePath)
        {
            this.modelDataCache.Remove(filePath);
        }

        /// <summary>
        /// トリミング向けプリミティブのキャッシュをフラッシュする.
        /// </summary>
        public void FlushTrimmingPrimitiveCache()
        {
            this.trimmingPrimitiveDataCache.Clear();
        }

        /// <summary>
        /// リロードする.
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        public void Reload(string filePath)
        {
            if (this.IsCached(filePath) == false)
            {
                return;
            }

            this.FlushCache(filePath);
            this.LoadModel(filePath, true);

            if (this.FileReloaded != null)
            {
                var args = new FileReloadedEventArgs
                {
                    FilePath = filePath
                };

                this.FileReloaded(this, args);
            }
        }

        /// <summary>
        /// コメントを読み込みます。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>読み込み結果</returns>
        public LoadComment LoadComment(string filePath)
        {
            var result = new LoadComment();

            // ローダを探す
            var modelLoader = this.modelLoaders.FirstOrDefault(x => x.CanLoad(filePath));

            // ローダが見つからないので失敗を返す
            if (modelLoader == null)
            {
                return result;
            }

            // 実際に読み込んで結果を返す
            string comment, color;
            if (modelLoader.LoadComment(filePath, out comment, out color))
            {
                result.Comment = comment;
                result.Color = color;
            }

            return result;
        }

        /// <summary>
        /// コメントとラベルカラーを上書き保存します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="comment">コメント</param>
        /// <param name="color">ラベルカラー</param>
        /// <returns>成功したらtrue,失敗したらfalse.</returns>
        public bool SaveComment(string filePath, string comment, string color)
        {
            try
            {
                var loadResult = G3dIfUtility.LoadG3dIf(filePath);
                loadResult.G3dIf.RootElement.comment = new commentType()
                {
                    text = comment,
                    color = color,
                };
                IfWriteUtility.Write(loadResult.G3dIf, loadResult.BinaryStreams, filePath);
            }
            catch (Exception e)
            {
                Logger.Log(LogLevels.Warning, "exception:{0}", e);
                return false;
            }

            return true;
        }

        /// <summary>
        /// モデルを読み込みます。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>読み込み結果と読み込んだモデルデータ</returns>
        public LoadModelResultWithData LoadModel(string filePath)
        {
            // ローダを探す
            var modelLoader = this.modelLoaders.FirstOrDefault(x => x.CanLoad(filePath));

            // ローダが見つからないので失敗を返す
            if (modelLoader == null)
            {
                return new LoadModelResultWithData
                {
                    Result = LoadModelResults.UnknowModelType,
                    ModelData = null
                };
            }

            // 実際に読み込んで結果を返す
            ModelData modelData;
            return new LoadModelResultWithData
            {
                Result = modelLoader.Load(filePath, out modelData),
                ModelData = modelData
            };
        }

        /// <summary>
        /// モデルを読み込みます.
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="enableCache">キャッシュからロードする場合はtrueを指定.</param>
        /// <returns>読み込み結果と読み込んだモデルデータ.</returns>
        public LoadModelResultWithData LoadModel(string filePath, bool enableCache)
        {
            LoadModelResultWithData resultWithData;

            if (this.CanLoad(filePath) == false)
            {
                resultWithData = new LoadModelResultWithData
                {
                    Result = LoadModelResults.FailedLoadingModel,
                };
            }
            else
            {
                // キャッシュ読み込みするかどうか?
                if (enableCache)
                {
                    var modelData = this.modelDataCache.Get(filePath);

                    // キャッシュ済みでも日付が違っていたら読み直す
                    if (modelData != null)
                    {
                        if (modelData.UpdateTimestamp != File.GetLastWriteTime(filePath))
                        {
                            this.modelDataCache.Remove(filePath);
                            modelData = null;
                        }
                    }

                    if (modelData != null)
                    {
                        resultWithData = new LoadModelResultWithData
                        {
                            Result = LoadModelResults.Success,
                            ModelData = modelData.Data
                        };
                    }
                    else
                    {
                        // 新規に読み込む.
                        resultWithData = this.LoadModel(filePath);

                        // 成功時にキャッシュに収める.
                        if (resultWithData.Result == LoadModelResults.Success)
                        {
                            var data = new ModelDataCache.ModelFileData
                            {
                                Data = resultWithData.ModelData,
                                UpdateTimestamp = File.GetLastWriteTime(filePath)
                            };

                            this.modelDataCache.Add(filePath, data);
                        }
                    }
                }
                else
                {
                    // 新規に読み込み.
                    resultWithData = this.LoadModel(filePath);
                }
            }

            return resultWithData;
        }

        /// <summary>
        /// fmdbファイルをbfresバイナリにコンバートし、ストリームに出力します。
        /// </summary>
        /// <param name="filePaths">fmdbファイルのリスト</param>
        /// <param name="outStream">出力するストリーム</param>
        /// <returns>コンバート結果を返します。</returns>
        public LoadModelResultWithFilePath LoadFmdbModels(IEnumerable<string> filePaths, Stream outStream)
        {
            LoadModelResultWithFilePath result = new LoadModelResultWithFilePath();
            result.DetailMessages = new List<LoadModelMessage>();
            result.Result = LoadModelResults.Success;

            // バイナリコンバータが初期化済みかチェック
            if (this.Converter == null)
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FailedInitConverter));
                result.FilePath = string.Empty;
                return result;
            }

            // アニメーションなど複数に分かれたファイルを1つのバイナリにまとめるように設定
            int errCode = 0;

            errCode = this.Converter.PreBinarizeUsingIgnoreAssignOption();

            if (errCode != 0)
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                result.DetailMessages.Add(new LoadModelMessage(this.TranslateErrorCode(errCode)));
                result.FilePath = string.Empty;
                return result;
            }

            foreach (var filePath in filePaths)
            {
                result.FilePath = filePath;

                // ファイルパスをチェック
                if (string.IsNullOrEmpty(filePath) == true || Path.IsPathRooted(filePath) == false)
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.InvalidFilePath));

                    return result;
                }

                // ファイルがあるかどうかチェック
                if (File.Exists(filePath) == false)
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FileNotFound));

                    return result;
                }

                nw4f_3difType loadedObj = null;
                List<G3dStream> streams = new List<G3dStream>();

                // モデルの中間ファイルを読み込む
                try
                {
                    loadedObj = IfReadUtility.Read(streams, filePath.ToLower(), G3dIfUtility.XsdBasePath);
                }
                catch
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FailedOpeningFile));

                    return result;
                }

                // モデルが読み込めたかチェック
                if (loadedObj == null)
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FailedLoadingModel));

                    return result;
                }

                // モデルデータの内容をチェック
                modelType model = loadedObj.Item as modelType;
                if (model == null || model.shape_array == null)
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FailedModelNotFound, filePath));

                    return result;
                }

                // モデルファイルをコンバータに追加
                LoadModelMessageCodes addModelResult = this.AddFileToConverter(filePath,
                                                                                this.Converter,
                                                                                IfBinaryFormatter.FormatStreamFast(loadedObj, streams));
                if (addModelResult != LoadModelMessageCodes.Success)
                {
                    Version latestVersion = G3dIfUtility.GetLatestFileVersion();
                    if (latestVersion == null)
                    {
                        result.Result = LoadModelResults.FailedConvertBinary;
                        result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.InvalidG3DXSDBasePath));

                        return result;
                    }

                    // モデルファイルのバージョンをチェック
                    if (G3dIfUtility.CheckG3DFileVersion(filePath, latestVersion) == false)
                    {
                        result.DetailMessages.Add(new LoadModelMessage(
                            LoadModelMessageCodes.InvalidFileVersion,
                            filePath,
                            latestVersion.ToString()));

                        return result;
                    }
                    else
                    {
                        result.Result = LoadModelResults.FailedConvertBinary;
                        result.DetailMessages.Add(new LoadModelMessage(addModelResult));

                        return result;
                    }
                }
            }

            // バイナリコンバートを実行
            try
            {
                errCode = this.Converter.Binarize(outStream);
                if (errCode != 0)
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(this.TranslateErrorCode(errCode)));

                    return result;
                }
            }
            catch (Exception)
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.UnknownFailure));

                return result;
            }

            return result;
        }

        /// <summary>
        /// トリミングプリミティブを作成します。
        /// </summary>
        /// <param name="filePath">テクスチャファイルパス</param>
        /// <param name="texWidth">テクスチャの横幅</param>
        /// <param name="texHeight">テクスチャの縦幅</param>
        /// <param name="x">トリミング領域のX座標</param>
        /// <param name="y">トリミング領域のY座標</param>
        /// <param name="w">トリミング領域の横幅</param>
        /// <param name="h">トリミング領域の縦幅</param>
        /// <returns>作成結果と作成したモデルデータを返します。</returns>
        public LoadModelResultWithData CreateTrimmingPrimitive(
            string filePath,
            int texWidth,
            int texHeight,
            int x,
            int y,
            int w,
            int h,
            bool inverse)
        {
            ModelData modelData = new ModelData();

            // Indexesを作成
            modelData.Indexes = new IndexBuffer();

            if ( inverse == false )
            {
                modelData.Indexes.Indexes = new int[]
                {
                    0, 2, 1,
                    0, 3, 2
                };
            }
            else
            {
                modelData.Indexes.Indexes = new int[]
                {
                    0, 1, 2,
                    0, 2, 3
                };

            }

            // Positionを作成
            float posLeft = (((float)x + 0.5f) / ((float)texWidth)) - 0.5f;
            float posRight = (((float)x + w + 0.5f) / ((float)texWidth)) - 0.5f;
            float posTop = 0.5f - (((float)y + 0.5f) / ((float)texHeight));
            float posBottom = 0.5f - (((float)y + h + 0.5f) / ((float)texHeight));

            float[] posElement = new float[]
            {
                posLeft, posBottom, 0.0f,
                posLeft, posTop, 0.0f,
                posRight, posTop, 0.0f,
                posRight, posBottom, 0.0f
            };

            modelData.Position = new VertexElement(posElement, 3, VertexElementUsages.Position);

            // TexCoordを作成
            float texLeft = ((float)x + 0.5f) / ((float)texWidth);
            float texRight = ((float)x + w + 0.5f) / ((float)texWidth);
            float texTop = ((float)y + 0.5f) / ((float)texHeight);
            float texBottom = ((float)y + h + 0.5f) / ((float)texHeight);

            float[] texCoordElement = new float[]
            {
                texLeft, texBottom,
                texLeft, texTop,
                texRight, texTop,
                texRight, texBottom
            };

            modelData.TexCoord0 = new VertexElement(texCoordElement, 2, VertexElementUsages.TextureCoordinate0);
            modelData.TexCoord1 = new VertexElement(texCoordElement, 2, VertexElementUsages.TextureCoordinate1);

            // Normalを作成
            float[] normalElement = new float[]
            {
                0.0f, 0.0f, -1.0f,
                0.0f, 0.0f, -1.0f,
                0.0f, 0.0f, -1.0f
            };

            modelData.Normal = new VertexElement(normalElement, 3, VertexElementUsages.Normal);

            // Tangentを作成
            float[] tangentElement = new float[]
            {
                0.0f, 1.0f, 0.0f,
                0.0f, 1.0f, 0.0f,
                0.0f, 1.0f, 0.0f
            };

            modelData.Tangent = new VertexElement(tangentElement, 3, VertexElementUsages.Tangent);

            // その他要素を作成
            float[] defaultElement = new float[0];

            modelData.Color = new VertexElement(defaultElement, 0, VertexElementUsages.Color);
            modelData.NumberOfTextures = 0;

            // 結果を返す
            LoadModelResultWithData resultWithData = new LoadModelResultWithData()
            {
                ModelData = modelData,
                Result = LoadModelResults.Success
            };

            return resultWithData;
        }

        /// <summary>
        /// トリミングプリミティブを作成します。
        /// </summary>
        /// <param name="filePath">テクスチャファイルパス</param>
        /// <param name="enableCache">キャッシュの有効/無効</param>
        /// <param name="texWidth">テクスチャの横幅</param>
        /// <param name="texHeight">テクスチャの縦幅</param>
        /// <param name="x">トリミング領域のX座標</param>
        /// <param name="y">トリミング領域のY座標</param>
        /// <param name="w">トリミング領域の横幅</param>
        /// <param name="h">トリミング領域の縦幅</param>
        /// <returns>作成結果と作成したモデルデータを返します。</returns>
        public LoadModelResultWithData CreateTrimmingPrimitive(
            string filePath,
            bool enableCache,
            int texWidth,
            int texHeight,
            int x,
            int y,
            int w,
            int h,
            bool inverse)
        {
            LoadModelResultWithData resultWithData;

            // キャッシュ読み込みするかどうか?
            if (enableCache)
            {
                var modelData = this.trimmingPrimitiveDataCache.Get(filePath);

                // キャッシュ済みでも日付が違っていたら読み直す
                if (modelData != null)
                {
                    if (modelData.UpdateTimestamp != File.GetLastWriteTime(filePath))
                    {
                        this.trimmingPrimitiveDataCache.Remove(filePath);
                        modelData = null;
                    }
                }

                if (modelData != null)
                {
                    resultWithData = new LoadModelResultWithData
                    {
                        Result = LoadModelResults.Success,
                        ModelData = modelData.Data
                    };
                }
                else
                {
                    // 新規に作る.
                    resultWithData = this.CreateTrimmingPrimitive(filePath, texWidth, texHeight, x, y, w, h, inverse);

                    // 成功時にキャッシュに収める
                    if (enableCache)
                    {
                        var data = new ModelDataCache.ModelFileData
                        {
                            Data = resultWithData.ModelData,
                            UpdateTimestamp = File.GetLastWriteTime(filePath)
                        };

                        this.trimmingPrimitiveDataCache.Add(filePath, data);
                    }
                }
            }
            else
            {
                // 新規に作る.
                resultWithData = this.CreateTrimmingPrimitive(filePath, texWidth, texHeight, x, y, w, h, inverse);
            }

            return resultWithData;
        }

        /// <summary>
        /// 中間ファイルをバイナリファイルに変換します.
        /// </summary>
        /// <param name="modelPath">モデルファイルパス</param>
        /// <param name="animationPaths">アニメーションファイルパス</param>
        /// <param name="outStream">出力ストリーム</param>
        /// <returns>成功したときtrueを返します.</returns>
        public LoadModelResultWithData ConvertBinary(
            string modelPath,
            string[] animationPaths,
            Stream outStream)
        {
            var result = new LoadModelResultWithData();
            result.DetailMessages = new List<LoadModelMessage>();

            this.texturePathList.Clear();

            // ファイルパスをチェック
            if (string.IsNullOrEmpty(modelPath) == true || Path.IsPathRooted(modelPath) == false)
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.InvalidFilePath));

                return result;
            }

            // ファイルがあるかどうかチェック
            if (File.Exists(modelPath) == false)
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FileNotFound));

                return result;
            }

            nw4f_3difType loadedObj = null;
            List<G3dStream> streams = new List<G3dStream>();

            // モデルの中間ファイルを読み込む
            try
            {
                loadedObj = IfReadUtility.Read(streams, modelPath.ToLower(), G3dIfUtility.XsdBasePath);
            }
            catch
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FailedOpeningFile));

                return result;
            }

            // モデルが読み込めたかチェック
            if (loadedObj == null)
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FailedLoadingModel));

                return result;
            }

            // モデルデータの内容をチェック
            modelType model = loadedObj.Item as modelType;
            if (model == null || model.shape_array == null)
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FailedModelNotFound, modelPath));

                return result;
            }

            // バイナリコンバータが初期化済みかチェック
            if (this.Converter == null)
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FailedInitConverter));

                return result;
            }

            // アニメーションなど複数に分かれたファイルを1つのバイナリにまとめるように設定
            int errCode = this.Converter.PreBinarizeUsingViewerOption();
            if (errCode != 0)
            {
                result.DetailMessages.Add(new LoadModelMessage(this.TranslateErrorCode(errCode)));
            }

            // モデルファイルをコンバータに追加
            LoadModelMessageCodes addModelResult = this.AddFileToConverter(
                modelPath,
                this.Converter,
                IfBinaryFormatter.FormatStreamFast(loadedObj, streams));
            if (addModelResult != LoadModelMessageCodes.Success)
            {
                Version latestVersion = G3dIfUtility.GetLatestFileVersion();
                if (latestVersion == null)
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.InvalidG3DXSDBasePath));

                    return result;
                }

                // モデルファイルのバージョンをチェック
                if (G3dIfUtility.CheckG3DFileVersion(modelPath, latestVersion) == false)
                {
                    result.DetailMessages.Add(new LoadModelMessage(
                        LoadModelMessageCodes.InvalidFileVersion,
                        modelPath,
                        latestVersion.ToString()));

                    // バージョン違いのエラーは後でまとめて返す
                }
                else
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(addModelResult));

                    return result;
                }
            }

            // アニメーションファイルをコンバータに追加
            foreach (string animPath in animationPaths)
            {
                // ファイルパスをチェック
                if (string.IsNullOrEmpty(animPath) == true ||
                    Path.IsPathRooted(animPath) == false ||
                    File.Exists(animPath) == false)
                {
                    continue;
                }

                // アニメの中間ファイルを読み込む
                nw4f_3difType animObj = null;
                List<G3dStream> animStreams = new List<G3dStream>();
                try
                {
                    animObj = IfReadUtility.Read(animStreams, animPath.ToLower(), G3dIfUtility.XsdBasePath);
                }
                catch
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FailedOpeningFile, animPath));

                    return result;
                }

                // アニメーションファイルをコンバータに追加
                LoadModelMessageCodes addAnimResult = this.AddFileToConverter(
                    animPath,
                    this.Converter,
                    IfBinaryFormatter.FormatStreamFast(animObj, animStreams));

                if (addAnimResult != LoadModelMessageCodes.Success)
                {
                    Version latestVersion = G3dIfUtility.GetLatestFileVersion();
                    if (latestVersion == null)
                    {
                        result.Result = LoadModelResults.FailedConvertBinary;
                        result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.InvalidG3DXSDBasePath));

                        return result;
                    }

                    // アニメーションファイルのバージョンをチェック
                    if (G3dIfUtility.CheckG3DFileVersion(animPath, latestVersion) == false)
                    {
                        result.DetailMessages.Add(new LoadModelMessage(
                            LoadModelMessageCodes.InvalidFileVersion,
                            animPath,
                            latestVersion.ToString()));

                        // バージョン違いのエラーは後でまとめて返す
                    }
                    else
                    {
                        result.Result = LoadModelResults.FailedConvertBinary;
                        result.DetailMessages.Add(new LoadModelMessage(addAnimResult));

                        return result;
                    }
                }
            }

            // モデルが参照しているリソースファイル(テクスチャ)のパスを取得
            List<string> resourcePaths = this.CollectExternalResourcePaths(loadedObj);
            if (resourcePaths == null)
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FailedCollectingResources));

                return result;
            }

            // テクスチャファイルの拡張子
            string[] extensions = new string[] { ".ftxb", ".ftxa", ".tga" };

            // モデルファイルのベースフォルダを取得
            string baseFolder = Path.GetDirectoryName(modelPath);

            // テクスチャファイルをコンバータに追加
            foreach (string resPath in resourcePaths)
            {
                string texPath;
                bool resFileFound = false;

                // ここで得られるテクスチャのファイル名には拡張子がついてないので、
                // それぞれの拡張子を付けてみて見つかったファイルを利用する
                foreach (string ext in extensions)
                {
                    // テクスチャファイルの検索・ロケート
                    bool resLocate = this.LocateTextureByFileName(baseFolder, resPath + ext, out texPath);
                    if (resLocate == false)
                    {
                        continue;
                    }

                    // テクスチャの中間ファイルを読み込む
                    nw4f_3difType texObj = null;
                    List<G3dStream> texStreams = new List<G3dStream>();
                    try
                    {
                        texObj = IfReadUtility.Read(texStreams, texPath.ToLower(), G3dIfUtility.XsdBasePath);
                    }
                    catch
                    {
                        result.Result = LoadModelResults.FailedConvertBinary;
                        result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.FailedOpeningFile, texPath));

                        return result;
                    }

                    if (this.BinaryMode == G3DBinaryMode.NnBinary)
                    {
                        // NNバイナリの時はBNTX以外のファイルは一時リストに追加するだけ
                        this.texturePathList.Add(texPath);
                    }
                    else
                    {
                        // テクスチャファイルをコンバータに追加
                        LoadModelMessageCodes addResResult = this.AddFileToConverter(
                            texPath,
                            this.Converter,
                            IfBinaryFormatter.FormatStreamFast(texObj, texStreams));
                        if (addResResult != LoadModelMessageCodes.Success)
                        {
                            Version latestVersion = G3dIfUtility.GetLatestFileVersion();
                            if (latestVersion == null)
                            {
                                result.Result = LoadModelResults.FailedConvertBinary;
                                result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.InvalidG3DXSDBasePath));

                                return result;
                            }

                            // テクスチャファイルのバージョンをチェック
                            if (G3dIfUtility.CheckG3DFileVersion(texPath, latestVersion) == false)
                            {
                                result.DetailMessages.Add(new LoadModelMessage(
                                    LoadModelMessageCodes.InvalidFileVersion,
                                    texPath,
                                    latestVersion.ToString()));

                                // バージョン違いのエラーは後でまとめて返す
                            }
                            else
                            {
                                result.Result = LoadModelResults.FailedConvertBinary;
                                result.DetailMessages.Add(new LoadModelMessage(addResResult));

                                return result;
                            }
                        }
                    }

                    resFileFound = true;
                    break;
                }

                // テクスチャが見つからなかったとき、エラー
                if (resFileFound == false)
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(
                        LoadModelMessageCodes.ResourceFilesNotFound,
                        resPath));
                }
            }

            if (this.BinaryMode == G3DBinaryMode.NnBinary &&
                this.texturePathList.Any() &&
                this.ConvertBntx != null)
            {
                var bntxPath = this.ConvertBntx(this.texturePathList);
                if (string.IsNullOrEmpty(bntxPath))
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(
                        LoadModelMessageCodes.FailedAddingResourceFiles,
                        new object[]{"Failed to make bntx."}));
                    return result;
                }

                LoadModelMessageCodes addResResult = this.AddFileToConverter(
                    bntxPath,
                    this.Converter,
                    null);
                if (addResResult != LoadModelMessageCodes.Success)
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(addResResult));
                }
            }

            // バージョン違いのエラーを返す
            if (result.DetailMessages.Count > 0)
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                return result;
            }

            // バイナリコンバートを実行
            try
            {
                errCode = this.Converter.Binarize(outStream);
                if (errCode != 0)
                {
                    result.Result = LoadModelResults.FailedConvertBinary;
                    result.DetailMessages.Add(new LoadModelMessage(this.TranslateErrorCode(errCode)));

                    return result;
                }
            }
            catch (Exception)
            {
                result.Result = LoadModelResults.FailedConvertBinary;
                result.DetailMessages.Add(new LoadModelMessage(LoadModelMessageCodes.UnknownFailure));

                return result;
            }

            return result;
        }

        /// <summary>
        /// 事前JITする
        /// </summary>
        /// <param name="modelData">モデルデータ</param>
        /// <param name="extName">モデルファイル拡張子</param>
        public void PreJit(byte[] modelData, string extName)
        {
            var filePath = Path.GetTempFileName() + extName;

            try
            {
                using (var file = new FileStream(filePath, FileMode.Create, FileAccess.Write))
                {
                    file.Write(modelData, 0, modelData.Length);
                }

                // 空読みする
                this.LoadModel(filePath);
            }
            finally
            {
                if (File.Exists(filePath))
                {
                    File.Delete(filePath);
                }
            }
        }

        /// <summary>
        /// バイナリコンバータにファイルの内容を追加します.
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="converter">バイナリコンバータ</param>
        /// <param name="fileImage">アップデート適用済みファイルイメージ</param>
        /// <returns>結果コード</returns>
        private LoadModelMessageCodes AddFileToConverter(string filePath, BinaryConverter converter, byte[] fileImage)
        {
            byte[] buffer = null;
            string fileName = Path.GetFileNameWithoutExtension(filePath);
            string fileExt = Path.GetExtension(filePath);

            // ファイルの内容を読み込む
            try
            {
                buffer = fileImage ?? File.ReadAllBytes(filePath);
            }
            catch (Exception)
            {
                return LoadModelMessageCodes.FailedAddingResourceFiles;
            }

            // ファイルの内容をコンバータに追加
            if (buffer != null && buffer.Length > 0)
            {
                int errCode;
                if (fileExt.ToLower() == ".bntx")
                {
                    const int alignPos = 14;
                    long alignment = 0;

                    // 64MB以下くらいの値を有効値とする
                    if (buffer.Length > alignPos && buffer[alignPos] <= 26)
                    {
                        alignment = 1 << buffer[alignPos];
                    }

                    errCode = converter.AddExternalFile(
                        buffer,
                        buffer.Length,
                        filePath,
                        fileName,
                        fileExt.ToLower(),
                        (int)alignment);
                }
                else
                {
                    errCode = converter.AddFile(
                        buffer,
                        buffer.Length,
                        filePath,
                        fileName,
                        fileExt.ToLower());
                }

                if (errCode != 0)
                {
                    return this.TranslateErrorCode(errCode);
                }
            }

            return LoadModelMessageCodes.Success;
        }

        /// <summary>
        /// バイナリコンバータが出力するエラーコードをLoadModelMessageCodesに変換します.
        /// </summary>
        /// <param name="errorCode">バイナリコンバータが出力するエラーコード</param>
        /// <returns>LoadModelMessageCodes</returns>
        private LoadModelMessageCodes TranslateErrorCode(int errorCode)
        {
            switch ((BinaryConverter.errCode)errorCode)
            {
                case BinaryConverter.errCode.notInitialize:
                    return LoadModelMessageCodes.FailedInitConverter;

                case BinaryConverter.errCode.SetOptions:
                    return LoadModelMessageCodes.FailedSetupConverterOptions;

                case BinaryConverter.errCode.AddFile:
                    return LoadModelMessageCodes.FailedAddingResourceFiles;

                case BinaryConverter.errCode.CalcSize:
                    return LoadModelMessageCodes.FailedComputingSize;

                case BinaryConverter.errCode.MallocOutBuf:
                    return LoadModelMessageCodes.FailedAllocOutBuffer;

                case BinaryConverter.errCode.Convert:
                    return LoadModelMessageCodes.FailedConvertingBinary;

                case BinaryConverter.errCode.Bind:
                    return LoadModelMessageCodes.FailedBinding;

                case BinaryConverter.errCode.EndianSwap:
                    return LoadModelMessageCodes.FailedSwappingEndian;

                case BinaryConverter.errCode.SaveFile:
                    return LoadModelMessageCodes.FailedSaveBinFile;

                default:
                    return LoadModelMessageCodes.UnknownFailure;
            }
        }

        /// <summary>
        /// モデルデータが参照しているリソースファイル(テクスチャ)のパスリストを取得します.
        /// </summary>
        /// <param name="loadedObj">モデルデータ</param>
        /// <returns>リソースファイルのパスリストを返します.</returns>
        private List<string> CollectExternalResourcePaths(nw4f_3difType loadedObj)
        {
            if (loadedObj == null)
            {
                return null;
            }

            // モデルデータを取得
            modelType model = loadedObj.Item as modelType;
            if (model == null)
            {
                return null;
            }

            // テクスチャのパスリストを取得
            var paths =
                from mat in model.material_array.Items
                where mat != null && mat.sampler_array != null
                from smp in mat.sampler_array.Items
                select smp.tex_name;

            // 重複を削除したリストを返す
            return paths.Distinct().ToList();
        }

        /// <summary>
        /// 与えられたテクスチャのファイルパスをロケートします.
        /// </summary>
        /// <param name="baseDir">モデルファイルのフォルダ</param>
        /// <param name="filePath">テクスチャのファイルパス</param>
        /// <param name="fullPath">ロケートしたファイルパス</param>
        /// <returns>ロケートしたときTrueを返します.</returns>
        private bool LocateTextureByFileName(
            string baseDir,
            string filePath,
            out string fullPath)
        {
            // ほぼ EffectMaker.BusinessLogic.IO.AssetPathValidator.LocateAssetPath() のコピー

            // out変数を初期化
            fullPath = string.Empty;

            // テクスチャファイルの名前を取得
            string fileName = Path.GetFileName(filePath);

            string path;

            // "basedir/fileName"のファイルがあるかチェック
            path = Path.Combine(baseDir, fileName);
            if (File.Exists(path))
            {
                fullPath = path;
                return true;
            }

            // アセットの相対パスを取得
            string assetDir = "Textures";
            string assetPath = Path.Combine(assetDir, fileName);

            // ファイルが見つかるまでbaseDirの上の階層を辿っていく
            do
            {
                // "baseDir/assetDir/fileName"にファイルがあるかチェック
                path = Path.Combine(baseDir, assetPath);
                if (File.Exists(path))
                {
                    fullPath = path;
                    return true;
                }

                // baseDir = "baseDir/../"
                baseDir = Path.GetDirectoryName(baseDir);
                if (string.IsNullOrEmpty(baseDir))
                {
                    break;
                }
            }
            while (Path.IsPathRooted(baseDir));

            return false;
        }

        /// <summary>
        /// 読み込んだプリミティブファイルの変更日時を取得します。
        /// </summary>
        /// <param name="filePath">プリミティブファイルパス</param>
        /// <returns>
        /// 読み込んだプリミティブファイルの変更日時を返します。
        /// プリミティブファイルが読み込まれていないときはデフォルト値を返します。
        /// </returns>
        public DateTime GetLoadedFileModifyTime(string filePath)
        {
            var cacheData = this.modelDataCache.Get(filePath);

            if (cacheData == null)
            {
                return new DateTime();
            }

            return cacheData.UpdateTimestamp;
        }
    }
}
