﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using EffectMaker.Foundation.Model;
using EffectMaker.Foundation.Model.Types;
using nw.g3d.nw4f_3dif;
using G3dIfLib;

namespace EffectMaker.PrimitiveManager.Loader
{
    /// <summary>
    /// Fmd モデルローダー
    /// </summary>
    public class FmdModelLoader : IModelLoader
    {
        /// <summary>
        /// ファイルが読み込めるか調査します。
        /// </summary>
        /// <param name="filePath">調査するファイルパス</param>
        /// <returns>読み込めるかどうか？</returns>
        public bool CanLoad(string filePath)
        {
            Debug.Assert(string.IsNullOrEmpty(filePath) == false, "パスの指定が不正");

            var ext = Path.GetExtension(filePath).ToLower();

            return (ext == ".fmda") || (ext == ".fmdb");
        }

        /// <summary>
        /// モデルデータをロードします。
        /// </summary>
        /// <param name="filePath">ロードするファイルパス</param>
        /// <param name="modelData">ロードしたモデルデータ</param>
        /// <returns>結果</returns>
        public LoadModelResults Load(string filePath, out ModelData modelData)
        {
            if (File.Exists(filePath) == false)
            {
                modelData = null;
                return LoadModelResults.FileNotFound;
            }

            if (this.CanLoad(filePath) == false)
            {
                modelData = null;
                return LoadModelResults.UnknowModelType;
            }

            Version latestVersion = G3dIfUtility.GetLatestFileVersion();
            if (latestVersion == null)
            {
                modelData = null;
                return LoadModelResults.UnknowModelType;
            }

            // モデルファイルのバージョンをチェック
            if (G3dIfUtility.CheckG3DFileVersion(filePath, latestVersion) == false)
            {
                modelData = null;
                return LoadModelResults.InvalidFileVersion;
            }

            try
            {
                return LoadInternal(filePath, out modelData);
            }
            catch (Exception e)
            {
                Debug.WriteLine("exception:{0}", e);

                modelData = null;
                return LoadModelResults.FailedLoadingModel;
            }
        }

        /// <summary>
        /// コメント情報だけを読み込みます。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="comment">[out]コメント</param>
        /// <param name="color">[out]ラベルカラー</param>
        /// <returns>成功したらtrue,失敗したらfalse.</returns>
        public bool LoadComment(string filePath, out string comment, out string color)
        {
            comment = color = string.Empty;
            if (File.Exists(filePath) == false || this.CanLoad(filePath) == false)
            {
                return false;
            }

            try
            {
                var loadResult = G3dIfUtility.LoadG3dIf(filePath);

                if (loadResult.G3dIf.RootElement.comment != null)
                {
                    comment = loadResult.G3dIf.RootElement.comment.text;
                    color = loadResult.G3dIf.RootElement.comment.color;
                }

                return true;
            }
            catch (Exception e)
            {
                Debug.WriteLine("exception:{0}", e);

                return false;
            }
        }

        /// <summary>
        /// インデックスバッファを読み込みます。
        /// </summary>
        /// <param name="src">元モデル</param>
        /// <param name="binaryStreams">バイナリストリーム</param>
        /// <returns>インデックスバッファ</returns>
        private static IndexBuffer LoadIndexBuffer(modelType src, List<G3dStream> binaryStreams)
        {
            if ((src.shape_array == null) ||
                (src.shape_array.shape == null) ||
                (src.shape_array.shape.Any() == false) ||
                (src.shape_array.shape[0].mesh_array == null) ||
                (src.shape_array.shape[0].mesh_array.mesh == null) ||
                (src.shape_array.shape[0].mesh_array.mesh.Any() == false))
            {
                return new IndexBuffer
                {
                    Indexes = new int[0]
                };
            }

            var streamIndex = src.shape_array.shape[0].mesh_array.mesh[0].stream_index;

            return new IndexBuffer
            {
                Indexes = binaryStreams[streamIndex].IntData.ToArray()
            };
        }

        /// <summary>
        /// モデルで使われているテクスチャ文字列を取得します。
        /// </summary>
        /// <param name="src">モデル</param>
        /// <returns>テクスチャ文字列郡を返します。</returns>
        private static IEnumerable<string> GetTextures(modelType src)
        {
            if ((src.material_array != null) &&
                (src.material_array.Items != null))
            {
                foreach (
                    var sampler in
                    from material in src.material_array.Items
                    where material != null && material.sampler_array != null
                    from sampler in material.sampler_array.Items
                    where sampler != null
                    select sampler)
                {
                    yield return sampler.tex_name;
                }
            }

            yield break;
        }

        /// <summary>
        /// LOD数を読み込みます。
        /// </summary>
        /// <param name="src">元モデル</param>
        /// <returns>LOD数を返します。</returns>
        private static int LoadLodCount(modelType src)
        {
            int lodCount = 0;

            // /vertex_array/shape_array/shape/mesh_array.length
            // の最大値がLOD数になっているのでそれを取得する
            if (src.shape_array != null && src.shape_array.Items != null)
            {
                foreach (var shape in src.shape_array.Items)
                {
                    if (shape.mesh_array != null && shape.mesh_array.Items != null)
                    {
                        if (shape.mesh_array.length > lodCount)
                        {
                            lodCount = shape.mesh_array.length;
                        }
                    }
                }
            }

            return lodCount;
        }

        /// <summary>
        /// 内部用ロード処理を行います。
        /// </summary>
        /// <param name="filePath">ロードするファイルパス</param>
        /// <param name="modelData">ロードしたモデルデータ</param>
        /// <returns>結果</returns>
        private static LoadModelResults LoadInternal(string filePath, out ModelData modelData)
        {
            var loadResult = G3dIfUtility.LoadG3dIf(filePath);
            var model = loadResult.G3dIf.Item as modelType;

            modelData = new ModelData
            {
                Position = LoadVertexElement(model, loadResult.BinaryStreams, VertexElementUsages.Position, "position0"),
                PositionName = LoadVertexElementName(model, "position0"),
                Normal = LoadVertexElement(model, loadResult.BinaryStreams, VertexElementUsages.Normal, "normal0"),
                NormalName = LoadVertexElementName(model, "normal0"),
                Tangent = LoadVertexElement(model, loadResult.BinaryStreams, VertexElementUsages.Tangent, "tangent0"),
                TangentName = LoadVertexElementName(model, "tangent0"),
                TexCoord0 = LoadVertexElement(model, loadResult.BinaryStreams, VertexElementUsages.TextureCoordinate0, "uv0"),
                TexCoordName0 = LoadVertexElementName(model, "uv0"),
                TexCoord1 = LoadVertexElement(model, loadResult.BinaryStreams, VertexElementUsages.TextureCoordinate1, "uv1"),
                TexCoordName1 = LoadVertexElementName(model, "uv1"),
                Color = LoadVertexElement(model, loadResult.BinaryStreams, VertexElementUsages.Color, "color0"),
                ColorName = LoadVertexElementName(model, "color0"),
                Indexes = LoadIndexBuffer(model, loadResult.BinaryStreams),
                NumberOfTexCoords = LoadVertexElementCount(model, "uv"),
                NumberOfTextures = GetTextures(model).Distinct().Count(),
                LodCount = LoadLodCount(model)
            };

            return LoadModelResults.Success;
        }

        /// <summary>
        /// 頂点要素を読み込みます。
        /// </summary>
        /// <param name="src">元モデル</param>
        /// <param name="binaryStreams">バイナリストリーム</param>
        /// <param name="usage">頂点情報</param>
        /// <param name="hint">ヒント文字列</param>
        /// <returns>頂点要素を返します。</returns>
        private static VertexElement LoadVertexElement(modelType src, List<G3dStream> binaryStreams, VertexElementUsages usage, string hint)
        {
            if ((src.vertex_array == null) ||
                (src.vertex_array.Items == null) ||
                (src.vertex_array.Items.Any() == false) ||
                (src.vertex_array.Items[0].vtx_attrib_array == null) ||
                (src.vertex_array.Items[0].vtx_attrib_array.Items == null))
            {
                return new VertexElement
                {
                    ElementData = new float[0],
                    Usage = usage
                };
            }

            var vtxAttribArray = src.vertex_array.Items[0].vtx_attrib_array.Items;
            var vtxAttrib = vtxAttribArray.FirstOrDefault(x => x.hint == hint);
            if (vtxAttrib == null)
            {
                return new VertexElement
                {
                    ElementData = new float[0],
                    Usage = usage
                };
            }

            var streamIndex = vtxAttrib.stream_index;

            // float配列に変換する
            float[] elementData = null;
            {
                switch (binaryStreams[streamIndex].type)
                {
                    case stream_typeType.@float:
                        elementData = binaryStreams[streamIndex].FloatData.ToArray();
                        break;
                    case stream_typeType.@int:
                        elementData = usage == VertexElementUsages.Color ?
                                        binaryStreams[streamIndex].IntData.Select(x => x / 255.0f).ToArray() :
                                        binaryStreams[streamIndex].IntData.Select(x => (float)x).ToArray();
                        break;
                    case stream_typeType.@byte:
                        elementData = binaryStreams[streamIndex].ByteData.Select(x => (float)x).ToArray();
                        break;
                    default:
                        Debug.Assert(elementData != null, "未定義の動作。要調査");
                        break;
                }
            }

            return new VertexElement
            {
                ElementData = elementData,
                Column = binaryStreams[streamIndex].column,
                Usage = usage,
                Type = (int)vtxAttrib.type,
                QuantizeType = (int)vtxAttrib.quantize_type,
                AttributeIndex = vtxAttrib.attrib_index
            };
        }

        /// <summary>
        /// 頂点の要素名を読み込みます。
        /// </summary>
        /// <param name="src">元モデル</param>
        /// <param name="hint">ヒント文字列</param>
        /// <returns>頂点の要素名を返します。</returns>
        private static string LoadVertexElementName(modelType src, string hint)
        {
            if ((src.vertex_array == null) ||
                (src.vertex_array.Items == null) ||
                (src.vertex_array.Items.Any() == false) ||
                (src.vertex_array.Items[0].vtx_attrib_array == null) ||
                (src.vertex_array.Items[0].vtx_attrib_array.Items == null))
            {
                return null;
            }

            var vtxAttribArray = src.vertex_array.Items[0].vtx_attrib_array.Items;
            var vtxAttrib = vtxAttribArray.FirstOrDefault(x => x.hint == hint);
            if (vtxAttrib == null)
            {
                return null;
            }

            return vtxAttrib.name;
        }

        /// <summary>
        /// 頂点の要素数を読み込みます。
        /// </summary>
        /// <param name="src">元モデル</param>
        /// <param name="hint">ヒント文字列</param>
        /// <returns>頂点の要素数を返します。</returns>
        private static int LoadVertexElementCount(modelType src, string hint)
        {
            if ((src.vertex_array == null) ||
                (src.vertex_array.Items == null) ||
                (src.vertex_array.Items.Any() == false) ||
                (src.vertex_array.Items[0].vtx_attrib_array == null) ||
                (src.vertex_array.Items[0].vtx_attrib_array.Items == null))
            {
                return 0;
            }

            var vtxAttribArray = src.vertex_array.Items[0].vtx_attrib_array.Items;
            int elementCount = vtxAttribArray.Count(x => x.hint.StartsWith(hint));

            return elementCount;
        }
    }
}
