﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using System.Reflection;
using EffectMaker.BusinessLogic.CurveEditorParameters;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;

using EffectMaker.DataModel.Specific.DataModels;

using EffectMaker.Foundation.Collections.Generic;

namespace EffectMaker.DataModelLogic.Delegates
{
    /// <summary>
    /// Class for post construct delegations for the data models.
    /// </summary>
    public class DataModelPostConstructor
    {
        /// <summary>
        /// Initialize the object.
        /// </summary>
        public static void Initialize()
        {
            DataModelManager.SetPostConstructForType(
                typeof(EmitterBasicBasicData),
                OnEmitterBasicBasicPostConstruct);

            // エミッタ時間アニメの初期値をここで与える
            DataModelManager.SetPostConstructForType(
                typeof(EmitterEmitterShapeData),
                OnEmitterShapePostConstruct);

            DataModelManager.SetPostConstructForType(
                typeof(EmitterEmitterTransformData),
                OnEmitterTransformPostConstruct);

            DataModelManager.SetPostConstructForType(
                typeof(EmitterEmissionTimingData),
                OnEmissionTimingPostConstruct);

            DataModelManager.SetPostConstructForType(
                typeof(EmitterEmissionVelocityData),
                OnEmissionVelocityPostConstruct);

            DataModelManager.SetPostConstructForType(
                typeof(EmitterEmissionGravityData),
                OnEmissionGravityPostConstruct);

            DataModelManager.SetPostConstructForType(
                typeof(EmitterParticleLifeData),
                OnParticleLifePostConstruct);

            DataModelManager.SetPostConstructForType(
                typeof(EmitterScaleBasicData),
                OnScaleBasicPostConstruct);

            DataModelManager.SetPostConstructForType(
                typeof(EmitterEmitterColorData),
                OnEmitterColorPostConstruct);

            // カスタムアクション・シェーダ用の初期化処理
            DataModelManager.SetPostConstructForType(
                typeof(CustomActionData),
                OnCustomActionDataPostConstruct);

            DataModelManager.SetPostConstructForType(
                typeof(EmitterCustomShaderData),
                OnEmitterCustomShaderDataPostConstruct);

            DataModelManager.SetPostConstructForType(
                typeof(EmitterExtParams),
                OnEmitterExtParamsPostConstruct);
        }

        /// <summary>
        /// The on emitter color post construct.
        /// </summary>
        /// <param name="dataModel">
        /// The data model.
        /// </param>
        private static void OnEmitterBasicBasicPostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as EmitterBasicBasicData;
            if (dm == null)
            {
                return;
            }

            if (SpecManager.CurrentSpec.ForceCpuCalc)
            {
                dm.ProcessType = 0;
            }
        }

        /// <summary>
        /// The on emitter color post construct.
        /// </summary>
        /// <param name="dataModel">
        /// The data model.
        /// </param>
        private static void OnEmitterColorPostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as EmitterEmitterColorData;
            if (dm == null)
            {
                return;
            }

            dm.Color0Animation.AnimationTable = new AnimationTableData(0, InitialKeyValues.Colors);
            dm.Color1Animation.AnimationTable = new AnimationTableData(0, InitialKeyValues.Colors);
            dm.Alpha0Animation.AnimationTable = new AnimationTableData(0, InitialKeyValues.Colors);
            dm.Alpha1Animation.AnimationTable = new AnimationTableData(0, InitialKeyValues.Colors);
        }

        /// <summary>
        /// The on scale basic post construct.
        /// </summary>
        /// <param name="dataModel">
        /// The data model.
        /// </param>
        private static void OnScaleBasicPostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as EmitterScaleBasicData;
            if (dm == null)
            {
                return;
            }

            dm.BasicScaleAnimation.AnimationTable =
                new AnimationTableData(0, InitialKeyValues.ScaleBasic);
        }

        /// <summary>
        /// The on particle life post construct.
        /// </summary>
        /// <param name="dataModel">
        /// The data model.
        /// </param>
        private static void OnParticleLifePostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as EmitterParticleLifeData;
            if (dm == null)
            {
                return;
            }

            dm.LifeAnimation.AnimationTable =
                new AnimationTableData(0, InitialKeyValues.ParticleLife);
        }

        /// <summary>
        /// The on emission gravity post construct.
        /// </summary>
        /// <param name="dataModel">
        /// The data model.
        /// </param>
        private static void OnEmissionGravityPostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as EmitterEmissionGravityData;
            if (dm == null)
            {
                return;
            }

            dm.GravityAnimation.AnimationTable =
                new AnimationTableData(0, InitialKeyValues.EmissionGravity);
        }

        /// <summary>
        /// The on emission velocity post construct.
        /// </summary>
        /// <param name="dataModel">
        /// The data model.
        /// </param>
        private static void OnEmissionVelocityPostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as EmitterEmissionVelocityData;
            if (dm == null)
            {
                return;
            }

            dm.OmnidirectionalVelocityAnimation.AnimationTable =
                new AnimationTableData(0, InitialKeyValues.OmniVelocity);
            dm.OrientedDirectionVelocityAnimation.AnimationTable =
                new AnimationTableData(0, InitialKeyValues.OrientedVelocity);
        }

        /// <summary>
        /// The on emission timing post construct.
        /// </summary>
        /// <param name="dataModel">
        /// The data model.
        /// </param>
        private static void OnEmissionTimingPostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as EmitterEmissionTimingData;
            if (dm == null)
            {
                return;
            }

            dm.EmitTimeRateAnimation.AnimationTable =
                new AnimationTableData(0, InitialKeyValues.EmitTimeRate);
        }

        /// <summary>
        /// The on emitter shape post construct.
        /// </summary>
        /// <param name="dataModel">
        /// The data model.
        /// </param>
        private static void OnEmitterShapePostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as EmitterEmitterShapeData;
            if (dm == null)
            {
                return;
            }

            dm.ShapeScaleAnimation.AnimationTable =
                new AnimationTableData(0, InitialKeyValues.ShapeScale);
        }

        /// <summary>
        /// The on emitter transform post construct.
        /// </summary>
        /// <param name="dataModel">
        /// The data model.
        /// </param>
        private static void OnEmitterTransformPostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as EmitterEmitterTransformData;
            if (dm == null)
            {
                return;
            }

            dm.EmitterPositionAnimation.AnimationTable =
                new AnimationTableData(0, InitialKeyValues.EmitterPosition);
            dm.EmitterRotationAnimation.AnimationTable =
                new AnimationTableData(0, InitialKeyValues.EmitterRotation);
            dm.EmitterScaleAnimation.AnimationTable =
                new AnimationTableData(0, InitialKeyValues.EmitterScale);
        }

        /// <summary>
        /// Post construct handler for EmitterExtParams.
        /// </summary>
        /// <param name="dataModel">The data model that is constructed.</param>
        private static void OnEmitterExtParamsPostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as EmitterExtParams;
            if (dm == null)
            {
                return;
            }

            var userDataInstances =
                UserDataManager.CreateUserDataForOwner(typeof(EmitterExtParams));

            dm.EmitterExtParamsSettingData = userDataInstances.FirstOrDefault() as EmitterExtParamsSettingData;
        }

        /// <summary>
        /// Post construct handler for CustomActionData.
        /// </summary>
        /// <param name="dataModel">The data model that is constructed.</param>
        private static void OnCustomActionDataPostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as CustomActionData;
            if (dm == null)
            {
                return;
            }

            var userDataInstances =
                UserDataManager.CreateUserDataForOwner(typeof(CustomActionData));

            var settings = new ArrayCollection<CustomActionSettingData>(8);

            int index = 0;
            foreach (UserDataBase userData in userDataInstances)
            {
                var customActionSettingData = userData as CustomActionSettingData;
                settings[index] = customActionSettingData;

                if (++index >= 8)
                {
                    break;
                }
            }

            dm.Settings = settings;
        }

        /// <summary>
        /// Post construct handler for EmitterCustomShaderData.
        /// </summary>
        /// <param name="dataModel">The data model that is constructed.</param>
        private static void OnEmitterCustomShaderDataPostConstruct(DataModelBase dataModel)
        {
            var dm = dataModel as EmitterCustomShaderData;
            if (dm == null)
            {
                return;
            }

            var userDataInstances =
                UserDataManager.CreateUserDataForOwner(typeof(EmitterCustomShaderData));

            var settings = new ArrayCollection<EmitterCustomShaderSettingData>(8);

            int index = 0;
            foreach (UserDataBase userData in userDataInstances)
            {
                var customShaderSettingData = userData as EmitterCustomShaderSettingData;
                settings[index] = customShaderSettingData;

                if (++index >= 8)
                {
                    break;
                }
            }

            dm.Settings = settings;
        }
    }
}
