﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Dynamic;
using System.Linq;
using System.Reflection;
using System.Runtime.CompilerServices;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;

using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.Events;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.DataModelProxies
{
    /// <summary>
    /// Class as a proxy that triggers event when the data model property is modified,
    /// so the modification can be send to the viewer through messages.
    /// </summary>
    public class DataModelProxy : DynamicObject, IDisposable
    {
        /// <summary>
        /// The hash set of all the property names that should be ignored when
        /// processing the modification states.
        /// </summary>
        private static readonly HashSet<string> SharedIgnoreModificationProperties = new HashSet<string>()
        {
        };

        /// <summary>
        /// The hash set of all the property names that should be ignored when
        /// processing the modification states for this data model proxy instance.
        /// </summary>
        private readonly HashSet<string> instanceIgnoreModificationProperties = new HashSet<string>();

        /// <summary>The encapsulated data model.</summary>
        private DataModelBase dataModel;

        /// <summary>
        /// Dictionary of data model properties.
        /// </summary>
        private Dictionary<string, PropertyData> dataModelPropInfoMap =
            new Dictionary<string, PropertyData>();

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="dataModel">The encapsulated data model.</param>
        public DataModelProxy(DataModelBase dataModel)
        {
            this.dataModel = dataModel;
            this.UpdateDataModelPropertyDescriptorMap();
        }

        /// <summary>
        /// Static event triggered when property value is modified.
        /// </summary>
        public static event EventHandler<DataModelPropertyModifiedEventArgs> PropertyModified = null;

        /// <summary>
        /// Gets the encapsulated data model.
        /// </summary>
        /// <remarks>
        /// DO NOT access the data model properties directly like:
        /// <code>
        /// dynamic proxy = new DataModelMessageProxy(dataModel);
        /// proxy.DataModel.MyProperty = 10;
        /// </code>
        /// Always set property values through this dynamic proxy like below:
        /// <code>
        /// dynamic proxy = new DataModelMessageProxy(dataModel);
        /// proxy.MyProperty = 10;
        /// </code>
        /// </remarks>
        public DataModelBase DataModel
        {
            get { return this.dataModel; }
        }

        /// <summary>
        /// Extract Guid array from data models, or return the value without any modification.
        /// </summary>
        /// <param name="value">The value to extract.</param>
        /// <returns>The extracted Guid array or the original value.</returns>
        public static object ExtractDataModelGuids(object value)
        {
            if (value == null)
            {
                return null;
            }

            Type dataModelType = typeof(DataModelBase);

            Type valueType = value.GetType();
            if (valueType.IsSubclassOf(dataModelType) == true)
            {
                // Even if the value is just one data model, return a Guid array with only one Guid.
                return new Guid[] { ((DataModelBase)value).Guid };
            }
            else if (value is IEnumerable && valueType.IsGenericTypeOf(dataModelType) == true)
            {
                var dataModels = ((IEnumerable)value).Cast<DataModelBase>();
                var guids = from dm in dataModels
                            select dm == null ? Guid.Empty : dm.Guid;
                return guids.ToArray();
            }
            else
            {
                var cloneableValue = value as ICloneable;
                if (cloneableValue != null)
                {
                    return cloneableValue.Clone();
                }

                return value;
            }
        }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public virtual void Dispose()
        {
        }

        /// <summary>
        /// Add the given data model to the children data models.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        public virtual void AddChildDataModel(DataModelBase dataModel)
        {
        }

        /// <summary>
        /// Add the given data model to the children data models.
        /// </summary>
        /// <param name="index">The index of the data model.</param>
        /// <param name="dataModel">The data model.</param>
        public virtual void InsertChildDataModel(int index, DataModelBase dataModel)
        {
        }

        /// <summary>
        /// Remove the specified data model from the children data models.
        /// </summary>
        /// <param name="dataModel">The dat model to be removed.</param>
        public virtual void RemoveChildDataModel(DataModelBase dataModel)
        {
        }

        /// <summary>
        /// Move the specified data model to the specified index.
        /// </summary>
        /// <param name="dataModel">The data model to be moved.</param>
        /// <param name="distance">
        /// The distance to move the child.
        /// E.g. 2 means move down 2 items, -3 means move up 3 items.
        /// </param>
        public virtual void MoveChildDataModel(DataModelBase dataModel, int distance)
        {
        }

        /// <summary>
        /// Try to get the requested property value from the data model.
        /// </summary>
        /// <param name="binder">The get member binder.</param>
        /// <param name="result">The property value.</param>
        /// <returns>True on success.</returns>
        public override bool TryGetMember(
            GetMemberBinder binder, out object result)
        {
            result = null;
            if (this.dataModel == null)
            {
                return false;
            }

            // Try to get the descriptor of the requested property from the data model.
            PropertyData data = null;
            if (this.dataModelPropInfoMap.TryGetValue(binder.Name, out data) == false)
            {
                return false;
            }

            result = data.Info.GetValue(this.dataModel, null);

            return true;
        }

        /// <summary>
        /// Try to set value to the property from the data model.
        /// </summary>
        /// <param name="binder">The set member binder.</param>
        /// <param name="value">The value to set.</param>
        /// <returns>True on success.</returns>
        public override bool TrySetMember(
            SetMemberBinder binder, object value)
        {
            if (this.dataModel == null)
            {
                return false;
            }

            // Try to get the descriptor of the requested property from the data model.
            PropertyData data = null;
            if (this.dataModelPropInfoMap.TryGetValue(binder.Name, out data) == false)
            {
                Debug.Assert(false, "DataModelProxy.TrySetMember : Failed at TryGetMember");
                return false;
            }

            // Try to convert the value to correct type.
            var canAssign = TypeConversionUtility.TryConvert(
                value.GetType(),
                data.Info.PropertyType,
                ref value);

            if (canAssign == false)
            {
                Logger.Log(LogLevels.Error, "DataModelProxy.TrySetMember : Failed setting incompatible value type.");
                return false;
            }

            // Set the new value...
            data.Info.SetValue(this.dataModel, value, null);

            // Fire event.
            this.TriggerPropertyModifiedEvent(binder.Name);

            return true;
        }

        /// <summary>
        /// Provides a list of available members in the data model.
        /// This method is used for debug purpose and called only by Visual Studio.
        /// </summary>
        /// <returns>Returns a list of the public properties contained in the data model.</returns>
        public override IEnumerable<string> GetDynamicMemberNames()
        {
            return this.dataModelPropInfoMap.Keys;
        }

        /// <summary>
        /// Check if the data model has the specified property.
        /// </summary>
        /// <param name="propertyName">The property name.</param>
        /// <returns>True if the property exists.</returns>
        public bool HasProperty(string propertyName)
        {
            return this.dataModelPropInfoMap.ContainsKey(propertyName);
        }

        /// <summary>
        /// Get the original value of the data model property.
        /// </summary>
        /// <param name="propertyName">The property name.</param>
        /// <param name="value">The original value.</param>
        /// <returns>True on success.</returns>
        public bool GetPropertyOriginalValue(string propertyName, out object value)
        {
            value = null;

            if (this.ShouldIgnorePropertyModificationStates(propertyName) == true)
            {
                return false;
            }

            PropertyData data;
            if (this.dataModelPropInfoMap.TryGetValue(propertyName, out data) == false)
            {
                return false;
            }

            if (data == null)
            {
                return false;
            }

            value = data.OriginalValue;

            return true;
        }

        /// <summary>
        /// Get the default value of the data model property.
        /// </summary>
        /// <param name="propertyName">The property name.</param>
        /// <param name="value">The default value.</param>
        /// <returns>True on success.</returns>
        public bool GetPropertyDefaultValue(string propertyName, out object value)
        {
            value = null;

            if (this.ShouldIgnorePropertyModificationStates(propertyName) == true)
            {
                return false;
            }

            // Get the default data model from data model manager.
            DataModelBase defaultDataModel = this.GetDefaultDataModel();
            if (defaultDataModel == null)
            {
                return false;
            }

            // Switch the data model to the default one,
            // so that TryGetMember gets the value the default data model.
            DataModelBase origDataModel = this.dataModel;
            this.dataModel = defaultDataModel;

            try
            {
                var binder = new EffectMakerGetMemberBinder(propertyName);
                if (this.TryGetMember(binder, out value) == false)
                {
                    return false;
                }
            }
            finally
            {
                // Change back to our data model.
                this.dataModel = origDataModel;
            }

            return true;
        }

        /// <summary>
        /// Get the data model which has the same type as the current data model
        /// but contains the default values.
        /// </summary>
        /// <returns>The default value data model.</returns>
        public DataModelBase GetDefaultDataModel()
        {
            // Get the default data model from data model manager.
            return DataModelManager.GetDefaultDataModel(this.DataModel.GetType());
        }

        /// <summary>
        /// Get the property info with the specified name.
        /// </summary>
        /// <param name="propertyName">The property name.</param>
        /// <param name="info">The property info.</param>
        /// <returns>True on success.</returns>
        public bool GetDynamicPropertyInfo(string propertyName, out PropertyInfo info)
        {
            PropertyData data;
            if (this.dataModelPropInfoMap.TryGetValue(propertyName, out data) == false)
            {
                info = null;
                return false;
            }

            if (data == null)
            {
                info = null;
                return false;
            }

            info = data.Info;

            return true;
        }

        /// <summary>
        /// Provide a list of dynamic property infos.
        /// </summary>
        /// <returns>A list of dynamic property infos.</returns>
        public IEnumerable<PropertyInfo> GetDynamicPropertyInfos()
        {
            return this.dataModelPropInfoMap.Select(item => item.Value.Info).ToArray();
        }

        /// <summary>
        /// Update the property descriptor map for the data model.
        /// </summary>
        public void UpdateDataModelPropertyDescriptorMap()
        {
            this.dataModelPropInfoMap.Clear();

            if (this.dataModel == null)
            {
                return;
            }

            // Get all the PropertyInfo from the data model.
            PropertyInfo[] properties = this.dataModel.GetType().GetProperties();

            // Save the property descriptors.
            foreach (var propertyInfo in properties)
            {
                var propertyData = new PropertyData(propertyInfo);

                object propertyValue = propertyInfo.GetValue(this.DataModel, null);
                object extractedValue = ExtractDataModelGuids(propertyValue);

                propertyData.OriginalValue = extractedValue;

                this.dataModelPropInfoMap.Add(propertyInfo.Name, propertyData);
            }
        }

        /// <summary>
        /// Add the specified property to the ignored list.
        /// </summary>
        /// <param name="propertyName">The property name.</param>
        protected void AddPropertyToIgnoreModificationStates(string propertyName)
        {
            this.instanceIgnoreModificationProperties.Add(propertyName);
        }

        /// <summary>
        /// Check if the data model proxy should ignore the specified property
        /// for modification states.
        /// </summary>
        /// <param name="propertyName">The property name.</param>
        /// <returns>True if the property should be ignored.</returns>
        protected virtual bool ShouldIgnorePropertyModificationStates(string propertyName)
        {
            // Check if the property should be ignored.
            if (SharedIgnoreModificationProperties.Contains(propertyName) == true ||
                this.instanceIgnoreModificationProperties.Contains(propertyName) == true)
            {
                return true;
            }

            // Get the property data to check if it has a public setter,
            // if not, ignore the modification states. (it can't be modified)
            PropertyData data;
            if (this.dataModelPropInfoMap.TryGetValue(propertyName, out data) == false)
            {
                return true;
            }

            return !data.HasPublicSetter;
        }

        /// <summary>
        /// Trigger PropertyModified event.
        /// </summary>
        /// <param name="propertyName">The property name.</param>
        /// <param name="ownerDataModel">Assign the owner data model of the modified property.</param>
        protected void TriggerPropertyModifiedEvent(
            [CallerMemberName]string propertyName = null,
            DataModelBase ownerDataModel = null)
        {
            EventHandler<DataModelPropertyModifiedEventArgs> handler =
                DataModelProxy.PropertyModified;

            if (ownerDataModel == null)
            {
                ownerDataModel = this.dataModel;
            }

            if (handler != null)
            {
                handler(
                    ownerDataModel,
                    new DataModelPropertyModifiedEventArgs(ownerDataModel, propertyName));
            }
        }

        /// <summary>
        /// Class to switch the data model proxy's data model to default data model
        /// temporarily, and switch back while dispose.
        /// </summary>
        public class SwitchDefaultDataModelBlock : IDisposable
        {
            /// <summary>The data model proxy.</summary>
            private DataModelProxy proxy;

            /// <summary>The original data model.</summary>
            private DataModelBase originalDataModel;

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="proxy">The data model proxy.</param>
            public SwitchDefaultDataModelBlock(DataModelProxy proxy)
            {
                this.proxy = proxy;

                this.originalDataModel = this.proxy.dataModel;

                if (this.originalDataModel != null)
                {
                    this.proxy.dataModel =
                        DataModelManager.GetDefaultDataModel(this.proxy.dataModel.GetType());
                }
            }

            /// <summary>
            /// Dispose.
            /// </summary>
            public void Dispose()
            {
                this.proxy.dataModel = this.originalDataModel;
            }
        }

        /// <summary>
        /// プロパティ情報と、ロードしたデータを格納するクラス.
        /// </summary>
        protected class PropertyData
        {
            /// <summary>
            /// Initializes the PropertyData instance.
            /// </summary>
            /// <param name="propertyInfo">The PropertyInfo instance.</param>
            public PropertyData(PropertyInfo propertyInfo)
            {
                this.Info = propertyInfo;

                // If this property has no public setter, ignore the modification states.
                this.HasPublicSetter = propertyInfo.GetSetMethod(false) != null;
            }

            /// <summary>プロパティ情報.</summary>
            public PropertyInfo Info { get; private set; }

            /// <summary>
            /// Get the flag indicating whether the property has a public setter or not.
            /// </summary>
            public bool HasPublicSetter { get; private set; }

            /// <summary>ロード時点の値.</summary>
            public object OriginalValue { get; set; }
        }
    }
}
