﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.IO;
using System.Linq;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.BinaryConverters;
using EffectMaker.DataModelLogic.Utilities;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryData
{
    /// <summary>
    /// Fixed-size binary data field instance created with the binary field definition.
    /// </summary>
    public class FixedSizeBinaryFieldInstance : BinaryFieldInstance
    {
        /// <summary>The binary field definition.</summary>
        private FixedSizeBinaryFieldDefinition myDef = null;

        /// <summary>The binary size.</summary>
        private int binarySize = 0;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="parent">The parent binary data structure instance.</param>
        /// <param name="definition">The binary data field definition.</param>
        public FixedSizeBinaryFieldInstance(
            BinaryStructInstance parent,
            BinaryFieldDefinition definition) :
            base(parent, definition)
        {
            this.myDef = definition as FixedSizeBinaryFieldDefinition;
            if (this.myDef == null)
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Get the size of the binary data.
        /// </summary>
        public override int Size
        {
            get { return this.binarySize; }
        }

        /// <summary>
        /// Get or set the size of the binary data including the binary headers if any.
        /// </summary>
        public override int SizeIncludeHeader
        {
            get { return this.binarySize; }
        }

        /// <summary>
        /// Called by a child element to report that it's size has changed
        /// and the offset needs to be recalculated.
        /// </summary>
        /// <param name="reportingElement">The reporting child element.</param>
        public override void ReportSizeChanged(IBinaryElementInstance reportingElement)
        {
            // The only chance this method is called is when the size of other fields are
            // changed, and the parent's ReportSizeChanged method calls this method, so
            // there is no need to report size change back to the parent.
        }

        /// <summary>
        /// Convert binary data from the data model.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool ConvertBinary()
        {
            this.CheckFieldOffsetForDebugBreak();

            if ((this.Definition.ConverterDefinition == null) ||
                (this.Definition.InputProperties == null) ||
                (this.Definition.InputProperties.Length <= 0))
            {
                // No conversion, just use the fixed size.
                this.binarySize = this.myDef.FixedSize;
                return true;
            }

            // Create the converter.
            var converter = this.Definition.ConverterDefinition.CreateConverter();

            // The required input value count of the converter
            // should match the input property count.
            int inputValueCount = this.Definition.InputProperties.Length;
            if (converter.MinInputValueCount > inputValueCount ||
                converter.MaxInputValueCount < inputValueCount)
            {
                Logger.Log(LogLevels.Warning, "FixedSizeBinaryFieldInstance.ConvertBinary : Input value count does not match the requested input count of the converter.");
                return false;
            }

            // Feed input values to the converter.
            if (this.FeedInputValueToConverter(converter) == false)
            {
                return false;
            }

            int originalBinarySize = this.binarySize;

            // Convert!!!
            if (converter.Convert() == false)
            {
                Logger.Log(LogLevels.Warning, "FixedSizeBinaryFieldInstance.ConvertBinary : The converter reported a failure when converting the data.");
                return false;
            }

            object binarySizeObj = converter.GetOutputValue();

            if (TypeConversionUtility.TryConvert(
                binarySizeObj.GetType(),
                typeof(int),
                ref binarySizeObj) == false)
            {
                Logger.Log(LogLevels.Warning, "FixedSizeBinaryFieldInstance.ConvertBinary : Failed converting the output value of the converter to an integer.");
                return false;
            }

            // Update the binary size.
            this.binarySize = (int)binarySizeObj;

            if (this.Parent != null && this.binarySize != originalBinarySize)
            {
                // Report size change.
                this.Parent.ReportSizeChanged(this);
            }

            return true;
        }

        /// <summary>
        /// Write the binary data for the given session.
        /// </summary>
        /// <param name="session">The WriteBinaryDataSession that is being processed now.</param>
        /// <returns>True on success.</returns>
        public override bool WriteBinaryData(WriteBinaryDataSession session)
        {
            if (this.binarySize > 0)
            {
                using (new WriteBinaryDataBlock(session, this))
                {
                    session.Stream.Write(
                        Enumerable.Repeat<byte>(0, this.binarySize).ToArray(),
                        0,
                        this.binarySize);
                }
            }

            return true;
        }
    }
}
