﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.BinaryConverters;
using EffectMaker.DataModelLogic.Utilities;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryData
{
    /// <summary>
    /// Binary data field instance that stores the size of another binary structure.
    /// </summary>
    public class BinaryStructSizeFieldInstance : BinaryFieldInstance
    {
        /// <summary>The binary field definition.</summary>
        private BinaryStructSizeFieldDefinition myDef = null;

        /// <summary>The target binary element instance to get offset from.</summary>
        private IBinaryElementInstance targetBinaryElementInstance = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="parent">The parent binary data structure instance.</param>
        /// <param name="definition">The binary data field definition.</param>
        public BinaryStructSizeFieldInstance(
            BinaryStructInstance parent,
            BinaryFieldDefinition definition) :
            base(parent, definition)
        {
            this.myDef = definition as BinaryStructSizeFieldDefinition;
            if (this.myDef == null)
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Get the size of the binary data.
        /// The size of an offset field is always 4 bytes long. (uint)
        /// </summary>
        public override int Size
        {
            get { return 4; }
        }

        /// <summary>
        /// Get or set the size of the binary data including the binary headers if any.
        /// </summary>
        public override int SizeIncludeHeader
        {
            get { return this.Size; }
        }

        /// <summary>
        /// Called by a child element to report that it's size has changed
        /// and the offset needs to be recalculated.
        /// </summary>
        /// <param name="reportingElement">The reporting child element.</param>
        public override void ReportSizeChanged(IBinaryElementInstance reportingElement)
        {
            // The only chance this method is called is when the size of other fields are
            // changed, and the parent's ReportSizeChanged method calls this method, so
            // there is no need to report size change back to the parent.
        }

        /// <summary>
        /// Convert binary data from the data model.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool ConvertBinary()
        {
            this.CheckFieldOffsetForDebugBreak();

            if ((this.Definition.ConverterDefinition == null) ||
                (this.Definition.InputProperties == null) ||
                (this.Definition.InputProperties.Length <= 0))
            {
                // If no input or converter being set, just use the parent binary structure.
                var parent = this.Parent;
                while (parent != null)
                {
                    if (parent is BinaryStructInstance)
                    {
                        this.targetBinaryElementInstance = (BinaryStructInstance)parent;
                        return true;
                    }

                    parent = parent.Parent;
                }

                // There is no parent binary structure, something is wrong!
                Logger.Log(LogLevels.Warning, "BinaryStructSizeFieldInstance.ConvertBinary : Unable to find parent binary structure.");
                return false;
            }

            // Create the converter.
            var converter = this.Definition.ConverterDefinition.CreateConverter();

            // The required input value count of the converter
            // should match the input property count.
            int inputValueCount = this.Definition.InputProperties.Length;
            if (converter.MinInputValueCount > inputValueCount ||
                converter.MaxInputValueCount < inputValueCount)
            {
                Logger.Log(LogLevels.Warning, "BinaryStructSizeFieldInstance.ConvertBinary : Input value count does not match the requested input count of the converter.");
                return false;
            }

            // Feed input values to the converter.
            if (this.FeedInputValueToConverter(converter) == false)
            {
                return false;
            }

            // Convert!!!
            if (converter.Convert() == false)
            {
                Logger.Log(LogLevels.Warning, "BinaryStructSizeFieldInstance.ConvertBinary : The converter reported a failure when converting the data.");
                return false;
            }

            DataModelBase targetDataModel = null;

            object output = converter.GetOutputValue();
            if (output is DataModelBase)
            {
                targetDataModel = (DataModelBase)output;
            }
            else if (output is IEnumerable<DataModelBase>)
            {
                targetDataModel = ((IEnumerable<DataModelBase>)output).FirstOrDefault();
            }

            // Find the binary element that holds the data model.
            if (targetDataModel == null)
            {
                this.targetBinaryElementInstance = null;
            }
            else
            {
                this.targetBinaryElementInstance = this.FindSourceDataModelOwner(targetDataModel);
            }

            //// The size of this field is always 4 bytes long, so no need to report
            //// size change to the parent.

            return true;
        }

        /// <summary>
        /// Write the binary data for the given session.
        /// </summary>
        /// <param name="session">The WriteBinaryDataSession that is being processed now.</param>
        /// <returns>True on success.</returns>
        public override bool WriteBinaryData(WriteBinaryDataSession session)
        {
            uint size = 0;
            if (this.targetBinaryElementInstance != null)
            {
                size = (uint)(this.targetBinaryElementInstance.Size + this.myDef.AdjustValue);
            }

            using (new WriteBinaryDataBlock(session, this))
            {
                session.Stream.Write(
                    BinaryConversionUtility.ForResource.Convert(size),
                    0,
                    this.Size);
            }

            return true;
        }
    }
}
