﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

namespace EffectMaker.DataModelLogic.BinaryData
{
    /// <summary>
    /// Class for the definition of a binary data structure.
    /// </summary>
    public class BinaryStructDefinition
    {
        /// <summary>
        /// The dictionary that maps binary structure definitions and
        /// their source data model type.
        /// </summary>
        private static Dictionary<Type, BinaryStructDefinition> binStructSrcTypeMap =
            new Dictionary<Type, BinaryStructDefinition>();

        /// <summary>The list of binary field definitions.</summary>
        private List<BinaryFieldDefinition> fields = new List<BinaryFieldDefinition>();

        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="sourceDataModelType">The source data model type.</param>
        public BinaryStructDefinition(Type sourceDataModelType)
        {
            if (sourceDataModelType != null)
            {
                binStructSrcTypeMap[sourceDataModelType] = this;
            }

            this.SourceDataModelType = sourceDataModelType;
            this.HasBinaryHeader = true;
        }

        /// <summary>
        /// Get the type of the source data model.
        /// </summary>
        public Type SourceDataModelType { get; private set; }

        /// <summary>
        /// Get or set the flag indicating whether the binary structure has binary header.
        /// </summary>
        public bool HasBinaryHeader { get; set; }

        /// <summary>
        /// Get or set the tag of the binary field.
        /// </summary>
        public string Tag { get; set; }

        /// <summary>
        /// Get the binary data field definitions.
        /// </summary>
        public IEnumerable<BinaryFieldDefinition> Fields
        {
            get { return this.fields; }
        }

        /// <summary>
        /// Find the binary structure definition that the source data model type
        /// matches the given type.
        /// </summary>
        /// <param name="srcDataModelType">The source data model type.</param>
        /// <returns>The binary structure definition or null if not found.</returns>
        public static BinaryStructDefinition FindBinaryStructureDefinition(Type srcDataModelType)
        {
            BinaryStructDefinition def;
            if (binStructSrcTypeMap.TryGetValue(srcDataModelType, out def) == true)
            {
                return def;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// Add a binary field definition for the binary structure definition.
        /// </summary>
        /// <param name="def">The binary field definition.</param>
        public void AddField(BinaryFieldDefinition def)
        {
            // Add the field to our map.
            this.fields.Add(def);
        }

        /// <summary>
        /// Create an instance with the definition.
        /// </summary>
        /// <param name="parent">The parent instance.</param>
        /// <param name="dataModel">The data model for the binary structure.</param>
        /// <returns>The created instance.</returns>
        public virtual BinaryStructInstance CreateInstance(
            IBinaryElementInstance parent,
            DataModelBase dataModel)
        {
            return new BinaryStructInstance(parent, this, dataModel);
        }

        /// <summary>
        /// Create binary data field instances.
        /// </summary>
        /// <param name="parent">The parent binary data structure instance.</param>
        /// <returns>The created instances.</returns>
        public virtual BinaryFieldInstance[] CreateFieldInstances(BinaryStructInstance parent)
        {
            var instances = new BinaryFieldInstance[this.fields.Count];

            int index = 0;
            foreach (BinaryFieldDefinition fieldDef in this.Fields)
            {
                instances[index++] = fieldDef.CreateInstance(parent);
            }

            return instances;
        }
    }
}
