﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

using EffectMaker.DataModel.Attributes;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// 弧度法に変換するコンバータです.
    /// </summary>
    public class RadianConverter : ConverterBase
    {
        /// <summary>
        /// 円周率です.
        /// </summary>
        private readonly float pi = 3.1415926535897932384626433832795f;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RadianConverter()
            : base(null)
        {
            /* DO_NOTHING  */
        }

        /// <summary>
        /// 最大入力数です.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// コンバータの概要説明です.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.RadianConverterDescription; }
        }

        /// <summary>
        /// コンバータの出力値の概要説明です.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.RadianConverterOutputValueDescription; }
        }

        /// <summary>
        /// コンバータの入力値の概要説明です.
        /// </summary>
        /// <param name="index">インデックスです.</param>
        /// <returns>概要説明の文字列</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.RadianConverterInputValue0Description;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// 変換処理を行います.
        /// </summary>
        /// <returns>変換に成功したらtrueを返却します.</returns>
        public override bool Convert()
        {
            // floating point
            {
                var inputValue = this.GetInputValue(0) as float?;
                if (inputValue != null)
                {
                    float outputValue = inputValue.Value * (this.pi / 180.0f);
                    this.SetOutputValue(outputValue);

                    return true;
                }
            }

            // integer
            {
                var inputValue = this.GetInputValue(0) as int?;
                if (inputValue != null)
                {
                    float outputValue = (float)inputValue.Value * (this.pi / 180.0f);
                    this.SetOutputValue(outputValue);

                    return true;
                }
            }

            // Vector1 float
            {
                var inputValue = this.GetInputValue(0) as Vector1f;
                if (inputValue != null)
                {
                    float outputValue = inputValue.X * (this.pi / 180.0f);
                    this.SetOutputValue(outputValue);

                    return true;
                }
            }

            // Vector1 integer
            {
                var inputValue = this.GetInputValue(0) as Vector1i;
                if (inputValue != null)
                {
                    float outputValue = (float)inputValue.X * (this.pi / 180.0f);
                    this.SetOutputValue(outputValue);

                    return true;
                }
            }

            // Vector2 float
            {
                var inputValue = this.GetInputValue(0) as Vector2f;
                if (inputValue != null)
                {
                    Vector2f outputValue = new Vector2f(
                        inputValue.X * (this.pi / 180.0f),
                        inputValue.Y * (this.pi / 180.0f));
                    this.SetOutputValue(outputValue);

                    return true;
                }
            }

            // Vector2 integer
            {
                var inputValue = this.GetInputValue(0) as Vector2i;
                if (inputValue != null)
                {
                    Vector2f outputValue = new Vector2f(
                        (float)inputValue.X * (this.pi / 180.0f),
                        (float)inputValue.Y * (this.pi / 180.0f));
                    this.SetOutputValue(outputValue);

                    return true;
                }
            }

            // Vector3 float
            {
                var inputValue = this.GetInputValue(0) as Vector3f;
                if (inputValue != null)
                {
                    Vector3f outputValue = new Vector3f(
                        inputValue.X * (this.pi / 180.0f),
                        inputValue.Y * (this.pi / 180.0f),
                        inputValue.Z * (this.pi / 180.0f));
                    this.SetOutputValue(outputValue);

                    return true;
                }
            }

            // Vector3 integer
            {
                var inputValue = this.GetInputValue(0) as Vector3i;
                if (inputValue != null)
                {
                    Vector3f outputValue = new Vector3f(
                        inputValue.X * (this.pi / 180.0f),
                        inputValue.Y * (this.pi / 180.0f),
                        inputValue.Z * (this.pi / 180.0f));
                    this.SetOutputValue(outputValue);

                    return true;
                }
            }

            // Vector4 float
            {
                var inputValue = this.GetInputValue(0) as Vector4f;
                if (inputValue != null)
                {
                    Vector4f outputValue = new Vector4f(
                        inputValue.X * (this.pi / 180.0f),
                        inputValue.Y * (this.pi / 180.0f),
                        inputValue.Z * (this.pi / 180.0f),
                        inputValue.W * (this.pi / 180.0f));
                    this.SetOutputValue(outputValue);

                    return true;
                }
            }

            // Vector4 integer
            {
                var inputValue = this.GetInputValue(0) as Vector4i;
                if (inputValue != null)
                {
                    Vector4f outputValue = new Vector4f(
                        (float)inputValue.X * (this.pi / 180.0f),
                        (float)inputValue.Y * (this.pi / 180.0f),
                        (float)inputValue.Z * (this.pi / 180.0f),
                        (float)inputValue.W * (this.pi / 180.0f));
                    this.SetOutputValue(outputValue);

                    return true;
                }
            }

            return false;
        }
    }
}
