﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Class to composite multiple values into a series of values.
    /// </summary>
    public class CompositionConverter : ConverterBase
    {
        /// <summary>Constant for element count parameter name.</summary>
        private const string ElementCountParamName = "ElementCount";

        /// <summary>Constant for element type parameter name.</summary>
        private const string ElementTypeParamName = "ElementType";

        /// <summary>The converter parameter holder.</summary>
        private static ConverterParamHolder paramHolder = new ConverterParamHolder();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static CompositionConverter()
        {
            // Register parameters.
            paramHolder.RegisterParameter(
                ElementCountParamName,
                "The output element count.",
                ValidateElementCountParamter);

            paramHolder.RegisterParameter(
                ElementTypeParamName,
                "The type of the output elements.",
                ValidateElementTypeParamter);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public CompositionConverter() :
            base(paramHolder)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MinInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 4; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.CompositionConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.CompositionConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.CompositionConverterInputValue0Desc;
            }
            else if (index == 1)
            {
                return Properties.Resources.CompositionConverterInputValue1Desc;
            }
            else if (index == 2)
            {
                return Properties.Resources.CompositionConverterInputValue2Desc;
            }
            else if (index == 3)
            {
                return Properties.Resources.CompositionConverterInputValue3Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            if (this.CountAssignedInputValues() < this.MinInputValueCount)
            {
                Logger.Log(LogLevels.Error, "CompositionConverter.Convert : The assigned input values are not enough for conversion.");
                return false;
            }

            object param;

            // Get the element count from the parameter.
            int elementCount = 4;
            if (this.GetParameter(ElementCountParamName, out param) == true)
            {
                elementCount = (int)param;
            }

            // Get the element type from the parameter.
            Type elementType = typeof(float);
            if (this.GetParameter(ElementTypeParamName, out param) == true)
            {
                elementType = (Type)param;
            }

            // Create the array.
            Array output = Array.CreateInstance(elementType, elementCount);

            // Create a default value.
            object defaultValue = null;
            if (elementType.IsValueType == true)
            {
                defaultValue = Activator.CreateInstance(elementType);
            }

            // Fill in the values to the array.
            for (int i = 0; i < elementCount; ++i)
            {
                // Is the input value assigned at the index?
                if (this.IsInputValueAssigned(i) == false)
                {
                    output.SetValue(defaultValue, i);
                    continue;
                }

                // The input is assigned, get it.
                object input = this.GetInputValue(i);
                if (input == null)
                {
                    continue;
                }

                // Convert the input value to the desired type.
                object convertedInputValue = input;
                if (TypeConversionUtility.TryConvert(
                    input.GetType(),
                    elementType,
                    ref convertedInputValue) == false)
                {
                    Logger.Log(LogLevels.Error, "CompositionConverter.Convert : Failed converting the input value to the target element type.");
                    output.SetValue(defaultValue, i);
                    continue;
                }

                // Set the converted input value to the output array.
                output.SetValue(convertedInputValue, i);
            }

            this.SetOutputValue(output);

            return true;
        }

        /// <summary>
        /// Validate element count parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateElementCountParamter(object value)
        {
            int count;
            if (value is int)
            {
                count = (int)value;
            }
            else if (value is string)
            {
                if (int.TryParse((string)value, out count) == false)
                {
                    return false;
                }
            }
            else
            {
                return false;
            }

            if (count <= 0 || count > 4)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Validate element type parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateElementTypeParamter(object value)
        {
            Type type;
            if (value is Type)
            {
                type = (Type)value;
            }
            else if (value is string)
            {
                string[] matches =
                    RegexUtility.ExtractStrings("^typeof\\(([^\\)]+)\\)$", (string)value);
                if (matches == null || matches.Length <= 0 || matches[0] == null)
                {
                    return false;
                }

                type = Type.GetType(matches[0], false);
                if (type == null)
                {
                    return false;
                }
            }
            else
            {
                return false;
            }

            return true;
        }
    }
}
